;; TimeSeries-core.nlogo v1.5
;; author: Doug Edmunds 24 Feb 2017 dougedmundsATgmailDOTcom
;; requires ver 6.0+

extensions [ csv ]

globals [
  data-source
  data
  vals-list vals-list-length
  data-pointer data-length
  logfile
  ;; The next globals are the names given to the fields in the CSV
  ;; If not appropriate names for your csv fields, change them here and
  ;; in the specifics-get-next-record procedure.  As a suggestion,
  ;; comment out the next line temporarily, to see all the places
  ;; where these globals are used in the code.
  data-date data-high data-low  data-open data-close data-volume

;; add new variables here
  ;; For calc-sma procedure (calculate simple moving average)
  ;; included just for demonstration purposes
  window-list   ;; a list (initially empty) of values used to calculate the average
  window-length ;; numeric, used to size the window-list
  ;; window-length gets value from sma-slider
  sma ;; simple moving average, numeric
]

;; interface globals
;; wait-switch?       boolean
;; wait-delay-slider  slider
;; print-to-logfile?  boolean
;; print-to-output?   boolean
;; chart-data         plot name

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to setup
  ca
  ;; to hide "world", uncomment the next line
    ; set-patch-size -1

  ;;set data-pointer -1
  set data-length 0

  ;; alternative A
  ;; use this to select a file from the directory
  set data read-data user-file
  ;; alternative B
  ;; to hardcode the data source, use this code instead
    ;   set data-source "my_cvs_file.cvs"
    ;   set data read-data data-source

  ;; create a logfile
  set logfile "logfile.txt"
  let log-entry word "\r -- New Session -- " date-and-time
  if print-to-logfile? [
    print-to-file logfile log-entry
    print-to-file logfile logfile
    print-to-file logfile data-source ]
  if print-to-logfile? [ output-print logfile]
  output-print log-entry

ifelse empty? data
    [ output-print word date-and-time " - no file selected"]
    [ output-print data-source

      ;; if you have problems with headers, then make changes here
      ;; this code reads the first line; if last item is a string,
      ;; it assumes it is a header, and removes it
      if is-string? last first data
         [set data butfirst data]

      set data-pointer 0
      set data-length length data
      set vals-list first data
      ;; show the length of first entry in observer area
      type "the number of fields in each record is "
      print length vals-list
      print vals-list

      ;; use length to determine if tick data or not
      set vals-list-length length vals-list
      ;; specifics-get-next-record is called once at startup
      ;; this is done to establish a y range for the chart-data plot
      specifics-get-next-record
      clear-plots
      specifics-setup
      specifics-logfile
      ;; only reset if a file is selected. Ties in with disabled go buttons
      reset-ticks ]
end

to specifics-setup
;; for calc-sma demonstration purposes, these global variables are initialized
  ;; calc-sma uses data-close. The first record of the CSV has been read,
  ;; so we use data-close here to initialize sma-average
  set window-list [ ]
  set window-length  sma-slider ;;NOTE, only set at start or restart
  set sma data-close
  calc-sma
end

;;data-pointer must be incremented
;;  before calling get-next-record

to go
  ;; first check if should stop
  ;; A. prevent runtime error if setup was not run first
  if data-length = -1 [output-print "no file selected" stop]
  ;; B. prevent runtime error if at end of file or no file selected
  if data-pointer >= data-length [output-print "end of file [1]" stop]
  ;; C. stop the stream if at end of file after increment
  set data-pointer data-pointer + 1
  if data-pointer >= data-length[ output-print "end of file [2]" stop]
  ;; not at end of file, so continue
  get-next-record
  if wait-switch? [ wait wait-delay-slider]
  specifics-go
  specifics-logfile
  tick
end

to specifics-go
  ;;add strategy code here
  calc-sma
end

;; sample logfile details
;; modify this section to print whatever details you want.
to specifics-logfile
  if print-to-logfile?
    [ print-to-file logfile (word "data-close " data-close) ]
  if print-to-output? [
    output-print vals-list
    output-print (word "data-close " data-close) ]
end

to get-next-record
  set vals-list item data-pointer data
  specifics-get-next-record
end

to specifics-get-next-record
  ;; variables that needs to be updated before
  ;; the current record is changed are updated here
  ;; for example, set data-close-previous  data-close

  ;; MODIFY THIS TO MATCH UP WITH CSV
  ;; the specific code here is a model to handle
  ;; data with a record length of 3 or 6.
  ;; You should modify this section as needed to
  ;; match up with the field structure of your CSV

  ifelse vals-list-length = 3 [
  ;;  for tick data with 3 fields  (order: Datetime Close Volume)
  set data-date   item 0 vals-list
  set data-close  item 1 vals-list
  set data-volume item 2 vals-list
  set data-open   data-close
  set data-high   data-close
  set data-low    data-close
  ][
  ;;  for financial data with 6 fields (order: Datetime High Low Open Close Volume)
  set data-date   item 0 vals-list
  set data-high   item 1 vals-list
  set data-low    item 2 vals-list
  set data-open   item 3 vals-list
  set data-close  item 4 vals-list
  set data-volume item 5 vals-list
  ]
end

to-report read-data [ file-name ]
;; create rows as an empty list
  let rows []
;; user-file reports false if no file is selected
;; if file-name is not a csv file, expect a run-time error.
  if file-name != false [
    set data-source file-name
    file-open file-name
 ;; csv:from-file creates a list of lists
    set rows csv:from-file file-name
    file-close ]
  report rows
end

to print-to-file [file-name file-text]
  file-open file-name
    file-print file-text
    file-close
end

to delete-logfile
  if logfile != 0 and file-exists? logfile [file-delete logfile]
end

to clear-plots
  ;;clear-all-plots
  ;; alternatively you can set and clear plots one by run and reset the ranges
   set-current-plot "chart-data"
   clear-plot
   set-plot-y-range  precision (data-close - 0.01) 4 precision (data-close + 0.01) 4
end

to restart-data
  ;; similar to setup, but doesn't load data from file
  set data-pointer 0
  set data-length length data
  set vals-list first data
  specifics-get-next-record

output-print "Reset to first record"
  output-print vals-list

  let log-entry word "\r -- New Session -- " date-and-time
  if print-to-logfile? [
    print-to-file logfile log-entry
    print-to-file logfile logfile
    print-to-file logfile data-source ]
  specifics-setup
  specifics-logfile
  clear-plots
  reset-ticks
end

;;;;;;;; UTILITIES ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;; If your csv file stores the newest records first, use one of the
;; procedures below to reverse the order of the records.
;; Type the procedure name on the observer line
;;  FIRST, inspect the csv in a text editor
;;   If the csv has a header, you must use 'reverse-csv-with-header'
;;   If the csv does not have a header, you must use 'reverse-csv-no-header'
;;  Note: this procedure overwrites 'reversed.csv' if it already exists
;;  It does not change the original file.

to reverse-csv-with-header
  set data read-data user-file
  ;remove the header and put it at the end
  let header first data
  set data butfirst data
  set data lput header data
  set data reverse data
    ; save the data.  Manually rename the file when done
  csv:to-file "reversed.csv" data
  set data []
  output-print "Done - changes have been saved to reversed.csv"
  output-print "Inspect new file before using it."
end

to reverse-csv-no-header
  set data read-data user-file
  set data reverse data
    ; save the data.  Manually rename the file when done
  csv:to-file "reversed.csv" data
  set data []
  output-print "Done - changes have been saved to reversed.csv"
  output-print "Inspect new file before using it."
end

;;;;;;;;; NEW CODE ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;put all new code below here

;; calculate a simple moving average on a moving window of data
to calc-sma
  ifelse length window-list < window-length
    [ set window-list fput data-close window-list ]
    [ set window-list but-last window-list
      set window-list fput data-close window-list ]
  set sma mean window-list
end
@#$#@#$#@
GRAPHICS-WINDOW
590
10
628
49
-1
-1
10.0
1
8
1
1
1
0
1
1
1
-1
1
-1
1
0
0
1
ticks
30.0

BUTTON
10
10
73
43
NIL
setup
NIL
1
T
OBSERVER
NIL
S
NIL
NIL
1

BUTTON
115
10
195
43
go once
go
NIL
1
T
OBSERVER
NIL
G
NIL
NIL
0

BUTTON
196
10
283
43
go forever
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

SWITCH
10
95
134
128
wait-switch?
wait-switch?
0
1
-1000

BUTTON
286
10
375
43
NIL
clear-plots
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
121
95
247
128
wait-delay-slider
wait-delay-slider
0.02
1.0
0.2
.02
1
NIL
HORIZONTAL

BUTTON
496
10
582
43
delete log
delete-logfile
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

OUTPUT
10
280
428
355
11

PLOT
9
135
429
279
chart-data
NIL
Price
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"pen-3" 1.0 0 -16777216 true "" "plot data-close"
"pen-1" 1.0 0 -2674135 true "" "plot sma"

SWITCH
10
50
150
83
print-to-logfile?
print-to-logfile?
1
1
-1000

SWITCH
155
50
299
83
print-to-output?
print-to-output?
1
1
-1000

MONITOR
450
130
535
175
NIL
data-pointer
17
1
11

MONITOR
540
130
620
175
NIL
data-length
17
1
11

BUTTON
386
10
483
43
restart data
restart-data\n;;sets the data-pointer back to beginning\n;;and some cleanup needed to rerun the data
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

SLIDER
455
250
547
283
sma-slider
sma-slider
3
100
61.0
1
1
NIL
HORIZONTAL

MONITOR
450
185
507
230
Close
precision data-close 4
17
1
11

MONITOR
510
185
567
230
SMA
precision sma 4
17
1
11

@#$#@#$#@
## WHAT IS IT?

This model processes a time series (sequential) CSV file. It uses a pointer method to advance through the CSV data.  It allows you to restart at any point in the data.

It takes care of the mechanics of selecting the CSV, checking for end of file, etc. It provides a sample plot of the output as a line-chart and text output to screen, as well the ability to write to a logfile.

By inserting code in "specific" procedures you can use this as a skeleton on which to build a sequential model based on your CSV data. 

It comes with a few sample CSV files of actual market transactions (high, low, open, close).  Run those to get an idea of what you can do with it.   

## HOW IT WORKS

The program reads a CSV (comma separated values) file into a list-structured variable using csv:from-file.  Each record of the CSV file is processed by the "go" and the "specifics-go" procedures.  Charts and monitors show the sequential changes.

A 'data-pointer' is advanced on each cycle. The current row is stored in "vals-list".  The item numbers of the vals-list (starting at item 0) are associated with variables.  For example, if "Close" is the third field in the CSV file, then data-close is associated with item 2 of vals-list.  The associations are set in the "specifics-get-next-record" procedure


## HOW TO USE IT

To see how it works, press setup and select one of the example CSV files provided.  These are simple ASCII files which you can open in a text editor.  

Turn on the wait-switch to create a delay between each tick. 

If you have a sequential data file, put it into the directory where this model is located. Press setup and select your data file. The program will attempt to load the first record, using the field order it finds in the "specifics-get-next-record" procedure.  If your data will process without error, then you can make changes to examine other aspects of the time series data. 

You can press the restart button to run the program again with reloading the data file. Try changing the sma slider value between runs. 

To see how it works, load one of the example CSV files provided.  If you want other data,
go online and get time series data from Yahoo financial and other sources.

NOTE 1: MAKE SURE YOUR CSV FILE IS CLEAN

This code expects numerical values in numerical fields. It does not check for errors such as missing values or strings (such as NA). If there are errors, you may get a runtime error at that point in the data, or the title of the default chart will turn red.

NOTE 2: OLDEST ENTRIES MUST BE FIRST

The program requires that the OLDEST data is at the beginning of the CSV file. Some utility code is provided that you can use to reverse the order (reverse-csv-with-header and reverse-csv-no-header). 

NOTE 3: IDENTIFY THE FIELDS 

Each record (row) of the CSV has the same field structure. Modify the code in the "specifics-get-next-record" procedure to match your CSV.  The default code is set to work with either 3 fields (date, close, volume) or 6 fields (date, high, low, open, close, volume) in that order. If your CSV has the same fields but in a different order, it is easier to change the item numbering in the code than to restructure your CSV.   

ifelse vals-list-length = 3 [
  ;;  for tick data with 3 fields  (Datetime Close Volume)
  set data-high-prev data-close
  set data-low-prev data-close
  set data-date   item 0 vals-list
  set data-close  item 1 vals-list
  set data-volume item 2 vals-list
  set data-high data-close
  set data-low data-close
  ][
  ;;  for financial data with 6 fields (Datetime High Low Open Close Volume)
  set data-high-prev data-high
  set data-low-prev data-low
  set data-date   item 0 vals-list
  set data-high   item 1 vals-list
  set data-low    item 2 vals-list
  set data-open   item 3 vals-list
  set data-close  item 4 vals-list
  set data-volume item 5 vals-list
  ]

The field names (data-date, data-high, etc.) are globals.  With a little revision, you can rename these fields to match up with the names of your fields. For example, if your CSV has a field called "size", as the third item, you could change the code to read 

   set data-size item 2 vals-list

You should make sure that each of the fields is assigned to a variable, even if you
don't use that variable.  

NOTE 4: USING DATA AVAILABLE ON THE INTERNET

You can easily re-order the data to match up with internet data. 

As an example, get the data file of Apple stock trades from http://chartapi.finance.yahoo.com/instrument/1.0/AAPL/chartdata;type=quote;range=1d/csv

It will look something like this (see below).  This looks like minute by minute data, with the oldest entries first.  
THE APPLE STOCK DATA FROM YAHOO ON 8 FEB 2017 (just the start of it):

uri:/instrument/1.0/AAPL/chartdata;type=quote;range=1d/csv
ticker:aapl
Company-Name:Apple Inc.
Exchange-Name:NMS
unit:MIN
timezone:EST
currency:USD
gmtoffset:-18000
previous_close:131.5300
Timestamp:1486564200,1486587600
labels:1486566000,1486569600,1486573200,1486576800,1486580400,1486584000,1486587600
values:Timestamp,close,high,low,open,volume
close:131.2700,132.2100
high:131.3110,132.2200
low:131.2200,132.1700
open:131.2500,132.2060
volume:0,1526743552
1486564258,131.3328,131.3500,131.2200,131.2500,698100
1486564319,131.3800,131.4000,131.3100,131.3300,281000
1486564322,131.4800,131.5400,131.3600,131.4000,134300
1486564380,131.5900,131.6600,131.5000,131.5300,178400
...
...(about 400 rows of data)

Find the parts needed for a csv file a) the header (which is on the line starting with "values" and b) the data.  Delete the other lines.  Save this to a csv file (unformatted text file). 

Timestamp,close,high,low,open,volume
1486564258,131.3328,131.3500,131.2200,131.2500,698100
1486564319,131.3800,131.4000,131.3100,131.3300,281000
1486564322,131.4800,131.5400,131.3600,131.4000,134300
1486564380,131.5900,131.6600,131.5000,131.5300,178400
...


The header indicates that the there are 6 fields, and the order of the fields is datetime-close-high-low-open-volume.  To make this work, modify the Code tab procedure "specifics-get-next-record".  Change the numbers of the items to point to the right  variables. The Yahoo data order is set to look like this:

  ;;  for financial data with 6 fields (Datetime High Low Open Close Volume)
  set data-high-prev data-high
  set data-low-prev data-low
  set data-date   item 0 vals-list
  set data-high   item 2 vals-list  ;; was 1
  set data-low    item 3 vals-list  ;; was 2
  set data-open   item 4 vals-list  ;; was 3
  set data-close  item 1 vals-list  ;; was 4
  set data-volume item 5 vals-list

That's all the code change you need to make! 

NOTE 5: LOG FILE AND SCREEN OUTPUT

You can maintain a written log of whatever you want to track. The log file is a basic ascii text file. Pressing the delete logfile button will remove it, with no backup.


## CREDITS AND REFERENCES

Author: Doug Edmunds Feb 2017
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
1
@#$#@#$#@
