globals [
  color-mode     ;; 0 = normal, 1 = when heard, 2 = times heard
  clique         ;; how many patches have heard the rumor
]

patches-own [
  times-heard    ;; tracks times the rumor has been heard
  first-heard    ;; clock tick when first heard the rumor
  just-heard?    ;; tracks whether rumor was heard this round -- resets each round
]

;;; setup procedures

to setup [seed-one?]
  ca
  set color-mode 0
  set clique 0
  set juiciness 4
  ask patches
    [ set first-heard -1
      set times-heard 0
      set just-heard? false
      recolor ]
  ifelse seed-one?
    [ seed-one ]
    [ seed-random ]
  update
  do-plots
end

to seed-one
  ;; tell the center patch the rumor
  ask patch 0 0
    [ hear-rumor ]
end

to seed-random
  ;; seed with random number of rumor sources governed by init-clique slider
  ask patches with [times-heard = 0]
    [ if (random-float 100.0) < init-clique
        [ hear-rumor ] ]
end

to go
  if all? patches [times-heard > 0]
    [ stop ]
  ask patches
    [if times-heard > 0
        [ spread-rumor ] ]
  update
  tick
  do-plots
end

to spread-rumor  ;; patch procedure
  let neighbor nobody
  set neighbor n-of random juiciness neighbors;; based on the juiciness of the rumor, it will tell this
  ;; # of its 8 neighbors
  ask neighbor [ set just-heard? true ];; tells the neighbor that they have now just heard the rumor
end

to hear-rumor  ;; patch procedure
  if first-heard = -1
    [ set first-heard ticks
      set just-heard? true ]
  set times-heard (times-heard + 1)
  recolor
end

to update
  ask patches with [just-heard?]
    [ set just-heard? false
      hear-rumor ]
end

;;; coloring procedures

to recolor  ;; patch procedure
  ifelse color-mode = 0
    [ recolor-normal ]
    [ ifelse color-mode = 1
      [ recolor-by-when-heard ]
      [ recolor-by-times-heard ] ]
end

to recolor-normal  ;; patch procedure
  ifelse first-heard >= 0
    [ set pcolor red ]
    [ set pcolor blue ]
end

to recolor-by-when-heard  ;; patch procedure
  ifelse first-heard = -1
    [ set pcolor black ]
    [ set pcolor scale-color yellow first-heard world-width 0 ]
end

to recolor-by-times-heard   ;; patch procedure
  set pcolor scale-color green times-heard 0 world-width
end

;;; mouse handling

to spread-rumor-with-mouse
  if mouse-down?
    [ ask patch mouse-xcor mouse-ycor
        [ hear-rumor ]
        display ]
end

;;; plotting procedures

to setup-plots
  set-current-plot "Successive Differences"
  set-plot-y-range 0 (count patches / 5)
end

to do-plots
  let new-clique count patches with [times-heard > 0]
  set-current-plot "Rumor Spread"
  plot (new-clique / count patches) * 100
  set-current-plot "Successive Ratios"
  ifelse clique = 0
    [ plot 1 ]
    [ plot new-clique / clique ]
  set-current-plot "Successive Differences"
  plot new-clique - clique
  set clique new-clique
end


; Copyright 1997 Uri Wilensky. All rights reserved.
; The full copyright notice is in the Information tab.
@#$#@#$#@
GRAPHICS-WINDOW
192
10
505
344
50
50
3.0
1
10
1
1
1
0
1
1
1
-50
50
-50
50
1
1
1
ticks

BUTTON
16
75
137
108
setup-random
setup false
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

PLOT
233
346
493
525
Successive Differences
time
difference
0.0
20.0
0.0
100.0
true
false
PENS
"default" 1.0 0 -2674135 true

PLOT
496
346
712
525
Successive Ratios
time
ratio
0.0
20.0
0.0
2.0
true
false
PENS
"default" 1.0 0 -2674135 true

PLOT
9
346
232
525
Rumor Spread
time
percent
0.0
20.0
0.0
100.0
true
false
PENS
"default" 1.0 0 -2674135 true

BUTTON
1
202
78
235
step
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

SLIDER
11
118
159
151
init-clique
init-clique
0
10
0.1
0.1
1
NIL
HORIZONTAL

BUTTON
16
41
137
74
setup-one
setup true
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

BUTTON
79
202
148
235
go
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL

BUTTON
538
115
684
148
color: normal
set color-mode 0\\nask patches\\n  [ recolor ]
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

BUTTON
538
149
684
182
color: when heard
set color-mode 1\\nask patches\\n  [ recolor ]
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

BUTTON
538
183
684
216
color: times heard
set color-mode 2\\nask patches\\n  [ recolor ]
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

MONITOR
25
252
115
297
clique %
(clique / count patches)\\n  * 100
3
1
11

BUTTON
522
55
703
88
NIL
spread-rumor-with-mouse
T
1
T
OBSERVER
NIL
NIL
NIL
NIL

SLIDER
3
159
175
192
juiciness
juiciness
0
8
4
1
1
NIL
HORIZONTAL

@#$#@#$#@
WHAT IS IT?
-----------
This program models the spread of a rumor.  The rumor spreads when a person who knows the rumor tells one of their neighbors.  In other words, spatial proximity is a determining factor as to how soon (and perhaps how often) a given individual will hear the rumor.

The neighbors can be defined as either the four adjacent people or the eight adjacent people.  At each time step, every person who knows the rumor randomly chooses a neighbor to tell the rumor to.  The simulation keeps track of who knows the rumor, how many people know the rumor, and how many "repeated tellings" of the rumor occur.


HOW TO USE IT
-------------
As with any rumor, it has to start somewhere, with one or more individuals.  There are three ways to control the start of the rumor:

1) Single source: Press the SETUP-ONE button. This starts the rumor at one point in the center of the world.
2) Random source: Press the SETUP-RANDOM button with the INIT-CLIQUE slider set greater than 0. This "seeds" the rumor randomly by choosing a percentage of the population that knows the rumor initially.  This percentage is set using the INIT-CLIQUE slider.
3) Choose source with mouse: Press either SETUP-ONE or SETUP-RANDOM, then press the SPREAD-RUMOR-WITH-MOUSE button.  While this button is down, clicking the mouse button on a patch in the view will tell the rumor to that patch.

To run the model, you can either "step" through each time step using the STEP button or allow the model to simply run continuously using the GO button.  The model will stop when everyone in the population knows the rumor.

There are three plot windows associated with this rumor model.

RUMOR SPREAD - plots the percentage of people who know the rumor at each time step.
SUCCESSIVE DIFFERENCES - plots the number of new people who are hearing the rumor at each time step.
SUCCESSIVE RATIOS - plots the percentage of new people who are hearing the rumor at each time step.

The monitor CLIQUE% is the percentage of the patches that have heard the rumor.

The three coloring buttons to the right of the view give you topographic maps of the world.  The COLOR: WHEN HEARD button colors the patches different shades of YELLOW according to the first time that location heard the rumor.  The COLOR: TIMES HEARD button colors the patches different shades of GREEN according to the number of times that location has heard the rumor.


THINGS TO NOTICE
----------------
The most interesting models to run are those where only a small number of individuals who initially know the rumor.  Use the SPREAD-RUMOR-WITH-MOUSE button to make an initial setup where only three or four patches know the rumor, and all of these patches are in the left half of the world.  Run your model once with "Wrap" off and then again with "Wrap" on.  notice the difference in how the rumor is spread.  Which version seems more realistic to you?

An interesting thing to notice about the spread of the rumor is that the "speed" with which the rumor spreads slows down as more and more people know the rumor.  Why is that?  How is that related to the number of "repeated" or "wasted" tellings of the rumor?  How do the two "differences" plot windows help you to understand the dynamics of the rumor spread?


THINGS TO TRY
-------------
Change the shape of the world (by changing the wrap settings in edit World & View) and see how the spread of the rumor changes in a box or a cylinder.

Use SPREAD-RUMOR-WITH-MOUSE to place four "seeds" for the rumor centered in each of the four quadrants of the grid.  Notice the pattern of "repeated tellings."  Move the four "seeds" closer into the center of the grid.  How does the pattern or "repeated tellings" change?  Move the "seeds" away from the center of the grid.  How does the pattern of repeated tellings change?  How will it change if  you turn "wrap" on or off?

Explore other patterns of seeding the rumor and its impact on the pattern of repeated tellings.

REFLECTION
-------------------
Fireworks
When gravity is higher, the fireworks do not go as high and the fragments come down much quicker.  Gravity is a force that accelerates objects towards the center of the earth and opposes the vertical component of an object_ motion.  Therefore, the firework and its fragments will slow down faster as it reaches the peak and accelerate quicker downwards compared to when the gravity = 1 (assuming this is the status quo).

When gravity = 0, there is no longer any force that is opposing the fireworks_motion, so they will continue to progress in the x and y direction that they were launched.  The fragments would then be released when the explosion is timed.  they will then radiate outwards and those fragments will then proceed in that trajectory indefinitely.  The color will continue until the chemical reaction that produces the color finishes.  According to Newton_ 1st Law of Motion, an object that is in motion will remain in motion unless acted on by an outside force.  Therefore, without the presence of gravity, the fragments will continue to move in their original trajectory until some other force opposes its motion (e.g. it runs into something).

Virus
In the initial set up of virus it is interesting to let the scenario play out for 27 years as is outlined on the _nformation_because what is seen is that about half the population builds up immunity to the infection (basically 350 to 390 of the population of 750 approximately).  The chance of recovery in the initial scenario was 50% so that was the first element with which to experiment.  

Only changing the chance-recovery to slightly less than 50% (to 47%) did not yield any huge impact to the overall scenario because number of immune people decreased slightly and the number of sick was also slightly less than in first scenario.  

In the next variation, we looked at a lower infectious virus (about 11%) and only having a chance of recover of 17%; this scenario the virus basically died out after only 10 years.  Next we increased the infectiousness to 60% and kept the chance-recovery at 17% -- this scenario offered a big difference because the size of the overall population was affected _it dropped down to as low as 225.  Furthermore, in this scenario sick individuals are the largest portion of the population at 160 to 241 in years 20 through 27.  

Finally, we looked at what happens with a duration of 4 weeks, infectiousness at 77% and chance-recovery at 41% -- and with these parameters the sick were the smallest segment of the population with healthy and immune and healthy people being the largest parts of the population, and overall population not being affected.


Rumor Mill
It is more realistic to have the _rapping off_ as if the visible view is the total number of people in a population, you wouldn_ actually have people _ff the screen_  In the wrapping view, it assumes that if it runs off the page, the rumor will spread to people on the opposite side of town, which is not realistic and does not fit the model, as it is based on the people who are closest to you at any particular moment.

When it is switched to _ight-mode_ the rumor spread is at much quicker rate, which makes sense as each _gent_is spreading it to twice as many people at each cycle, so the rate of exponential increase is much higher.

As the rumor matures, there are less people who do not know the rumor.  Therefore, as the agents look around to the four nearest people, their efforts are futile because they have already heard the rumor, thus a wasted telling.  Therefore, we would expect that there would be a higher rate of _epeated_tellings as the rumor matures and this should correlate with a slower rate of spread of the rumor.  

The successive differences graph shows how many people are hearing the rumor for the first time, and we can see that there is a general increase until a certain point, and then we start to see greater diminishing returns for the tellings and fewer new people are hearing the rumor for the first time.  The peaks of this graph and the previous graph correspond to each other.

The successive ratio graph uses the above data to get the percentage of the population that is hearing the rumor for the first time, and we can see that it_ at constant rate (~1% increase at every step.)

When you change the shape of the world to a cylinder versus a box, the time to spread the rumor is the same, but the rate of increase begins to decrease much earlier.  The agents have more access to people because the world wraps, unlike the box, so they have greater access to people who have not heard the rumor, which explains the more rapid increase and the higher peak in the number of people who are hearing the rumor for the first time.  However, it will quickly drop because they have running out of people to who to tell the rumor.

When you have the rumor start in each of four quadrants with the wrap on it spreads relatively fast in 108 clicks and has a typical distribution reaching its highest point of 261 by the 48 tick or click.  When you do the same distribution but controlling for the "wrap off" you see the rumor is 100% spread by the 97th click - so it isn't making a huge difference.  You see the rumor spread more quickly when the "seeds" are initally well distributed and the rumor spreads less quickly when the rumor starts in a tighter geographical area.  When you move the "seeds" of the rumor away from the source then the rumor spreads more quickly.  When the wrap is off, then the rumor transmission is slowed because there is a barrier to spreading the rumor.   


OUR EXTENSION
-------------------
We chose to extend this model by including a _uiciness_factor that would influence how readily the rumor would be spread.  This depicts reality, as if the secret is juicier, a person is more likely to tell the rumor compared to when it is less interesting.  Our _uiciness_scale ranged from 0-8, which correlated to the number of neighbors around the agent.  We removed the  _ight-mode?_slider and the associated steps in the _pread rumor_procedure and added the following command:

- _et neighbor n-of random juiciness neighbors_
To replace the steps that said whether the rumor would be spread to 4 or 8 neighbors, this step basically said that it would set the maximum number of neighbors to whom the agent would be spreading the rumor, as some individuals may choose not to spread the rumor despite its juiciness.  This number is determined by _uiciness_ which was set by the slider on the interface.

Upon setup, the _uiciness_factor is set at 0.  As the slider is increased, the time for 100% coverage decreases.


NETLOGO FEATURES
----------------
Note the use of the "neighbors" primitive to implement 8-mode.

RELATED MODELS
--------------
Virus, AIDS


CREDITS AND REFERENCES
----------------------
This model is itself an extension of a physical experiment where spatial proximity was not a factor in the spread of the rumor.  Contact Helen M. Doerr at hmdoerr@syr.edu regarding papers in preparation.  Thanks to Dr. Doerr for inspiration for this model.


HOW TO CITE
-----------
If you mention this model in an academic publication, we ask that you include these citations for the model itself and for the NetLogo software:
- Wilensky, U. (1997).  NetLogo Rumor Mill model.  http://ccl.northwestern.edu/netlogo/models/RumorMill.  Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.
- Wilensky, U. (1999). NetLogo. http://ccl.northwestern.edu/netlogo/. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

In other publications, please use:
- Copyright 1997 Uri Wilensky. All rights reserved. See http://ccl.northwestern.edu/netlogo/models/RumorMill for terms of use.


COPYRIGHT NOTICE
----------------
Copyright 1997 Uri Wilensky. All rights reserved.

Permission to use, modify or redistribute this model is hereby granted, provided that both of the following requirements are followed:
a) this copyright notice is included.
b) this model will not be redistributed for profit without permission from Uri Wilensky. Contact Uri Wilensky for appropriate licenses for redistribution for profit.

This model was created as part of the project: CONNECTED MATHEMATICS: MAKING SENSE OF COMPLEX PHENOMENA THROUGH BUILDING OBJECT-BASED PARALLEL MODELS (OBPML).  The project gratefully acknowledges the support of the National Science Foundation (Applications of Advanced Technologies Program) -- grant numbers RED #9552950 and REC #9632612.

This model was converted to NetLogo as part of the projects: PARTICIPATORY SIMULATIONS: NETWORK-BASED DESIGN FOR SYSTEMS LEARNING IN CLASSROOMS and/or INTEGRATED SIMULATION AND MODELING ENVIRONMENT. The project gratefully acknowledges the support of the National Science Foundation (REPP & ROLE programs) -- grant numbers REC #9814682 and REC-0126227. Converted from StarLogoT to NetLogo, 2001.

@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 4.1.2
@#$#@#$#@
setup true
repeat 90 [ go ]
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
