;;;;;;;;;;;;;;;;;;;;;;
;;; Identification ;;;
;;;;;;;;;;;;;;;;;;;;;;

;; Program Name: REGULOPOLIS
;; Author(s): Tony Casey
;; Email: tony.casey@ucdconnect.ie
;; Requirements: Netlogo 5.2

;;;;;;;;;;;;;;;;;;;;;;;
;;; Version Control ;;;
;;;;;;;;;;;;;;;;;;;;;;;

;; Version: v0.01  9th  July 2015   (Tony Casey)    Initial version.
;;          v0.1   23rd July 2015   (Tony Casey)    Added in basic 'resource' and 'distance' based modeling.
;;                                                  Added in 'review panel' logic.
;;                                                  Added in 'spring' visualisation logic (after Uri Wilensky).
;;                                                  Split orchestration / choreography code.
;;                                                  Added in 'scorecard' logic.
;;                                                  Added in ECJ enforcement logic.
;;          v0.2   24th July 2015   (Tony Casey)    Added in 'policy interdependence' logic, weighted links are conceptualised as a proxy for policy interdependence.
;;          v0.3   30th July 2015   (Tony Casey)    Added in logic to ensure that algorithmically generated hubs are plausible, i.e. SRA's and large countries.
;;          v0.4a  31st July 2015   (Tony Casey)    Added in path-distance from NRA to Agency/Network logic as proxy for time to complete OECD stage
;;          v0.5   01st Aug  2015   (Tony Casey)    Added in variable centrality
;;          v0.6b  13th Aug  2015   (Tony Casey)    Added in 'oversight'logic (a.k.a. Abbott et al.), compliance plot
;;          v0.6c  14th Aug  2015   (Tony Casey)    Added in 'goal divergence logic' (a.k.a Abbott et al.), refined compliance plot, much tweaking of compliance logic

;;;;;;;;;;;;;;;;;;;;;;
;;; General ToDo's ;;;
;;;;;;;;;;;;;;;;;;;;;;

;; Tidy up temporarilyInSetUpCode (any stable parameters should be moved to user-selectable interface).
;; Independent domestic oversight variable for each NRA
;; Independent goal divergence variable for each NRA

;;;;;;;;;;;;;;;;
;;; Acronyms ;;;
;;;;;;;;;;;;;;;;

;; ERR           European Regulatory Regime
;; ERN           European Regulatory Network
;; IN            Independent Network
;; CN            Commission Network
;; A             Agency;
;; AN            Agencified Network
;; NA            Networked Agency
;; NRA           National Regulatory Authority
;; SNA           Supra National Authority
;; NT            Node Type
;; EPC           Epistimic Community
;; COM           European Commission
;; EPRG          European Regulator Group for Posal Services
;; MS            Member State



;;;;;;;;;;;;;;;;;;
;;; Extensions ;;;
;;;;;;;;;;;;;;;;;;

extensions [matrix nw]

;;;;;;;;;;;;;;;;;
;;; Interface ;;;
;;;;;;;;;;;;;;;;;

;; enforcementPyramid?        ; user defined chooser on inteface, takes values 'system defined (all)', 'EU review panel', 'EU scorecard', 'ECJ ruling', from 'soft' to 'hard' enforecement.
;; reveiwPanelEffect?         ; user defined slider on interface, 0 to 100% uplift of regulatory compliance on 'regulatory gap' as a result of effect of review panel.
;; rcPeerEffectOnGap?         ; user defined slider on interface, 0 to 100% uplift of regulatory compliance on 'regulatory gap' as a result of effect of scorecards.
;; preferential-attachment?   ; type of 'centrality' measure, used to layout visualisation and weight path distance from NRA to central agency/network
;; tight DomesticOversight?   ; user defined switch on interface, simple global variable, in later versions could be set independently for each NRA
;; goalDivergence?            ; user defined slider on interface, takes values 0 = none, 1 = medium, 2 = high

;;;;;;;;;;;;;;;;;;;;;;;;
;; Agent Data Model  ;;;
;;;;;;;;;;;;;;;;;;;;;;;;

globals [
  euEnlargementNum            ; can take values 6, 12, 15, 25, 28
  errNum                      ; varies depending how many non-NRA agents
  errList                     ; varies depending on EU-6, EU-12, EU-15, EU-25, EU-28, Commission, Epistemic Community, Independent Network etc.
  errMatrix                   ; to hold externally excell created text matrix
  errType                     ; "12" used (1-12 Levi-Faur ERR types identified, differentiates topology and processing rules, partially used this version)
  errRegulatoryDistance       ; holds list representing "distance" of new eu-regime to existing nra-regime
  errSharedOntology           ; holds list representing shared "understanding" i.e. existance/adoption of standards = high, non-existance/non-adoption = low
  errCycle                    ; counter holding the current regulatory regime cycle (used to maintain overall state of regulatory regime)
  errRegTranspositionNum      ; stage number for regulatory transposition, currently 7 (stage 7)
  avgCompliance               ; regulatory regime average compliance (total pct-complance / euEnlargementNum)
  medCompliance               ; regulatory regime median compliance
  regulatory-hub1             ; keep a track of where the important hubs are
  ia-stage-time-required      ; default is 60 days per ia-stage, but this can be multiplied by an NRA's shortest path to an EU agency or network as a proxy for co-ordination effort required
  allTickedOut                ; so we know when to stop the model
  p                           ; general purpose counter
]

breed [errNodes errNode]      ; European Regulatory Regime Node, EU-15 or Commission, Epistemic Community, Independent Network
breed [SNAs SNA]              ; "EPRG" used (European Regulator Group for Postal Services, many other INs identified  by Levi-Faur, partially used this version)
breed [NRAs NRA]              ; "EU-15" used (To impliment EU-6, EU-12, EU-15, EU-25, EU-28, partially used this version)
breed [INs IN]                ; not used this version
breed [CNs CN]                ; not used this version
breed [As A]                  ; not used this version
breed [ANs AN]                ; not used this version
breed [NAs NA]                ; not used this version

errNodes-own [
  node-id                     ; IE, UK, DE, EPRG etc.
  node-type                   ; SNA, NRA, EPC etc.)
  ia-stage                    ; "impact assessment-stage" values 0 to 8, drives the transposition of EU regulation to domestic regulation (based on OECD (OffCom) 'Better Regulation', 2005) 
                              ; (Stage) 0:  undefined
                              ; (Stage) 1:  defining the issue we need to consider and identifying the citizen or consumer interest
                              ; (Stage) 2:  defining the policy objective
                              ; (Stage) 3:  identifying the options
                              ; (Stage) 4:  identifying the impacts on different types of stakeholders
                              ; (Stage) 5:  identifying any impacts on competition
                              ; (Stage) 6:  assessing the impacts and choosing the best option
                              ; (Stage) 7:  transpose EU regulation to domestic regulation
                              ; (Stage) 8:  (enforcement pyramid) EU review panel
                              ; (Stage) 9:  (enforcement pyramid) EU scorecard
                              ; (Stage) 10: (enforcement pyramid) ECJ ruling
  ia-stage-time-passed        ; value is 0 to 59, placeholder for elapsed time at each is-stage
  stand-off                   ; binary, takes values "yes" "no", it is possible that regulation is 'moth-balled' indefinitely in consultation stalement with stake-holders 
  oecd-ceiling                ; where in the OECD IA cycle we are stuck, currently takes the values default 0 (all OECD IA stages possible) but the value 4 if a strong NRA is pushing back and the value 6 if a weak NRA cannot get it accross the domestic legislative line  
  eurozone-cluster            ; PIIGS, Northern, Scandinavian etc., not used this version
  regulatory-distance         ; distance of existing NRA regime from proposed EU regime
  enforcement-escalation      ; takes values 0,1,2,3; maintains where each NRA is in the enforcement pyramid
  pct-compliance              ; percent compliance of NRA regime to proposed EU regime
  world-view                  ; distance of errNode view from "zeitgeist"
  ontological-distance        ; distance of "formal" standards
  zeitgeist                   ; owned by the Epistemic Community          
  budget                      ; owned by the European Commission
  resources                   ; staff, budget, capacity
  all-ticked-out              ; so we know when to stop NRA  
]

SNAs-own [
  working-groups
  review-panels
]

links-own [
  policyInterdependence       ; a.k.a. 'weight'
]

;;;;;;;;;;;;;;;;;;;;;;;;
;;; Setup Procedures ;;;
;;;;;;;;;;;;;;;;;;;;;;;; 

to setup
  clear-all
  set-default-shape errNodes "circle"
  setupERR                    ; setup the European Regulatory Regime
  reset-ticks
end

to setupERR
  temporarilyInSetUpCode      ; all this procedure to be user selectable and eventually moved to interface
  setupERRNetwork
  setupERRNodes
  setupERRLinks
end

to temporarilyInSetUpCode     ; general first-cut dumping ground for potential independent variable parameters before moving to user selectable interface
  set errNum 18               ; EU-15, plus Commission, Epistemic Community, Independent Network
  set euEnlargementNum 15     ; EU-15
  set errType 12
  ; test data arranged in ascending order of centrality
  set errList ["LU" "FI" "IE" "PT" "EL" "AT" "DK" "BE" "NL" "ES" "SE" "IT" "UK" "FR" "DE" "EPC" "Commission" "Reg Agency"]
  ; test data which is mapped to errList
  set errRegulatoryDistance [0 1 0 -2 -2 0 0 -1 0 0 1 1 1 2 0 1 0 0]
  ; test data which is mapped to errList
  ;set errSharedOntology [1 0 0 0 -1 0 0 1 -1 0 0 0 0 0 1 0 1 -1]
  set-current-directory "C:\\Users\\Tony\\OneDrive\\Msc Public Policy (UCD)\\Course Descriptions\\Thesis (GSHS40200)\\My NetLogo Code"
  set errMatrix "errMatrix.txt"
  set errRegTranspositionNum 7; currently 7 because 'regulatory transposition happens at 7 in the 8 stage OECD impact assessment model
  set ia-stage-time-required 2;default is 60 days per stage * length of NRA path to Agency/Network hub (as a proxy for the coordination effort required) 
  set allTickedOut "no"
end

to setupERRNetwork            
  if regulatoryForm? = "orchestration (star)" [                                          ; create 'star' network from excel sheet 
    nw:load-matrix errMatrix errNodes links]                                             ; test matrix to be replaced with empirically constructed matrix in later versions 
  if regulatoryForm? = "choreography (small-world)" [                                    ; create 'small-world' network using the generator in netlogo nw exension                                           
    nw:generate-small-world errNodes links 6 3 2.0 false]                                ; with live data this will be swapped out for empirically constructed matrix
  if regulatoryForm? = "choreography (scale-free)" [                                     ; create 'scale-free' network using the generator in netlogo nw exension                                           
    nw:generate-preferential-attachment errNodes links errNum]                           ; with live data this will be swapped out for empirically constructed matrix
end

to setupERRNodes                                                                         ; populate network nodes
  if regulatoryForm? = "orchestration (star)" [                                          ; Nice and easy, matrix data is already ordered plausibly
    ask errNodes [
      print (word "node-id = " node-id " who = " who)
      set node-id item who errList
      print (word "node-id = " node-id " who = " who)
      set regulatory-distance item who errRegulatoryDistance
      set ia-stage 0
      set ia-stage-time-passed 0
      set resources 50
      set pct-compliance 0
      set all-ticked-out "no"
      set stand-off "no"
      set oecd-ceiling 0
      if labelsOn? [
        set label node-id]   
      ifelse (node-id != "Reg Agency") and (node-id != "Commission") and (node-id != "EPC")
        [set node-type "NRA"
         set color red
        ]         
        [set node-type "SRA"         
         if node-id = "Reg Agency"[       
           set regulatory-hub1 who]                                                     ; so we can calculate the shortest distance
         ifelse node-id = "EPC"
           [set color green]
           [set color blue]
         ]           
    ]  
  ]
  if (regulatoryForm? = "choreography (small-world)") or (regulatoryForm? = "choreography (scale-free)") [ ; These networks generated algorithmically so needs some pre-processing to ensure hubs are plauasable, i.e SRA's and big countries 
    set p 0
    foreach sort-on [runresult measure] errNodes [
      ask ? [
        set node-id item p errList
        print (word "p = " p " errList = " (item p errList))                            ; debug code
        set regulatory-distance item who errRegulatoryDistance
        set ia-stage 0
        set ia-stage-time-passed 0
        set resources 50
        set pct-compliance 0
        set all-ticked-out "no"
        set stand-off "no"
        set oecd-ceiling 0
        if labelsOn? [
          set label node-id]   
        ifelse (node-id != "Reg Agency") and (node-id != "Commission") and (node-id != "EPC")
          [set node-type "NRA"
           set color red
          ]         
          [set node-type "SRA"
             if node-id = "Reg Agency" [
               set regulatory-hub1 who
               print (word "regulatory-hub1 = " regulatory-hub1)                        ; debug code
               set node-id "Reg Body"
               if labelsOn? [
                 set label node-id]                                                     ; No agencies in this regulatory form
             ]
             ifelse node-id = "EPC"
               [set color green]
               [set color blue]
          ]
      set p p + 1
      ]  
    ] 
 ]
end

to setupERRLinks
  if (regulatoryForm? = "choreography (small-world)") or (regulatoryForm? = "choreography (scale-free)") [
    ; these networks generated algorithmically so make sure there are no orphan "EPC" or "Commission" nodes not linked to "Reg Body"
    ask errNodes [
      if (node-id = "EPC") or (node-id = "Commission") [
        if not (link-neighbor? errNode regulatory-hub1) [
          create-link-with (errNode regulatory-hub1) ]
      ]   
    ]
  ] 
end 

;;;;;;;;;;;;;;;;;;;;;;;
;;; Main Procedures ;;;
;;;;;;;;;;;;;;;;;;;;;;;

to go
  ifelse allTickedOut = "no" [    
    errConverge                                                                         ; main processing loop, NRA regulation 'converges' on EU directive
    foreach sort errNodes [                                                             ; debug code
      ask ? [print (word "who = " who " id = " node-id " node-type = " node-type " ia = " ia-stage " rd = " regulatory-distance " resources = " resources " tick = " ticks " oecd-ceiling = " oecd-ceiling " pct compliance = " pct-compliance)]
      ask ? [print (word "path length to agency/network = " (nw:distance-to (errNode regulatory-hub1)))]
    ]
  ]
    [stop]
  if all? (errNodes with [node-type = "NRA"]) [all-ticked-out = "yes"]
    [set allTickedOut "yes"]
  adjust-sizes
  layout
  tick
end  

to errConverge                                                                   ; loop through the OECD 'better regulation' impact assessment life-cycle; core logic is 'resources' and 'distance' based (to which is be added progressively more sophisticated agent behaviour)
  foreach sort errNodes [
    ask ? [
      if (ia-stage < errRegTranspositionNum) [                                   ; i.e. regulation not yet 'transposed'
        ifelse ia-stage = 0 [
          if regulatory-distance = 0
            [set ia-stage 3]                                                     ; low domestic alignment required
          if (regulatory-distance = 1) or (regulatory-distance = -1)             ; left/right behaviour not differentiated this version
            [set ia-stage 2]                                                     ; higher domestic alignment required
          if (regulatory-distance = 2) or (regulatory-distance = -2) [           ; left/right behaviour not differentiated this version
            if tightDomesticOversight? [
              set oecd-ceiling 4]                                                ; the strong NRA is going to push back
            set ia-stage 1                                                       ; highest domestic alignment required
          ]                                                       
          ]       
          [ifelse ia-stage-time-passed < (ia-stage-time-required * (nw:distance-to (errNode regulatory-hub1)))
             [set ia-stage-time-passed ia-stage-time-passed + 1]
             [ifelse oecd-ceiling != 0                                           ; put the strong NRA into a holding pattern
               [ifelse (ia-stage = oecd-ceiling) and (enforcementPyramid? != "3. ECJ ruling (all)") ; nuclear option required to break out of holding pattern
                 [set ia-stage-time-passed 0 
                   set stand-off "yes"
                   if all-ticked-out = "no" [
                     set resources resources - 5]
                 ]
                 [set ia-stage ia-stage + 1
                  set ia-stage-time-passed 0 
                  if ia-stage = oecd-ceiling [
                    set stand-off "yes"
                  ]
                  if all-ticked-out = "no" [
                    set resources resources - 5]
                 ]
               ]             
               [set ia-stage ia-stage + 1                                         
                set ia-stage-time-passed 0 
                if all-ticked-out = "no" [
                  set resources resources - 5]
               ]
            ]
          ]
      ]          
      if (ia-stage = errRegTranspositionNum) and (node-type = "NRA") [           ; OECD stage 7, regulation transposed, set pct-compliance of NRAs
        ifelse ia-stage-time-passed < (ia-stage-time-required * (nw:distance-to (errNode regulatory-hub1)))
           [set ia-stage-time-passed ia-stage-time-passed + 1]
           [set ia-stage-time-passed 0
            set ia-stage ia-stage + 1                                             
            if all-ticked-out = "no" [
                set resources resources - 5]       
            if (regulatory-distance = 0)   
              [set pct-compliance 90]
            if (regulatory-distance = 1) or (regulatory-distance = -1)
              [set pct-compliance 70]
            if (regulatory-distance = 2) or (regulatory-distance = -2)
              [set pct-compliance 50]
            if (enforcementPyramid? = "4. (none)")                                                                
              [set all-ticked-out "yes"]
            ]     
        ]      

      if (ia-stage > errRegTranspositionNum) and (node-type = "NRA") [         
      ; finish, regulation 'transposed' but EU enforcement pyramid may apply 'peer' preasure to increase domestic compliance 
        if (enforcementPyramid? = "1. EU review panel (1 only)")[                                                                
          if ia-stage = errRegTranspositionNum + 1 [
            ifelse ia-stage-time-passed < (ia-stage-time-required * (nw:distance-to (errNode regulatory-hub1))) [
              errReviewPanel
              set ia-stage-time-passed ia-stage-time-passed + 1]
             [if all-ticked-out = "no" [
                set resources resources - 5]
              set all-ticked-out "yes"]
          ]
        ]
        if (enforcementPyramid? = "2. EU scorecard (1 and 2)") or (enforcementPyramid? = "0. 'soft' default (1 and 2)")[
          if ia-stage = errRegTranspositionNum + 1 [
            ifelse ia-stage-time-passed < (ia-stage-time-required * (nw:distance-to (errNode regulatory-hub1))) [
              errReviewPanel
              set ia-stage-time-passed ia-stage-time-passed + 1]
             [set ia-stage ia-stage + 1
              set ia-stage-time-passed 0                                        
              if all-ticked-out = "no" [
                set resources resources - 5]
             ]
          ]        
          if (ia-stage = errRegTranspositionNum + 2) [
            ifelse ia-stage-time-passed < (ia-stage-time-required * (nw:distance-to (errNode regulatory-hub1))) [
               print (word "got into 2nd loop " "ia-stage = " ia-stage)
               errScoreCard
               set ia-stage-time-passed ia-stage-time-passed + 1]
              [if all-ticked-out = "no" [
                 set resources resources - 5]
               set all-ticked-out "yes"]
          ]
        ]
        if (enforcementPyramid? = "3. ECJ ruling (all)") [   
          if ia-stage = errRegTranspositionNum + 1 [
            ifelse ia-stage-time-passed < (ia-stage-time-required * (nw:distance-to (errNode regulatory-hub1))) [
              print (word "got into 1st loop " "ia-stage = " ia-stage)
              errReviewPanel
              set ia-stage-time-passed ia-stage-time-passed + 1]
             [set ia-stage ia-stage + 1
              set ia-stage-time-passed 0                                         
              if all-ticked-out = "no" [
                set resources resources - 5]
             ]
          ]        
          if (ia-stage = errRegTranspositionNum + 2) [
            ifelse ia-stage-time-passed < (ia-stage-time-required * (nw:distance-to (errNode regulatory-hub1))) [
              print (word "got into 2nd loop " "ia-stage = " ia-stage) 
              errScoreCard
              set ia-stage-time-passed ia-stage-time-passed + 1]
             [set ia-stage ia-stage + 1                                         
              set ia-stage-time-passed 0
              if all-ticked-out = "no" [
                set resources resources - 5]
            ]
          ]      
          if (ia-stage = errRegTranspositionNum + 3) and (regulatoryForm? = "orchestration (star)") [                   ; the nuclear option is only available to agencies with statutory powers
              print (word "got into 2.5th loop " "ia-stage = " ia-stage)
            ifelse (ia-stage-time-passed < (ia-stage-time-required * (nw:distance-to (errNode regulatory-hub1)))) [
              print (word "got into 3rd loop " "ia-stage = " ia-stage)      
              errECJRuling
              set ia-stage-time-passed ia-stage-time-passed + 1]
             [if all-ticked-out = "no" [
                set resources resources - 5]
              if stand-off = "no" [
                set all-ticked-out "yes"]
             ]
          ]
          if (ia-stage = errRegTranspositionNum + 3) and (regulatoryForm? != "orchestration (star)") [
              if stand-off = "no" [
                set all-ticked-out "yes"]
          ]
        ]
      ]  
      ]
    ]
end

;to-report path-length
;  report task [ nw:path-to turtle (who with (node-id = "Reg Agency")) ]
;end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Enforcement Pyramid Procedures ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to errReviewPanel                                                                                                  ; modelled to effect 'volume' of domestic regulation (after Maggetti & Gilardi, 2014)                                                                 
  set avgCompliance mean [pct-compliance] of errNodes with [node-type = "NRA"]                                     ; peer uplift a pct of 'un-transposed' i.e. the regulatory gap, another possibility is simple peer uplift of pct 'transposed', should be user selectable in new version
  set medCompliance median [pct-compliance] of errNodes with [node-type = "NRA"]
  print (word "pct compliance = " pct-compliance " rp average compliance = " avgCompliance " oecd ceiling = " oecd-ceiling " ia-stage = " ia-stage)                         ; debug code
  ifelse (pct-compliance >= avgCompliance) or (node-type = "SRA")  
    [set pct-compliance pct-compliance * 1]                                                                        ; do nothing, making assumption that peer preasure has no effect when transposition already greater than or equal to the average or a strong NRA.
    [ifelse oecd-ceiling = 0
      [set pct-compliance (((100 - pct-compliance) * (1 + (reveiwPanelEffect? / 100))) - (100 - pct-compliance)) + pct-compliance] ; simulate 'peer' preasure to increase compliance of 'regulatory gap' by (default) 20%
      [set pct-compliance pct-compliance * 1
        if oecd-ceiling = ia-stage [
          set stand-off "yes"
        ] 
      ]
    ] 
  set avgCompliance mean [pct-compliance] of errNodes with [node-type = "NRA"] 
  set medCompliance median [pct-compliance] of errNodes with [node-type = "NRA"]
  print (word "rp avg compliance = " avgCompliance)                                                                 ; debug code
  print (word "rp median compliance = " medCompliance)                                                              ; debug code
end 

to errScoreCard
  set avgCompliance mean [pct-compliance] of errNodes with [node-type = "NRA"]                                      ; peer uplift a pct of 'un-transposed' i.e. the regulatory gap, another possibility is simple peer uplift of pct 'transposed', should be user selectable in new version
  set medCompliance median [pct-compliance] of errNodes with [node-type = "NRA"]
  ifelse (pct-compliance >= avgCompliance) or (node-type = "SRA")
    [set pct-compliance pct-compliance * 1]                                                                         ; do nothing, making assumption that peer preasure has no effect when transposition already greater than or equal to the average or a strong NRA.
    [ifelse oecd-ceiling = 0
      [set pct-compliance (((100 - pct-compliance) * (1 + (scoreCardEffect? / 100))) - (100 - pct-compliance)) + pct-compliance] ; simulate 'peer' preasure to increase conpliance of 'regulatory gap' by (default) 20%
      [set pct-compliance pct-compliance * 1
        if oecd-ceiling = ia-stage [
          set stand-off "yes"
        ]
      ]
    ]
  set avgCompliance mean [pct-compliance] of errNodes with [node-type = "NRA"] 
  set medCompliance median [pct-compliance] of errNodes with [node-type = "NRA"]
  print (word "sc avg compliance = " avgCompliance)                              ; debug code
  print (word "sc median compliance = " medCompliance)                           ; debug code

end

to errECJRuling
  set avgCompliance mean [pct-compliance] of errNodes with [node-type = "NRA"]   ; this is a 'nuclear' rarely used option, modelled as user selectable probability of ECJ winning the case
  set medCompliance median [pct-compliance] of errNodes with [node-type = "NRA"]
  ifelse (pct-compliance >= (avgCompliance + complianceMargin?)) or (stand-off = "yes") or (node-type = "SRA")                      
    [set pct-compliance pct-compliance * 1                                       ; do nothing, making the assumption this is last resort and reserved for NRAs not reaching a threshold 'median' transposition 
     set stand-off "no"                                                          ; the EC is not going after these
    ]                         
    [ifelse random 100  <= (ECJWinProbability?)                                  ; produces a pseudo-random number between 0 and 99 and if it is less than or equal to the probability of the ECJ winning then the ruling is enforced
       [set pct-compliance 100
        set stand-off "no"]                                                      ; EC wins
       [set stand-off "yes"]

    ]
  set avgCompliance mean [pct-compliance] of errNodes with [node-type = "NRA"] 
  set medCompliance median [pct-compliance] of errNodes with [node-type = "NRA"]
  print (word "sc avg compliance = " avgCompliance)                              ; debug code
  print (word "sc median compliance = " medCompliance)                           ; debug code
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Visualisation Procedures ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to layout
  ;if not layout? [ stop ]
  ;; the number 10 here is arbitrary; more repetitions slows down the
  ;; model, but too few gives poor layouts
  repeat 10 [
    do-layout
    display  ;; so we get smooth animation
  ]
end

to do-layout
  ;layout-spring errNodes links 0.4 6 1                                           ; baseline for tests
  if regulatoryForm? = "orchestration (star)" [                                   ; create 'star' network from excel sheet (text file)
    layout-spring errNodes links 0.3 (world-width / (sqrt errNum)) 1]             ; a bit prettier
  if regulatoryForm? = "choreography (small-world)" [                             ; create 'small-world' network using the generator in netlogo nw exension                                             
    layout-spring errNodes links 0.8 12 2]                                        ; a bit prettier
  if regulatoryForm? = "choreography (scale-free)" [                              ; create 'small-world' network using the generator in netlogo nw exension                                             
    layout-spring errNodes links 0.8 12 2]                                        ; a bit prettier

end

to adjust-sizes
  ;; We want the size of the errNodes to reflect their centrality, but there are different centrality measures 
  ;; giving different ranges of size, so we normalize the sizes according to the formula 
  ;; below. We then use the normalized sizes to pick an appropriate color.
  if count errNodes > 0 [
    let results sort [ runresult measure ] of errNodes ;; results of the measure in increasing order
    let delta last results - first results ;; difference between biggest and smallest
    let base-size world-width / 50 ; set a base-size relative to the width of the world
    ifelse delta = 0 [ ;; if they all have the same value
      ask errNodes [ set size base-size ]
    ]
    [ ;; remap the size to a range of sizes from the base size to quadruple the base size
      let size-range base-size * 2
      ; note that we call runresult measure a second time, but since the centrality results
      ; are stored behind the scene by the nw extension, the cost of doing that is acceptable
      ask errNodes [ set size base-size + (((runresult measure - first results) / delta) * size-range) ]
    ]    
  ]
end

to-report measure
  ; returns a different reporter task that can be run by the turtles (errNodes), depending on which
  ; preferential attachment mechanism is selected in the chooser (to be implimented).
  if preferential-attachment? = "1. degree centrality" [ report task [ count my-links ] ]
  if preferential-attachment? = "2. betweenness centrality" [ report task [ nw:betweenness-centrality ] ]
  if preferential-attachment? = "3. closeness centrality" [ report task [ nw:closeness-centrality ] ]
  if preferential-attachment? = "4. eigenvector centrality" [ report task [ nw:eigenvector-centrality ] ]
  report task [ 1 ] ; used for "none" - they all get equal values
end            



 
@#$#@#$#@
GRAPHICS-WINDOW
449
10
888
470
16
16
13.0
1
10
1
1
1
0
1
1
1
-16
16
-16
16
0
0
1
ticks
30.0

BUTTON
18
40
81
73
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
96
40
159
73
go
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

SLIDER
236
134
408
167
reveiwPanelEffect?
reveiwPanelEffect?
0
100
20
1
1
NIL
HORIZONTAL

CHOOSER
17
154
221
199
regulatoryForm?
regulatoryForm?
"orchestration (star)" "choreography (small-world)" "choreography (scale-free)"
1

CHOOSER
235
37
435
82
enforcementPyramid?
enforcementPyramid?
"0. 'soft' default (1 and 2)" "1. EU review panel (1 only)" "2. EU scorecard (1 and 2)" "3. ECJ ruling (all)" "4. (none)"
3

SWITCH
18
84
127
117
labelsOn?
labelsOn?
0
1
-1000

SLIDER
236
180
408
213
scoreCardEffect?
scoreCardEffect?
0
100
20
1
1
NIL
HORIZONTAL

SLIDER
236
223
408
256
ECJWinProbability?
ECJWinProbability?
0
100
100
1
1
NIL
HORIZONTAL

SLIDER
235
92
407
125
complianceMargin?
complianceMargin?
0
100
20
1
1
NIL
HORIZONTAL

SLIDER
904
152
1086
185
policyInterdependence?
policyInterdependence?
0
5
5
1
1
NIL
HORIZONTAL

SLIDER
905
81
1077
114
pctOutliers?
pctOutliers?
0
100
20
1
1
NIL
HORIZONTAL

SLIDER
905
38
1077
71
regDistanceRange?
regDistanceRange?
0
3
2
1
1
NIL
HORIZONTAL

TEXTBOX
236
10
424
35
Regulatory Enforcement
16
0.0
1

TEXTBOX
17
127
167
147
Topology (Nodes)
16
0.0
1

TEXTBOX
905
124
1055
144
Topology (Links)
16
0.0
1

TEXTBOX
19
10
197
50
REGULOPOLIS v0.6c
16
0.0
1

TEXTBOX
236
266
386
286
Centrality
16
0.0
1

CHOOSER
235
296
428
341
preferential-attachment?
preferential-attachment?
"1. degree centrality" "2. betweenness centrality" "3. closeness centrality" "4. eigenvector centrality"
0

SLIDER
17
251
189
284
goalDivergence?
goalDivergence?
0
2
0
1
1
NIL
HORIZONTAL

SWITCH
17
209
218
242
tightDomesticOversight?
tightDomesticOversight?
0
1
-1000

MONITOR
129
297
218
342
non-compliers
count errNodes with [stand-off = \"yes\"]
17
1
11

PLOT
18
356
218
506
non-compliance
time
non-compliance
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"non-compliers" 1.0 0 -16777216 true "" "plot count errNodes with [stand-off = \"yes\"]"

TEXTBOX
20
294
170
314
Compliance
16
0.0
1

TEXTBOX
904
10
1054
30
Work in Progress
16
0.0
1

@#$#@#$#@
## WHAT IS IT?

(a general understanding of what the model is trying to show or explain)

## HOW IT WORKS

(what rules the agents use to create the overall behavior of the model)

## HOW TO USE IT

(how to use the model, including a description of each of the items in the Interface tab)

## THINGS TO NOTICE

(suggested things for the user to notice while running the model)

## THINGS TO TRY

(suggested things for the user to try to do (move sliders, switches, etc.) with the model)

## EXTENDING THE MODEL

(suggested things to add or change in the Code tab to make the model more complicated, detailed, accurate, etc.)

## NETLOGO FEATURES

(interesting or unusual features of NetLogo that the model uses, particularly in the Code tab; or where workarounds were needed for missing features)

## RELATED MODELS

(models in the NetLogo Models Library and elsewhere which are of related interest)

## CREDITS AND REFERENCES

(a reference to the model's URL on the web if it has one, as well as any other necessary credits, citations, and links)
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.2.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
