extensions [
  nw
]

breed [nodes node]
breed [messages message]

messages-own [
  location ;;holds a node in which a message is
]

nodes-own[
  ;; Use Net
  target ;;target as a message destination
  ;; this RECEIVDED is not used well in this implementation
  ;; But this is useful to see which node is "actually used as a "hub"
  received ;;if a node received, add 1
  ;; Choose Net
  distance-to-other-nodes   ;; list of distances of this node to other turtles
  ;; Local Search
  localset ;; holds a local agentset
  ;; Homophily
  gender ;; gender to either 0 "male" or 1 "female"
]

links-own [
  used ;;if a message went through a link, add 1
]

globals [
  ;; Use Net
  origin ;;holds a starting node
  current-location ;; holds a node where a message is currently
  next-location ;; owns a node where a message goes next
  target-node ;;holds a target node
  unreachable ;;holds true/false
  number-of-unreachable ;;counts unreachable cases
  steps ;;holds count of steps
  steps-record ;;holds a list of steps per session
  min-pl-record ;; holds a list of min path length per session
  min-pl ;; holds min distance between the origin and target
  number-of-sessions ;;owns the count of sessions
  incomplete ;;holds the number of imcomplete cases
  incomplete-case ;;holds true/false
  deadend ;;holds true/false
  number-of-deadend ;;counts deadend cases
  actual-average-pl ;; average path length of the network
  number-of-neighbors ;; holds [out-link-neighbors] of location
  ;; Choose Net
  average-path-length ;; average path length of the network
  pl-list ;; list of distances of this node to other turtles
  unconnected ;; holds the number of unconnected pairs
  connected ;; holds the number of connected pairs
]

;;;;;;;;;;;;;;;;;;;;;
;; Setup Procedure ;;
;;;;;;;;;;;;;;;;;;;;;

to setup
  clear-all
  set-default-shape nodes "circle"
  create-nodes number-of-nodes [set color blue]
  ask nodes [
    create-links-to n-of number-of-links other nodes
    set target one-of other nodes
    set received 0
    set gender random 2 ;; set gender to either 0 "male" or 1 "female"
  ]
  layout
  message-setup
  ask target-node [
    set color red
  ]
  ask nodes [
    set localset (min-n-of 4 other nodes [distance myself]) ;; holds a turtles-own variable
  ]
  local-search
  link-setup
  ask origin [
    ifelse nw:distance-to target = False [
    set min-pl 0 ][
    set min-pl nw:distance-to target]
  ]
  set number-of-sessions 0
  set unreachable false
  set incomplete 0
  unreachable-setup
  set steps-record []
  set min-pl-record []
  ;; Choose Net
  find-path-lengths
  find-connected-pairs
  path-length-list
  find-average
  reset-ticks
end

;;circle layout
to layout
  layout-circle (sort nodes) max-pxcor - 1
end

to go
  if any? messages with [location = target-node][
    set steps number-of-steps
    set steps-record lput number-of-steps steps-record
    set min-pl-record lput min-pl min-pl-record
    set number-of-sessions number-of-sessions + 1
    target-resetup
  ]
  unreachable-setup
  if unreachable = true [
    set unreachable false
    set number-of-unreachable number-of-unreachable + 1
    set steps 0
    set steps-record lput 0 steps-record ;; tentatively assign 0 but INF should be more appropriate...
    set min-pl-record lput 0 min-pl-record
    set number-of-sessions number-of-sessions + 1
    target-resetup
  ]
  wrong-choice
  if deadend = true [
    set deadend false
    set number-of-deadend number-of-deadend + 1
    set steps 0
    set steps-record lput 0 steps-record;; tentatively assign 0 but INF should be more appropriate...
    set min-pl-record lput 0 min-pl-record
    set number-of-sessions number-of-sessions + 1
    target-resetup
  ]
  ask links [
    set thickness 0
    ask links with [used >= 1]
      [set thickness 0.3]
  ]
  message-procedure
  tick
end

;;;;;;;;;;;;;;;;;;;;;;;
;; Use Net procedure ;;
;;;;;;;;;;;;;;;;;;;;;;;

;;mesasge procedure
to message-procedure
  ask messages [
    if not any? [out-link-neighbors] of location [
      stop
    ]
    set current-location location
    ;set upper limits of passing messages through links
    ifelse route-limits? [
      ;; 1 as an arbitrary limit
      let out-link-node [my-out-links with [(used < 1) and (self != myself)]] of current-location
      ifelse  count out-link-node > 1 ;;count out-link-node = number-of-links
      [set number-of-neighbors turtle-set [end2] of out-link-node
        if search-strategy = "random-search"
        [set next-location one-of number-of-neighbors] ;[out-link-neighbors] of location]
        if search-strategy = "local-search"
        [ls-strategy]
        if search-strategy = "homophilious-search"
        [homophily]
        if search-strategy = "preferential-attachment"
        [preferential]
      ]
      ;; if running out of available links a node can use, end the session
      [ifelse count out-link-node = 0
        [set incomplete incomplete + 1
        set incomplete-case true
        stop]
        [set next-location one-of turtle-set [end2] of out-link-node]
      ]
    ][
    set number-of-neighbors [out-link-neighbors] of current-location
      if search-strategy = "random-search"
      [set next-location one-of number-of-neighbors] ;[out-link-neighbors] of location]
      if search-strategy = "local-search"
      [ls-strategy]
      if search-strategy = "homophilious-search"
      [homophily]
      if search-strategy = "preferential-attachment"
      [preferential]
    ]
    let new-location next-location
    ;; change the thickness of the link as a message moves
    ask [out-link-to new-location] of location [set thickness 0.3 set used used + 1]
    face new-location ;; not strictly necessary, but improves the visuals a bit
    move-to new-location
    set location new-location
    ask location [
      set received received + 1 ;; increase received
    ]
  ]
  if incomplete-case = true [
    set number-of-sessions number-of-sessions + 1
    set incomplete-case false
    target-resetup
  ]
end


;;new round setup
to target-resetup
  ask nodes [
    set target one-of other nodes
    ;set received 0
    set color blue
  ]
  ask messages [die]
  message-setup
  ask target-node [
    set color red
  ]
  link-setup
  ;unreachable-setup
  ask origin [
    ifelse nw:distance-to target = False [
    set min-pl 0 ][
    set min-pl nw:distance-to target]
  ]
  reset-ticks
end

;;procedure for target
to message-setup
  create-messages 1 [ ;; TO DO: replace 1 to a slider
    set color orange
    set location one-of nodes
    set origin location
    set target-node [target] of location
    move-to location
  ]
end

;;link weight
to link-setup
  ask links [
    set used 0
    set thickness 0
    ;ask links with [used >= 1]
      ;[set thickness 0.3]
  ]
end

;;detect an unreachable case
to unreachable-setup
  ;;an empty means no path b/w the origin and target exists
  if empty? [nw:path-to target-node] of origin [
    set unreachable true
  ]
end

;;critical mistakes of route choices
;;where a path to the target becomes no longer avaiable
to wrong-choice
  ask messages [
    ask location [
      if empty? nw:path-to target-node [
        set deadend true
      ]
    ]
  ]
end

;; average simulated path length
to-report calculate-average
  let avg-pl filter [? != 0] steps-record
  set actual-average-pl (sum avg-pl) / (length avg-pl)
  report actual-average-pl
end

to-report unreachable-pairs
  report number-of-unreachable
end

to-report number-of-steps
  report ticks
end

to-report sessions
  report number-of-sessions
end

to-report incompletes
  report incomplete
end

;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Choose Net procedure ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;

to path-length-list
  set pl-list []
  foreach sort nodes [
    let x [distance-to-other-nodes] of ?
    set pl-list sentence pl-list x
  ]
end

;; calculate the number of unconnected pairs and connected pairs
to find-connected-pairs
  set connected 0
  set unconnected 0
  let i 0 ;;holds a target turtle number
  let node-count count nodes
  foreach sort nodes[
    ask ? [
      while [ i < node-count ]
      [ifelse ? = node i ;;if a turtle is the same as i, skip it
        [set i i + 1]
        [ifelse empty? nw:path-to node i ;;since if there is no path from a turtle to i, return an empty list
          [
            set i i + 1
            set unconnected unconnected + 1
          ]
          [set i i + 1
            set connected connected + 1]
        ]
      ]
    ]
   set i 0 ;;set the default again
  ]
end

;; calculate distance from turtles to other turtles only if they are connected with each other
to find-path-lengths
  ask nodes
  [
    set distance-to-other-nodes filter [is-number? ?] [[nw:distance-to myself] of myself] of other nodes
  ]
end

;; average path length
to find-average
  set average-path-length (sum [sum distance-to-other-nodes] of nodes) / (connected)
end

to-report avg-path-length
  report average-path-length
end

to-report unreachable-path
  report unconnected
end

;; calculate global clustering coefficient
to-report clustering-coefficient
  let closed-triplets sum [ nw:clustering-coefficient * count my-links * (count my-links - 1) ] of nodes
  let triplets sum [ count my-links * (count my-links - 1) ] of nodes
  report closed-triplets / triplets
end

;; local search implementation of small-world networks
to local-search
  if small-world? [
    ask links [die]
    ask nodes [
      ;; if NUMBER of links are below 4, use n-of for localset since localset includes 4 neighbors
      ifelse number-of-links <= 4 [
        create-links-to n-of number-of-links localset
        if (random-float 1) < rewiring-probability [
          let rewire one-of other nodes with [ (self != myself) and (localset != myself) ]
          ask one-of links [ die ]
          create-link-to rewire
        ]
      ]
      [create-links-to localset
        create-links-to n-of (number-of-links - 4) other nodes with [ (self != myself) and (localset != myself) ]
      ]
    ]
  ]
end

;;;;;;;;;;;;;;;
;; Cognition ;;
;;;;;;;;;;;;;;;

;; local search strategy
to ls-strategy
  ;; this procedure gives local knowledge of who is the target to nodes
  ;; but nodes don't hold any knowledge of long nodes which are not in their local set
  ;; If nodes hold long nodes and they don't find the target in their localset, then they send a message to a long node
  ask nodes [
    let l1 sort-on [who] [localset] of current-location
    let l2 sort-on [who] number-of-neighbors
    let long-node filter [? != l1] l2
    ifelse member? [target] of origin number-of-neighbors [
    ifelse member? [target] of origin [localset] of current-location [
      set next-location [target] of origin][
        ;; since some nodes don't rewire, make sure whether nodes have links with long-distant nodes
        ifelse empty? long-node[
          set next-location one-of number-of-neighbors][
          set next-location item random length long-node long-node
      ]
    ]
    ][ifelse empty? long-node[
          set next-location one-of number-of-neighbors][
          set next-location item random length long-node long-node
          ]
    ]
  ]
end

;; homophily search
to homophily
  ;; A node holding a message prefers to send the same gender neighbor as target's gender
  ;; if they don't have any neighbor's gender same as target's, then they randomly choose next destination in their link neighbors
  ask nodes [
    ifelse any? number-of-neighbors with [gender = [gender] of target-node] [
      ;; a 70 percent chance to choose a contact whose gender is the same as target
      ;; Otherwise, follow the random choice procedure
      ;; this random implementation is not parcilarly something I'm not proud of...
      ;; 70 is an abtraral number, but it works both with & without route limits
      ifelse random 99 < 70 [set next-location one-of number-of-neighbors with [gender = [gender] of target-node]][
        set next-location one-of number-of-neighbors]
    ][set next-location one-of number-of-neighbors
    ]
  ]
end

;; preferential attachment search
to preferential
  ;; nodes look at maximum indegree neighbor as their next message destination
  ask nodes [
    set next-location max-one-of number-of-neighbors [count [in-link-neighbors] of self ]
  ]
end
@#$#@#$#@
GRAPHICS-WINDOW
289
10
728
470
16
16
13.0
1
10
1
1
1
0
0
0
1
-16
16
-16
16
0
0
1
ticks
30.0

BUTTON
12
9
76
42
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
102
9
182
42
go-once
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
202
9
265
42
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

MONITOR
12
113
131
162
NIL
unreachable-pairs
17
1
12

MONITOR
12
165
125
214
NIL
number-of-steps
17
1
12

PLOT
733
10
1002
160
# of steps per round
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"actual" 1.0 0 -2674135 true "" "plotxy sessions steps"
"conceptual" 1.0 0 -7500403 true "" "plotxy sessions min-pl"

MONITOR
12
216
77
265
NIL
sessions
17
1
12

SLIDER
12
366
184
399
number-of-nodes
number-of-nodes
1
100
20
1
1
NIL
HORIZONTAL

SLIDER
12
400
184
433
number-of-links
number-of-links
1
10
3
1
1
NIL
HORIZONTAL

SWITCH
12
44
144
77
route-limits?
route-limits?
0
1
-1000

MONITOR
134
113
221
162
NIL
incompletes
17
1
12

MONITOR
134
165
264
214
NIL
number-of-deadend
17
1
12

PLOT
733
161
1002
311
Path length
NIL
NIL
1.0
20.0
0.0
10.0
true
true
"" ""
PENS
"observed" 1.0 1 -2674135 true "" "histogram steps-record"
"expected" 1.0 1 -7500403 true "histogram pl-list" ""

MONITOR
12
267
131
316
expected avg pl
average-path-length
5
1
12

MONITOR
134
267
274
316
number-of-unconnected
unconnected
17
1
12

MONITOR
12
316
122
365
observedl avg pl
calculate-average
5
1
12

SWITCH
145
44
281
77
small-world?
small-world?
1
1
-1000

SLIDER
12
78
184
111
rewiring-probability
rewiring-probability
0
1
0.3
0.01
1
NIL
HORIZONTAL

MONITOR
134
216
243
265
number-of-links
count links
17
1
12

MONITOR
134
316
274
365
NIL
clustering-coefficient
5
1
12

CHOOSER
733
375
921
420
search-strategy
search-strategy
"random-search" "local-search" "homophilious-search" "preferential-attachment"
3

@#$#@#$#@
## WHAT IS IT?

This model explores **how network structure and agent choice of routing influence the observed average steps between a message origin and target**. In terms of network structure, the model produces two types of networks:

* a random network (default), and

* a small-world network.

Speaking of agent rounting choice, the model has four different choice heuristics (See details on HOW TO USE IT):

* random (default)

* local search

* homophilious search and

* preferential attachment search.

The "actual" average path length (observed) means how many steps take until messages are reached, while the average path length (expected) can be conceptually calculated as the average possible shortest path between pairs of agents. However, research shows the actual paths are longer than computed expected shortest paths in real human practice due to the fact that people choose not shortest path once in two choices (Killworth et al., 2006). This model shows this phenomenon and the mechanisms of why this is the case.

## HOW IT WORKS

In the model, two agents are set up: (1) nodes who circulate a message and (2) a message who is circulated by nodes. Also, nodes have links, which create paths to other nodes.

When you press the SETUP button (See also the Choose Network Model):

* Create NUMBER nodes and NUMBER of directed links for each nodes - either randomly or in a small-world way (the SMALL-WORLD switch)
* Default is a circle layout
* One message is randomly set up on a node
* One target is randomly set up on another node
* Calculate the number of paths unavailable between pairs of nodes as NUMBER-OF-UNCONNECTED

Each tick (GO or GO ONCE):

* The message moves to a new location via a link
* A link becomes thicker as a mark of usage of the route
* Repeat the aforementioned processes until the message reaches or becoming one of the two conditions:
  (1) INCOMPLETES: When the ROUTE-LIMITS switch is on, nodes often exhaust their paths as a result that the message comes back them again and again. When nodes have no usable paths, STOP.
  (2) DEADEND: When not all nodes have avaiable access to every other node (i.e., NUMBER-OF-UNCONNECTED is not 0), the message can suddenly go a DEADEND path and can't get out to reach to the target. If so, it adds NUMBER-OF-DEADEND and STOP.

Once the message is reached:

* It counts as one session and adds NUMBER OF SESSIONS
* Re-setup a message and target at a random node, respecively
* If there is no path available between a randomly setup message and target, the case is recorded as UNREACHABLE-PAIRS

## HOW TO USE IT

This models provides a experimental setting for a small-world experiment. Changing the switches and sliders helps you understand how observed path lengths are emerged.

* ROUTE-LIMITS: This gives constrains on use of links (paths). If the switch is on, agents can't use the same path twice.

* SMALL-WORLD: This switch changes the network setting from a random to a small-world network (shorter path length and high clustering coefficient). Using REWIRING PROBABILITY, change the likelihood of rewiring links to create a small-world network.

* NUMBER-OF-NODES: This gives you a control for creating how many nodes are in a network.

* NUMBER-OF-LINKS: This determines how many links each node can hold.

* SEARCH-STRATEGY: This chooser allows you to test how different cognitive strategies work in message delivery:
   (1) **random search**: All links agents hold have a equal chance to be chose,
   (2) **local search**: First, agents look at their local neighbors. If no neighbors are the target, they pass a message to furthest link-neighbor or one of local neighbors if they don't have further link-neighbor,
   (3) **homophilious search**: Based on target's gender, agents determine which contact they will send. If the target is female, agents are more likely to choose female contact than male one, and
   (4) **preferential attachment**: Agents prefer to send a message to the highest in-link neighbors)

## THINGS TO NOTICE

You see a similarity and difference in terms of observed and expected average path length and the freqency distribution of path lengths. In particular, look at an observed line or bar (red) and an expected line or bar (gray) in the plots. How do their shapes differ?

Also, if you turn ROUTE-LIMITS on, what is the NUMBER-OF-INCOMPLETES? What is the successful chain rate? What is the incomplete rate? You can calculate the imcomplete rate as follows: (NUMBER-OF-INCOMPLETES + NUMBER-OF-DEADEND) / (SESSIONS - UNREACHABLE PAIRS).

## THINGS TO TRY

First, you should try to switch between a random network and small-world network, and see how observed path lengths change.

Second, use the SEARCH-STRATEGY chooser. The different search strategies return different results of observed path lengths.

Third, using the ROUTE-LIMITS switch, see how a message is circulated within a network. Particularly, no route limit condition is barely able to be observed in the reality. Compared to no route limit condition, see how much the route limit condition chages observed path lengths.

## EXTENDING THE MODEL

Try to change the number of message propagation. The current model is set one message at a time. However, circulating multiple messages is a reasonable scenario in the reality. For doing so, you may require to change some other procedures.

It is also possible to implement a memory for each agents, for instance, using the LevelSpace extension. This extension may enable to test how agents become better or worse at routing over time.

## NETLOGO FEATURES

This model uses the [NW extension](https://github.com/NetLogo/NW-Extension) to calculate path length, clustering coefficient and in-degree centrality for each node. However, for the average path length, it doesn't use the NW extension, but it returns the same result as a command of nw:mean-path-length in the NW extension.

## RELATED MODELS

This model is based on Stanley Milgram's small-world experiment:

* Milgram, S. (1967). The Small World Problem,  Psychology Today,  2: 60-67.

* Travers, J., & Milgram, S. (1969). An experimental study of the small world problem. Sociometry, 425-443.

In terms of the expected emergence patterns for this model:

* Killworth, P. D., McCarty, C., Bernard, H. R., & House, M. (2006). The accuracy of small world chains in social networks. Social networks, 28(1), 85-96.

Also, this model extensively refers to in Models Library:

* Small Worlds: Wilensky, U. (2005).  NetLogo Small Worlds model.  http://ccl.northwestern.edu/netlogo/models/SmallWorlds.  Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

* Link-Walking Turtle Example

## CREDITS AND REFERENCES

This model is developed by Kyosuke Tanaka at Northwestern University. If you have any questions or want to use this model not in your own use, please email at kyosuke [at] u.northwestern.edu.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 6.0-M6
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="random network" repetitions="50" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="5000"/>
    <metric>sessions</metric>
    <metric>unreachable-pairs</metric>
    <metric>incompletes</metric>
    <metric>number-of-deadend</metric>
    <metric>unconnected</metric>
    <metric>average-path-length</metric>
    <metric>clustering-coefficient</metric>
    <metric>calculate-average</metric>
    <enumeratedValueSet variable="number-of-nodes">
      <value value="20"/>
      <value value="40"/>
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-links">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="local-search-strategy?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="homophily?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="preferential-attachment?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0.3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="local-search?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="directed?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="route-limits?">
      <value value="true"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="small-world network" repetitions="50" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="5000"/>
    <metric>sessions</metric>
    <metric>unreachable-pairs</metric>
    <metric>incompletes</metric>
    <metric>number-of-deadend</metric>
    <metric>unconnected</metric>
    <metric>average-path-length</metric>
    <metric>clustering-coefficient</metric>
    <metric>calculate-average</metric>
    <enumeratedValueSet variable="number-of-nodes">
      <value value="20"/>
      <value value="40"/>
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-links">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="local-search-strategy?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="homophily?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="preferential-attachment?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0.3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="local-search?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="directed?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="route-limits?">
      <value value="true"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="random network [pereferential]" repetitions="50" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="5000"/>
    <metric>sessions</metric>
    <metric>unreachable-pairs</metric>
    <metric>incompletes</metric>
    <metric>number-of-deadend</metric>
    <metric>unconnected</metric>
    <metric>average-path-length</metric>
    <metric>clustering-coefficient</metric>
    <metric>calculate-average</metric>
    <enumeratedValueSet variable="number-of-nodes">
      <value value="20"/>
      <value value="40"/>
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-links">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="local-search-strategy?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="homophily?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="preferential-attachment?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0.3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="local-search?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="directed?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="route-limits?">
      <value value="true"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="small-world network [pereferential]" repetitions="50" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="5000"/>
    <metric>sessions</metric>
    <metric>unreachable-pairs</metric>
    <metric>incompletes</metric>
    <metric>number-of-deadend</metric>
    <metric>unconnected</metric>
    <metric>average-path-length</metric>
    <metric>clustering-coefficient</metric>
    <metric>calculate-average</metric>
    <enumeratedValueSet variable="number-of-nodes">
      <value value="20"/>
      <value value="40"/>
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-links">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="local-search-strategy?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="homophily?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="preferential-attachment?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0.3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="local-search?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="directed?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="route-limits?">
      <value value="true"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="random network [homophilious search]" repetitions="50" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="5000"/>
    <metric>sessions</metric>
    <metric>unreachable-pairs</metric>
    <metric>incompletes</metric>
    <metric>number-of-deadend</metric>
    <metric>unconnected</metric>
    <metric>average-path-length</metric>
    <metric>clustering-coefficient</metric>
    <metric>calculate-average</metric>
    <enumeratedValueSet variable="number-of-nodes">
      <value value="20"/>
      <value value="40"/>
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-links">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="local-search-strategy?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="homophily?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="preferential-attachment?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0.3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="local-search?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="directed?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="route-limits?">
      <value value="true"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="random network [local search]" repetitions="50" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="5000"/>
    <metric>sessions</metric>
    <metric>unreachable-pairs</metric>
    <metric>incompletes</metric>
    <metric>number-of-deadend</metric>
    <metric>unconnected</metric>
    <metric>average-path-length</metric>
    <metric>clustering-coefficient</metric>
    <metric>calculate-average</metric>
    <enumeratedValueSet variable="number-of-nodes">
      <value value="20"/>
      <value value="40"/>
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-links">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="local-search-strategy?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="homophily?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="preferential-attachment?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0.3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="local-search?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="directed?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="route-limits?">
      <value value="true"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="small-world network [local search]" repetitions="50" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="5000"/>
    <metric>sessions</metric>
    <metric>unreachable-pairs</metric>
    <metric>incompletes</metric>
    <metric>number-of-deadend</metric>
    <metric>unconnected</metric>
    <metric>average-path-length</metric>
    <metric>clustering-coefficient</metric>
    <metric>calculate-average</metric>
    <enumeratedValueSet variable="number-of-nodes">
      <value value="20"/>
      <value value="40"/>
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-links">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="local-search-strategy?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="homophily?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="preferential-attachment?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0.3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="local-search?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="directed?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="route-limits?">
      <value value="true"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="small-world network [homophilious search]" repetitions="50" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="5000"/>
    <metric>sessions</metric>
    <metric>unreachable-pairs</metric>
    <metric>incompletes</metric>
    <metric>number-of-deadend</metric>
    <metric>unconnected</metric>
    <metric>average-path-length</metric>
    <metric>clustering-coefficient</metric>
    <metric>calculate-average</metric>
    <enumeratedValueSet variable="number-of-nodes">
      <value value="20"/>
      <value value="40"/>
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-links">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="local-search-strategy?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="homophily?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="preferential-attachment?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0.3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="local-search?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="directed?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="route-limits?">
      <value value="true"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="random [random]" repetitions="100" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="5000"/>
    <metric>sessions</metric>
    <metric>unreachable-pairs</metric>
    <metric>incompletes</metric>
    <metric>number-of-deadend</metric>
    <metric>unconnected</metric>
    <metric>average-path-length</metric>
    <metric>clustering-coefficient</metric>
    <metric>calculate-average</metric>
    <enumeratedValueSet variable="number-of-nodes">
      <value value="20"/>
      <value value="40"/>
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-links">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0.3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="small-world?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="route-limits?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="search-strategy">
      <value value="&quot;random-search&quot;"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="random [local]" repetitions="100" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="5000"/>
    <metric>sessions</metric>
    <metric>unreachable-pairs</metric>
    <metric>incompletes</metric>
    <metric>number-of-deadend</metric>
    <metric>unconnected</metric>
    <metric>average-path-length</metric>
    <metric>clustering-coefficient</metric>
    <metric>calculate-average</metric>
    <enumeratedValueSet variable="number-of-nodes">
      <value value="20"/>
      <value value="40"/>
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-links">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0.3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="small-world?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="route-limits?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="search-strategy">
      <value value="&quot;local-search&quot;"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="random [homophily]" repetitions="100" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="5000"/>
    <metric>sessions</metric>
    <metric>unreachable-pairs</metric>
    <metric>incompletes</metric>
    <metric>number-of-deadend</metric>
    <metric>unconnected</metric>
    <metric>average-path-length</metric>
    <metric>clustering-coefficient</metric>
    <metric>calculate-average</metric>
    <enumeratedValueSet variable="number-of-nodes">
      <value value="20"/>
      <value value="40"/>
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-links">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0.3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="small-world?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="route-limits?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="search-strategy">
      <value value="&quot;homophilious-search&quot;"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="random [preferential]" repetitions="100" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="5000"/>
    <metric>sessions</metric>
    <metric>unreachable-pairs</metric>
    <metric>incompletes</metric>
    <metric>number-of-deadend</metric>
    <metric>unconnected</metric>
    <metric>average-path-length</metric>
    <metric>clustering-coefficient</metric>
    <metric>calculate-average</metric>
    <enumeratedValueSet variable="number-of-nodes">
      <value value="20"/>
      <value value="40"/>
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-links">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0.3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="small-world?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="route-limits?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="search-strategy">
      <value value="&quot;preferential-attachment&quot;"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="small [random]" repetitions="100" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="5000"/>
    <metric>sessions</metric>
    <metric>unreachable-pairs</metric>
    <metric>incompletes</metric>
    <metric>number-of-deadend</metric>
    <metric>unconnected</metric>
    <metric>average-path-length</metric>
    <metric>clustering-coefficient</metric>
    <metric>calculate-average</metric>
    <enumeratedValueSet variable="number-of-nodes">
      <value value="20"/>
      <value value="40"/>
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-links">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0.3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="small-world?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="route-limits?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="search-strategy">
      <value value="&quot;random-search&quot;"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="small [local]" repetitions="100" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="5000"/>
    <metric>sessions</metric>
    <metric>unreachable-pairs</metric>
    <metric>incompletes</metric>
    <metric>number-of-deadend</metric>
    <metric>unconnected</metric>
    <metric>average-path-length</metric>
    <metric>clustering-coefficient</metric>
    <metric>calculate-average</metric>
    <enumeratedValueSet variable="number-of-nodes">
      <value value="20"/>
      <value value="40"/>
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-links">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0.3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="small-world?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="route-limits?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="search-strategy">
      <value value="&quot;local-search&quot;"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="small [homophily]" repetitions="100" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="5000"/>
    <metric>sessions</metric>
    <metric>unreachable-pairs</metric>
    <metric>incompletes</metric>
    <metric>number-of-deadend</metric>
    <metric>unconnected</metric>
    <metric>average-path-length</metric>
    <metric>clustering-coefficient</metric>
    <metric>calculate-average</metric>
    <enumeratedValueSet variable="number-of-nodes">
      <value value="20"/>
      <value value="40"/>
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-links">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0.3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="small-world?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="route-limits?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="search-strategy">
      <value value="&quot;homophilious-search&quot;"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="small [preferential]" repetitions="100" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="5000"/>
    <metric>sessions</metric>
    <metric>unreachable-pairs</metric>
    <metric>incompletes</metric>
    <metric>number-of-deadend</metric>
    <metric>unconnected</metric>
    <metric>average-path-length</metric>
    <metric>clustering-coefficient</metric>
    <metric>calculate-average</metric>
    <enumeratedValueSet variable="number-of-nodes">
      <value value="20"/>
      <value value="40"/>
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-links">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rewiring-probability">
      <value value="0.3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="small-world?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="route-limits?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="search-strategy">
      <value value="&quot;preferential-attachment&quot;"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
