globals[
  nodes             ;;list of all the patches that act as nodes in the network (# of nodes = 2^NETWORK-SIZE)
  #-of-rows         ;;# of rows in network
  #-of-columns      ;;# of columns in network
  error-set         ;;if error during setup, will not proceed with simulation

  top-edges         ;;if torus network, identifies the North boundary nodes that will cross to the South boundary nodes
  right-edges       ;;if torus network, identifies the East boundary nodes that will cross to the West boundary nodes
  left-edges        ;;if torus network, identifies the West boundary nodes that will cross to the East boundary nodes
  bottom-edges      ;;if torus network, identifies the South boundary nodes that will cross to the North boundary nodes

  current-workload  ;;sum of read + write sent to each directory
  read-workload     ;;reads sent to each directory
  write-workload    ;;writes sent to each directory

  assigned-nodes    ;;tracks nodes who have already been assigned a directory (used during directory placement)
  assignable-nodes  ;;tracks nodes who have not been assigned a directory (used during directory placement)

  network-set?      ;;flag indicating network set
  directories-set?  ;;flag indicating directories are set

  total-times-write-prioritized  ;;counts the number of times conflict in assigning directories resulted in prioritizing the node with the higher amount of writes to this directory
  total-times-read-prioritized   ;;counts the number of times conflict in assigning directories resulted in prioritizing the node with the higher amount of reads to this directory

  total-hops              ;;counts the total hops every message created has taken once it reaches its home node (destination)
  total-messages          ;;counts the total messages sent and received during the simulation
  total-workload-messages ;;counts the total messages there are for this workload
]

breed [messages message]  ;;messages

patches-own[
  id                          ;;node id number
  directory-id                ;;directory id number
  message-in-transit?         ;;true if this node already has a message traveling the network
  node-workload               ;;workload for this node; indexed by directory number, #of messages sent to that directory
  node-workload-probabilities ;;upper workload probability for each directory for this node; indexed by directory number, used to create messages fairly in a proportioned manner

]

messages-own[
  local-node          ;;node id created at
  home-directory      ;;directory id traveling to
  hops                ;;# of hops taken through network from local-node to home-directory
]


;;setup procedure
;;create the network based on the configuration settings selected
to setup-network
  clear-all
  set error-set false
  set-network            ;; sets if mesh or torus
  set-nodes              ;; creates network of 2^network-size nodes
  set-workload           ;; based on network size and testbench selected: sets up workloads
  set-workload-per-node  ;; creates workloads for each node
  set-globals            ;; global variable set up

  set network-set? true  ;; mark network has been set
  reset-ticks

end

;;setup procedure
;;assign directories based on placement plan selected, label the nodes with their directory id
to setup-directories

    if-else network-set? != true
    [user-message "The network has not been set, please setup the network before continuing." stop]
    [if not error-set
      [place-directories]  ;; place directories based on algorithm selected

    ask nodes [set plabel directory-id]  ;;show each nodes directory id assigned in the model

    set directories-set? true  ;;mark directories have been set

    set assigned-nodes [] ;;reset globals
    set assignable-nodes n-values (count nodes) [?] ;;reset globals

    reset-ticks]

end


;;go procedure
;;each node will only have one message in the system at one time
;;each message will travel from its local node to the home node
;;(the node with the matching directory id)
to go

  if network-set? != true [user-message "The network has not been set, please setup the network before continuing." stop]
  if not directories-set? [user-message "The network setup has changed, please setup the diretories before continuing." stop]
  if error-set [user-message "Invalid model settings." stop]
  if (total-messages >= total-workload-messages) or (ticks >= 2000000) [stop]

  ask messages [decide-direction]
  send-one-message-at-time

  tick
end


;;node procedure
;;if the node doesn't already have a message in the network, send one new message.
;;the message will select a home node directory based on the workload at this node.
;;this allows the messages to be fairly proportioned like the workload but still generated randomly at each tick.
to send-one-message-at-time


  let node-sending []
  let send-to-dir n-values  (count nodes) [-100]
  let max-work 0
  ask nodes with [message-in-transit? = false]
    [

      set node-sending (lput [id] of self node-sending)
      let send-me random-float 100
      let send-to-this-directory 0
      let bigger-than-previous true
      foreach node-workload-probabilities
      [
         if bigger-than-previous and send-me <= ?
         [set send-to-this-directory (position ? node-workload-probabilities)
          ]

         if send-me < ? [set bigger-than-previous false]
      ]

      set send-to-dir (replace-item ([id] of self) send-to-dir send-to-this-directory)
    ]

  let x 0
  let y 0
  foreach node-sending
      [
        ask nodes with [id = ?]
          [set x pxcor
            set y pycor
            set message-in-transit? true
          ]


        create-messages 1
        [setxy x y
          set size 1.5
          set hops 0
          set local-node ?
          set home-directory (item ? send-to-dir)
          if-else (? = watch-messages-from-this-node) ;;for visualization purposes only
             [set color yellow
               watch-me
               clear-output
               output-print home-directory]
             [set color blue]

        ]
      ]

end

;;node procedure
;;set up each node's unique workload and the probabilies of sending a message
;;to each directory based on that workload
to set-workload-per-node

;;set workload
foreach current-workload
  [ ask nodes
    [set node-workload (lput (item id (but-first ?)) node-workload)] ]

;;set probability upper bounds for each directory for given node
ask nodes
  [ let cumulative 0
    let total sum(node-workload)
    foreach node-workload
    [ set cumulative (cumulative + ?)
      let upper-probability (100 * (cumulative / total))
      set node-workload-probabilities (lput upper-probability node-workload-probabilities)
    ]
  ]


end


;; turtle procedure
;; messages decides which node to go to next (N-S-E-W or stay if current = home)
to decide-direction

  let node-id 0
  ask patch-here [set node-id id]

  let destination home-directory
  ask nodes with [directory-id = destination] [set destination id]

  let current-row floor(node-id / #-of-columns)
  let desired-row floor(destination / #-of-columns)
  let current-col (node-id mod #-of-columns)
  let desired-col (destination mod #-of-columns)


  if-else node-id = destination ;; ARRIVED
  [let done-sending local-node
    ask nodes with [id = done-sending] [set message-in-transit? false] ;; allow the node to send a new message the next clock cycle
    set total-hops (total-hops + hops)  ;;update global hop count with this message's total hops
    set total-messages (total-messages + 1) ;;increase number of messages successfully sent
    die ;;remove message from simulation
  ]
  [
    if-else  current-row = desired-row ;;Prioritize N-S direction first, followed by E-W
    [
      let wrap-it? abs(current-col - desired-col) > (#-of-columns / 2)
      if-else (not torus? and current-col > desired-col) or (torus? and (wrap-it? and current-col < desired-col) or (not wrap-it? and current-col >= desired-col))
        [go-west set hops (hops + 1)]
        [go-east set hops (hops + 1)]
    ]

    [
      let wrap-it? abs(current-row - desired-row) > (#-of-rows / 2)
      if-else (not torus? and current-row > desired-row) or (torus? and (wrap-it? and current-row < desired-row) or (not wrap-it? and current-row >= desired-row))
        [go-north set hops (hops + 1)]
        [go-south set hops (hops + 1)]
    ]


  ]


end

;;message procedure
;;travel north through the network; if torus wrap around to the bottom boundary nodes
to go-north

  let my-id id
  let go-x 0
  let go-y 0
  if-else id <= #-of-columns
    [if-else torus?
      [ask nodes with[ id = (#-of-rows * #-of-columns) - (#-of-columns - my-id) ] [set go-x pxcor set go-y pycor]]
      [ask patch-here [set go-x pxcor set go-y pycor]]]
    [ask nodes with[ id = (my-id - #-of-columns)] [set go-x pxcor set go-y pycor]]
  face patch go-x go-y
  move-to patch go-x go-y

end

;;message procedure
;;travel south through the network; if torus wrap around to the top boundary nodes
to go-south

  let my-id id
  let go-x 0
  let go-y 0
  if-else id >= ((#-of-rows * #-of-columns) - #-of-columns)
    [if-else torus?
      [ask nodes with[ id = ( #-of-columns - ((#-of-rows * #-of-columns) - my-id  ))] [set go-x pxcor set go-y pycor]]
      [ask patch-here [set go-x pxcor set go-y pycor]]]
    [ask nodes with[ id = (my-id + #-of-columns)] [set go-x pxcor set go-y pycor]]
  face patch go-x go-y
  move-to patch go-x go-y

end

;;message procedure
;;travel east through the network; if torus wrap around to the left boundary nodes
to go-east

  let my-id id
  let go-x 0
  let go-y 0
  if-else id mod #-of-columns =  (#-of-columns - 1)
    [if-else torus?
      [ask nodes with[ id = (my-id - #-of-columns + 1)] [set go-x pxcor set go-y pycor]]
      [ask patch-here [set go-x pxcor set go-y pycor]]]
    [ask nodes with[ id = (my-id + 1)] [set go-x pxcor set go-y pycor]]
  face patch go-x go-y
  move-to patch go-x go-y

end

;;message procedure
;;travel west through the network; if torus wrap around to the right boundary nodes
to go-west

  let my-id id
  let go-x 0
  let go-y 0
  if-else id mod #-of-columns = 0
    [if-else torus?
      [ask nodes with[ id = (my-id + #-of-columns - 1)] [set go-x pxcor set go-y pycor]]
      [ask patch-here [set go-x pxcor set go-y pycor]]]
    [ask nodes with[ id = (my-id - 1)] [set go-x pxcor set go-y pycor]]
  face patch go-x go-y
  move-to patch go-x go-y

end

;;setup procedure
;; sets nework to either be a mesh or torus
to set-network
  if-else torus?
    [__change-topology true true]
    [__change-topology false false]
end


;;setup procedure
;;creates nodes as patches evenly spaced out in the screen
;;number of nodes created = 2 ^ NETWORK-SIZE
;;NOTE: this model screen size supports networks up to NETWORK-SIZE = 8 (256 nodes)
to set-nodes
  ask patches [set id -1] ;;to ensure won't treat patches that are nodes like nodes

  let grid-x-inc 0
  let grid-y-inc 0
  if-else network-size mod 2 = 0
  [
    set grid-x-inc world-width / (sqrt (2 ^ network-size ))
    set grid-y-inc world-height / (sqrt (2 ^ network-size ))
  ]

  [
    set grid-x-inc world-width / (sqrt (2 ^ (network-size + 1)))
    set grid-y-inc world-height / (sqrt (2 ^ (network-size - 1)))
  ]

  set nodes patches with
    [(floor((pxcor + max-pxcor - floor(grid-x-inc - 1)) mod grid-x-inc) = 0) and
      (floor((pycor + max-pycor) mod grid-y-inc) = 0)]

  let n 0
  foreach sort nodes [
    ask ? [
      set id n
      set plabel id
      set n n + 1
      set pcolor green
      set message-in-transit? false
      set node-workload []
      set node-workload-probabilities []
    ]
  ]

end

;;setup procedure
;;resets all global values
to set-globals
  let row 0
  let col 0

  ask nodes with [id = 1]
  [set row pxcor
    set col pycor]

  set #-of-rows count nodes with [pxcor = row]
  set #-of-columns count nodes with [pycor = col]

  set assigned-nodes []
  set assignable-nodes n-values (count nodes) [?]
  set total-hops 0
  set total-messages 0

  ask nodes [set total-workload-messages (total-workload-messages + (sum node-workload))]
  set total-times-read-prioritized 0
  set total-times-write-prioritized 0
  set directories-set? false
end

;;setup procedure
;;combines read + write workloads to make a general workload
to make-current-workload

  set current-workload []
  foreach read-workload
    [set current-workload (lput (list (first ?)) current-workload)]

  ;; CREATE READ + WRITE WORKLOAD INFO
  let temp current-workload
  let i 0
  foreach read-workload
    [ let n 1
      let add-me 0
      let add-all-me []
      foreach read-workload
      [

        let read-value item n (item i read-workload)
        let write-value item n (item i write-workload)
        set add-me (read-value + write-value)
        set add-all-me (lput add-me add-all-me)
        set n n + 1
      ]

      let directory-num item i current-workload
      let directory-entry (sentence  directory-num add-all-me)
      set current-workload (replace-item i current-workload  directory-entry)
      set i i + 1
    ]
end

;; node procedure
;; optimize the directory placement based on the workload given by choosing
;; to assign the most used directories to the node that uses it that often
;; and conflicts arise if workloads for that directory are the same between nodes
;; and thus resolved with the type priority selected.
to optimal-max [this-workload type-priority]
ask nodes
  [
    let index-list []
    let max-list []
    let allowable-node-loads []
    let n 0

    foreach this-workload [

      set index-list lput first (item n this-workload) index-list

      let original-node-loads (but-first (item n this-workload))
      set allowable-node-loads original-node-loads
      let temp-node original-node-loads

      foreach original-node-loads

      [if member? (position ? temp-node) assigned-nodes
        [set temp-node (replace-item (position ? temp-node) temp-node -4)

          let c1 length allowable-node-loads
          set allowable-node-loads (remove-item (position ? allowable-node-loads)  allowable-node-loads)
          let c2 length allowable-node-loads
          if c1 - c2 > 1[print "ERRR 1"]]
      ]

      set max-list lput max allowable-node-loads max-list
      let node-max-list []
      set node-max-list lput (position (max allowable-node-loads) max-list) node-max-list
      set n n + 1

    ]


    let list-pos -1
    let node-selected -1
    let priority-directory []
    let priority-directory-workload []
    let priority max max-list
    let doubles-list filter [? = priority] max-list

    if-else (length doubles-list > 1)

      [ ;; PRIORITY CONFLICT, RESOLVE BASED ON LARGER WRITE or READ WORKLOAD
        let temp-max-list max-list
        let node-conflict []
        let dir-decide []
        let p-workload []
        if-else type-priority = "W"
        [set total-times-write-prioritized total-times-write-prioritized + 1 ;;WRITE PRIORITIZE
          set p-workload write-workload]
        [set total-times-read-prioritized total-times-read-prioritized + 1 ;;READ PRIORITIZE
          set p-workload read-workload]

        let index-list2 []
        let max-list2 []
        let allowable-node-loads2 []
        let n2 0
        let temp-node2 []

        foreach p-workload
        [

          set index-list2 lput first (item n2 p-workload) index-list2
          let original-node-loads2 (but-first (item n2 p-workload))
          set allowable-node-loads2 original-node-loads2
          set temp-node2 original-node-loads2


          foreach original-node-loads2
            [
              if member? (position ? temp-node2) assigned-nodes
                [set temp-node2 (replace-item (position ? temp-node2) temp-node2 -4)
                  set allowable-node-loads2 (remove-item  (position ? allowable-node-loads2)  allowable-node-loads2)]
            ]

           set max-list2 lput max allowable-node-loads2 max-list2
           set n2 n2 + 1
        ]

        let priority2 max max-list2
        let spot (position priority2 max-list2)

          ;;recreate preference list for the node with the highest priority
          set index-list2 lput first (item spot p-workload) index-list2
          let original-node-loads2 (but-first (item spot p-workload))
          set allowable-node-loads2 original-node-loads2
          set temp-node2 original-node-loads2

          foreach original-node-loads2
          [
             if member? (position ? temp-node2) assigned-nodes
                [set temp-node2 (replace-item (position ? temp-node2) temp-node2 -4)
                 set allowable-node-loads2 (remove-item  (position ? allowable-node-loads2)  allowable-node-loads2)]
          ]

          set list-pos (position priority2 temp-node2)
          set node-selected list-pos
          set priority-directory item spot index-list
    ]

    ;;No conflict, so no prioritizing needed
    [set list-pos (position priority max-list)
    set priority-directory item list-pos index-list
    set priority-directory-workload item list-pos this-workload


    set priority-directory-workload item list-pos this-workload
    set node-selected position priority (but-first priority-directory-workload)
    ]


   ;; remove this directory from the workload so it won't be assigned to again
    let this-remove -1
    let write-remove -1
    let read-remove -1
    let counter -1
    foreach this-workload
    [set counter counter + 1
     if first ? = priority-directory
        [set this-remove counter]]

    set counter -1
    foreach write-workload
    [set counter counter + 1
     if first ? = priority-directory
        [set write-remove counter]]

    set counter -1
    foreach read-workload
    [set counter counter + 1
     if first ? = priority-directory
        [set read-remove counter]]

    set this-workload  (remove-item this-remove this-workload)
    set write-workload (remove-item write-remove write-workload)
    set read-workload  (remove-item read-remove read-workload)

   ;;assign the winning node their most used directory
    ask nodes with [id = node-selected]
    [set directory-id priority-directory]


   ;;update global tracking of assigned/still available nodes
    set assigned-nodes (lput node-selected assigned-nodes)
    set assignable-nodes (remove node-selected assignable-nodes)


  ]


end


;;setup procedure
;;based on algorithm selected, assign directories to nodes
to place-directories

  let r-load read-workload
  let w-load write-workload
  let c-load current-workload

  if placement = "Baseline"
    [ask nodes [set directory-id id]]

  if placement = "Most Used - W Priority"
  [ask nodes [set directory-id -1]
    optimal-max current-workload "W"
    ask nodes [if directory-id = -1 [print "ERROR directory assignment"]]]

  if placement = "Most Used - R Priority"
  [ask nodes [set directory-id -1]
    optimal-max current-workload "R"
    ask nodes [if directory-id = -1 [print "ERROR directory assignment"]]]

  if placement = "Most Used - W Only"
  [ask nodes [set directory-id -1]
    optimal-max write-workload "W"
    ask nodes [if directory-id = -1 [print "ERROR directory assignment"]]]

  if placement = "Most Used - R Only"
  [ask nodes [set directory-id -1]
    optimal-max read-workload "R"
    ask nodes [if directory-id = -1 [print "ERROR directory assignment"]]]

  ;;restore workloads
  set read-workload r-load
  set write-workload w-load
  set current-workload c-load

end


;; setup procedure
;; assign workload to the model depending on user selection
to set-workload

  if workload = "FFT" [set-FFT]
  if workload = "Radix" [set-Radix]
  if workload = "Ocean" [set-Ocean]

  if-else empty? read-workload
    [user-message "Error: Workload and network size combination currently unavailable. Please choose a different combination."
      set error-set true]
    [make-current-workload]

end

;; setup FFT workload from the SPLASH2 testbench for given network size
;; these values were obtained through analysis of the testbenches using PinTool
to set-FFT

  if network-size = 2
  [
    set read-workload
    [[0  3123366  3724399  12333  294459]
      [1  1240941  876085  488310  732227]
      [2  2641067  20323  1783150  2272619]
      [3  2581761  377807  2848042  1798971]]

    set write-workload
    [[0  78017  143575  152325  101822]
      [1  105064  174922  129648  38247]
      [2  35436  151924  96745  141078]
      [3  122961  45588  104044  195921]]
  ]

  if  network-size = 3
  [
    set read-workload
    [[0  3418  57247  53387  67488  3399  99897  98794  65940]
      [1  52627  54298  53802  58398  62634  46031  76519  311]
      [2  53117  27475  5725  18254  14414  15660  67177  41791]
      [3  74232  35241  31963  19692  20855  57389  91551  42836]
      [4  15604  89230  45794  93291  26566  92124  11223  69120]
      [5  27565  23168  61268  12391  36496  11956  37015  39426]
      [6  54544  39729  61738  63491  75865  71160  59971  96649]
      [7  62598  18260  50201  33457  74722  12533  30541  39548]]

    set write-workload
    [[0  2124  51486  23865  13038  24269  50806  14806  24334]
      [1  19191  29878  49043  10916  17231  4347  14893  7935]
      [2  53018  57037  3231  15478  28837  29762  30210  5808]
      [3  46002  108  38305  23982  2400  54398  46040  33682]
      [4  20890  30700  51988  11606  11530  23970  24566  24485]
      [5  42770  35471  37773  49987  24089  12582  13697  28098]
      [6  25016  40774  38576  43329  59542  14595  12522  24983]
      [7  42964  33904  22050  34972  25335  11212  19412  31379]]
  ]

  if  network-size = 4
  [
    set read-workload
    [[0  27947  19963  22452  22991  4877  2506  20709  24321  9005  20482  18854  21970  15864  1496  8331  5709]
      [1  4018  22167  29095  10145  2588  17779  13776  13697  2788  9847  28176  140  23360  21652  7589  23905]
      [2  1245  4648  12553  17527  22464  478  8150  1050  4350  23137  894  18431  4543  14743  25481  25457]
      [3  20695  8266  7917  23257  14370  25472  27886  14414  26255  4279  27712  18452  25028  9284  4797  7969]
      [4  21852  6287  29393  6357  27683  14250  5646  15406  16076  18135  19761  12610  18703  29533  16227  7392]
      [5  3365  23672  24042  8104  16590  13202  19721  20839  22782  18521  24560  23488  6177  1508  4803  11798]
      [6  12117  7324  15562  5839  14395  12370  5076  10078  18017  13704  16309  28114  29251  19702  4158  12531]
      [7  13767  16194  17473  8712  560  12453  8157  5939  29583  8369  6212  28466  2057  15635  13000  26526]
      [8  14807  9951  26984  28548  21748  3623  4649  27040  3974  1166  20659  9364  7250  26250  13380  13411]
      [9  25844  8848  11223  22277  2153  2671  8646  3214  18336  3848  2012  29918  13968  2172  1010  17417]
      [10  21053  15002  28053  12507  13719  20516  26981  28188  3240  15423  24160  1387  3235  19728  1491  11745]
      [11  2552  14775  18910  22433  28348  25857  24721  19281  6900  28058  5052  7167  3575  6412  19436  8012]
      [12  2744  26543  21619  7155  28288  11731  3161  13445  8433  23085  10141  13515  1365  19675  21825  9810]
      [13  16762  29514  15579  9661  27014  28917  3893  5979  22904  28059  13724  3155  21045  18368  8271  353]
      [14  6977  17105  8330  23454  20002  11168  8813  20289  27079  28958  4558  1987  25579  14127  12955  24564]
      [15  13853  11470  8180  11119  6732  17867  2872  2662  25834  14853  13338  8609  18962  4379  23716  6386]]


    set write-workload
    [[0  1441  5385  9781  556  1413  8052  10341  6957  13068  12414  10897  8199  11432  7152  11851  680]
      [1  670  5672  2730  13599  10969  11690  12472  4546  11751  9091  12648  9781  6592  11746  4653  14012]
      [2  793  4894  14179  9364  14560  7745  3585  13663  9337  8034  11871  6525  11355  2024  13211  12640]
      [3  2653  8476  10990  3273  13030  2946  11622  4102  9446  11262  5816  14799  6105  8815  2528  4315]
      [4  3542  12371  9945  3880  9545  4673  5373  5535  11170  5116  1107  13224  8957  3406  11983  11032]
      [5  8930  3754  11697  31  2453  7863  1101  9698  4022  8074  1463  273  2652  632  6211  10410]
      [6  13030  10459  13906  5134  6257  6557  11731  2502  3862  10510  9606  10489  2192  803  10997  2803]
      [7  13929  6116  12925  13115  5386  5243  8383  8964  13684  7249  7750  11361  2364  5537  6  1925]
      [8  127  5045  3373  10638  14765  14531  5517  7463  14286  4231  1107  176  2495  1316  9212  7671]
      [9  5148  8438  10936  4630  5322  13212  11366  12306  7623  6256  11584  12243  5038  14208  11780  11208]
      [10  9956  12592  10987  239  1775  4907  11039  5916  13720  8510  10128  6257  1874  2258  12235  14090]
      [11  14370  13383  12168  3586  7706  9797  14657  12571  6520  1895  4878  4936  4520  4144  6430  6269]
      [12  5126  512  7503  7277  3495  11722  11695  9327  3608  3125  1736  2319  3453  9525  11040  11457]
      [13  3186  4996  2445  5289  4864  4569  14350  7355  4463  7120  5194  7381  8794  10611  13562  7622]
      [14  6040  12565  3948  1631  1991  3710  14912  4345  207  1065  10223  12944  12753  11202  2971  9644]
      [15  10113  12568  14722  3165  14505  10445  7075  6898  367  1971  14877  2924  13402  3570  8654  11123]]
  ]

  if  network-size = 5
  [
    set read-workload
      [[0  7586  4914  155  8018  8932  4453  1990  7744  1789  720  7171  1004  1383  7870  8114  7578  8041  830  8883  3208  6965  7457  8341  8962  1718  2109  1725  5521  4652  2511  8085  1256]
        [1  7100  832  2670  2848  3110  3681  4379  8275  1516  510  5749  4242  3823  7356  4281  2562  8905  4940  291  5021  7130  844  7856  8753  623  2855  1496  3832  8882  7866  5014  399]
        [2  4724  3166  4796  5479  5117  112  8875  5226  5790  6534  3447  6692  5796  2794  6950  4560  5518  4248  3602  5870  5000  2866  1710  8161  8634  1744  2970  7370  7434  3746  7109  5137]
        [3  6954  6691  6163  7621  791  2383  6056  1802  5851  4778  8571  1080  4300  8310  6374  4765  8768  797  5045  2356  1626  7755  2716  1006  6211  2925  7406  2986  3478  5592  4813  4954]
        [4  4795  5622  2810  1672  4089  1296  5790  3712  3533  852  8488  4201  4694  3358  718  2256  4454  5422  2200  2772  7570  5422  7318  7202  6453  4968  2507  1230  4380  6833  7429  272]
        [5  7173  5988  8631  4597  5963  3134  199  2161  3320  812  1125  2209  4870  4383  6711  6304  6504  8787  3720  6534  4888  6569  7991  7938  453  3844  4916  2469  1684  6856  1914  6318]
        [6  2165  7259  688  919  8923  6676  5292  5701  4839  3383  1652  7086  5971  8492  1300  4208  3418  5174  6106  2135  285  5408  7663  2486  4735  6748  5075  765  8964  3379  4471  305]
        [7  8246  8225  1859  8876  6647  7729  6100  4266  3136  6675  4156  2214  6889  2631  8553  1961  1051  6056  2143  5321  5072  3468  8893  3266  4530  2822  1209  8116  7863  5035  875  2122]
        [8  7903  3182  2214  4383  6878  4815  2124  6629  2320  184  489  2233  2303  5324  220  6829  908  3698  2608  3543  4138  3411  5814  3829  4532  6803  5044  3500  1500  8140  6857  1177]
        [9  745  6511  8785  1730  4497  4330  4311  4587  3074  7293  2148  3601  5026  7959  7347  36  8451  5309  7751  6735  78  2073  3872  1353  33  5297  7374  7894  5877  4768  4809  7211]
        [10  2580  6341  261  6828  6632  3816  7397  7398  5973  870  6412  5650  6428  4580  1579  579  162  239  911  6673  7067  6384  4249  5844  5078  5626  1250  3554  8143  1324  7912  3624]
        [11  6709  3991  1407  4314  1619  1398  4321  7892  3751  4735  6927  3321  6854  4012  1093  7186  7469  439  1764  4213  489  6930  7326  1129  201  14  5199  3374  3849  7748  1095  1673]
        [12  672  4436  2211  5511  5075  7429  5489  1497  3113  1468  6047  3122  6531  8392  2663  4148  7653  5276  8494  907  3417  56  7077  3017  8850  596  6600  3954  1474  193  2910  1607]
        [13  4689  5970  5898  2712  4199  2750  4825  6089  7731  626  2710  6042  5658  4474  1463  3670  8037  991  5430  1149  1970  7155  2274  7990  8520  2228  2972  8362  4965  3196  2751  84]
        [14  5697  405  4838  3848  6743  4142  7260  5873  1025  2334  1185  5075  5509  3605  6954  838  522  8998  4503  4537  5322  788  4235  6517  1269  6357  6510  6491  4512  8235  778  8036]
        [15  3119  3287  8655  813  2870  8150  1273  7585  2526  5308  4950  3225  8304  6843  4785  840  1555  5773  7698  2351  5460  4482  6283  3205  6424  4879  5972  6432  1524  6847  2817  543]
        [16  2960  2917  2483  6693  3475  6134  7188  514  7932  6269  1167  5719  4094  5959  4968  8650  7612  2259  7810  7644  3603  841  776  3526  928  6263  5802  3414  4695  755  2112  5665]
        [17  5464  5989  7610  7657  3544  3359  6970  4522  8386  2733  2325  8812  6578  7954  3149  3776  6392  1535  881  6622  4857  4543  1071  351  7873  4160  1711  3653  3446  2680  7754  6404]
        [18  689  5469  5669  5266  7342  3532  6891  2247  301  6685  3654  6436  2849  3372  2521  809  107  2139  6860  8179  2393  1735  3564  7514  311  2233  3809  1782  2929  5675  5608  5146]
        [19  6944  847  3884  8758  1466  2590  784  8541  3925  3106  1256  2335  1927  4334  6889  1777  7765  2716  5498  3416  5818  1612  2406  860  3316  8222  5438  4546  1337  7123  8044  4771]
        [20  7081  4999  1693  2813  3371  5047  3466  1050  1050  8777  7322  7860  3216  3503  2115  3873  4243  4532  5893  4424  8550  1073  4438  2399  4676  5723  8400  5014  1317  1893  7519  359]
        [21  8695  90  3017  8400  1317  8064  2748  1622  3703  1216  8777  3577  2794  8224  7825  6881  4848  1069  5121  6138  7015  7911  2472  2159  7801  6932  3075  2334  4289  7398  6308  870]
        [22  3601  5029  3157  2397  2536  2334  3195  7950  5846  1644  7624  5788  1915  599  1479  7400  1006  4454  5656  5013  168  1683  7734  1679  5341  277  130  593  6461  3595  2679  3923]
        [23  6568  1122  6516  2699  4114  113  4952  259  5413  6752  6575  2888  6656  5713  445  6511  8637  5265  4547  5048  1823  7820  329  3537  1379  4951  6046  3448  483  8150  5429  2374]
        [24  7313  5145  2257  5800  7297  5724  636  6791  5943  6378  3972  954  6939  3332  1908  737  670  4135  179  5268  288  575  1587  7053  3166  6834  5180  71  7374  7148  6057  8339]
        [25  1163  5458  7640  807  5592  5621  7551  7791  3544  7635  8450  7552  1377  6393  8248  5722  5985  8745  8157  1771  2879  2900  8531  6954  3424  5233  819  5362  7133  6053  3254  5082]
        [26  7958  3964  1776  2602  2029  7468  6199  5588  4727  5508  2293  8597  5558  3360  5434  297  1182  3215  373  8933  2466  2043  5498  1418  8949  8185  1937  5077  543  5265  2911  6376]
        [27  7898  4124  4238  482  2866  2438  5213  327  1908  986  5724  1511  2966  1739  961  2174  4626  5340  2270  1793  3731  1004  2185  4922  5646  2457  5587  428  5488  735  5756  8396]
        [28  4721  1347  7780  4513  5975  4902  424  2178  5139  126  831  7664  1207  8535  6766  1243  2385  1360  6754  7644  2602  2488  6657  2257  1641  7497  1104  7261  7648  8933  644  911]
        [29  1587  2927  1462  8303  349  1867  3773  809  1418  7553  5743  409  4342  6392  7146  5822  7177  6888  419  6399  7519  8554  2948  1794  8623  473  1007  3860  569  7408  5943  7974]
        [30  4913  1448  6672  99  4716  4291  2392  5595  8953  4244  8209  8852  2659  286  2543  5754  3206  5392  8093  3613  2083  541  8568  6800  5766  5768  8590  4797  3366  1331  2756  4931]
        [31  591  186  7532  6021  7359  6027  6368  7232  352  1071  2011  8842  1488  2630  7758  3637  5327  2411  3977  4191  3498  4203  8993  417  5515  3006  350  4212  6325  2075  1528  8091]

      ]

    set write-workload
      [
        [0  720  1208  4522  4530  2972  3506  3760  3952  1721  4631  4242  2090  4991  2717  2753  363  859  2642  2586  2599  3738  489  4342  462  1908  474  574  2118  2021  1392  3  2462]
        [1  1385  1253  2599  1674  3939  4156  3625  3976  3820  2009  4235  129  1468  4096  85  994  380  3926  187  2731  3370  2329  2729  4673  1996  4983  2826  1327  2536  2909  796  2688]
        [2  1577  1080  2913  3594  4549  319  1286  4504  2483  481  4124  3520  2629  3435  1585  1202  29  3537  2732  2244  509  1366  3222  385  2819  1826  2227  4975  3673  4228  3627  2397]
        [3  3501  4923  711  800  2866  521  1665  1269  327  431  3039  1271  1506  2939  4233  2919  4019  526  1437  2047  1093  1520  2747  2543  1102  97  3620  25  4802  1678  2691  3852]
        [4  3447  2501  1555  2023  2217  2835  167  1604  4781  851  2766  4691  2614  465  4347  1482  4733  558  1311  727  240  3541  1439  2685  4879  3177  1860  1962  2682  2965  3477  2042]
        [5  1344  2995  1113  2128  3983  2260  3505  2832  1488  871  2989  308  4023  809  4738  2352  1084  1406  4973  1882  715  1983  677  4840  4576  1350  1229  2341  3204  3965  3145  2490]
        [6  3633  3594  4440  1003  4628  283  2718  4001  1700  3077  1255  2473  4595  3719  4241  686  2673  1965  853  4126  3231  2523  1593  421  3956  3372  498  2232  1497  2108  2865  684]
        [7  3192  1638  394  2761  3972  244  4684  1158  4361  4838  1665  2802  2600  1178  1043  3802  3516  4942  3228  193  3545  2772  4482  4729  1141  4099  1605  3072  3061  2260  2269  4801]
        [8  3996  784  2709  4348  3564  4882  4896  4512  896  1245  2296  837  3801  4685  3086  1070  97  4851  3472  1501  3503  1852  4291  3549  2353  4511  2882  3827  2742  623  3178  2378]
        [9  962  4191  3321  4562  3268  1498  4124  2450  847  3573  3552  4926  2962  1503  3305  1750  4781  4861  397  2790  1629  2355  3829  1647  3876  1205  1974  2009  2649  4504  2456  2714]
        [10  4970  3333  2784  2750  957  2155  921  3411  93  4596  4807  153  2536  4339  1037  2528  3467  2734  3402  107  2301  4635  3379  1389  1244  1895  754  1931  1585  2657  4481  615]
        [11  2890  1283  2622  2045  1031  310  44  1910  3655  4366  4297  3580  3759  3899  1894  4624  3681  4118  574  2431  353  1686  4753  1516  397  4024  3842  2551  1342  4646  221  439]
        [12  2217  2756  4859  2856  468  1214  1525  4626  1920  2176  1409  3118  1022  3511  4867  742  3232  2206  2293  492  1077  1122  1969  2320  4509  4686  3244  1440  4361  5  818  3202]
        [13  4814  2231  662  4547  3102  2754  4282  3760  44  1872  1442  389  1763  2772  1861  4047  153  1665  3332  3070  398  2309  1670  1785  3842  4062  1377  4218  548  4576  1875  3050]
        [14  159  2060  2675  2047  2752  1613  2362  1995  4316  3978  4690  1083  2280  835  832  1905  450  592  38  34  4694  3170  179  1486  3840  4487  1326  4349  2205  4931  3789  3590]
        [15  962  97  4959  4409  4128  677  3974  3034  3404  745  1298  2628  4632  290  963  4587  998  88  2875  2266  2006  4488  4898  3892  4919  3297  2435  1929  3718  1529  224  2135]
        [16  68  4264  1333  263  4202  2083  3613  2173  3139  788  4940  2163  3808  4549  98  898  2881  2610  2743  3820  806  666  209  959  1607  1983  2354  4403  2967  2266  2015  3087]
        [17  3039  4141  1066  602  4733  1845  1835  2710  993  277  4715  4148  2968  4897  1212  2643  74  1424  750  380  2864  3751  3458  96  3410  982  4048  555  2487  773  1711  3380]
        [18  4661  355  3867  2334  3301  4248  4855  405  1860  3174  3667  3844  4879  1065  2807  4196  1435  3201  4355  281  1221  1742  2963  113  701  1712  435  2806  4916  932  4843  1978]
        [19  701  1230  3418  1360  2031  4058  3495  1787  4904  4956  4670  659  2209  4537  1290  1780  1243  4303  2937  4594  2647  1147  3920  439  1913  2708  4131  3249  2597  4867  2998  3177]
        [20  3814  2824  856  88  1735  4072  11  1657  4728  672  1270  476  3805  233  3868  405  1376  2632  4347  552  406  1195  4794  3255  2033  3248  3518  2293  704  2345  3302  540]
        [21  949  1384  1935  479  72  3630  4126  1725  4213  4749  1056  1258  1076  4645  493  3207  3576  3033  3866  1491  2315  631  792  2226  4755  3063  3796  4247  25  1045  4288  2798]
        [22  4231  1012  4416  2632  1568  2001  805  2455  4199  611  2041  1102  783  681  2896  2310  874  904  4436  1987  4691  3880  884  4922  4675  158  2162  1848  2498  4268  1323  1340]
        [23  3013  2177  3492  1031  1865  4826  473  2480  3451  902  2411  436  4079  1207  3919  3124  1852  3156  3453  2995  4000  206  4993  3420  3779  1514  2392  1458  4049  4778  623  1036]
        [24  810  1654  2319  4308  791  868  4078  1070  3635  646  1413  667  4969  33  1959  2186  4932  4431  1787  4796  3224  143  2802  3334  2696  3705  3118  1232  447  1058  4524  4424]
        [25  2178  2377  2163  2560  1311  325  130  3274  4649  4475  3628  566  2945  26  4784  433  3898  3487  2962  1410  813  897  2507  4700  87  1724  1861  4446  3741  4227  154  1581]
        [26  3629  2947  2171  701  693  1064  3755  4995  1574  3392  4008  2346  2944  4138  1966  1690  2317  1259  3361  3144  130  4173  1722  3294  1258  4592  2774  2358  1159  741  3218  4135]
        [27  3814  3530  3382  2452  2630  1844  4917  223  3653  4444  854  4616  2554  1903  2478  3814  4152  2447  13  700  2712  1539  111  3947  4287  2563  361  3775  1775  1602  2121  4252]
        [28  4386  592  2186  1208  697  2556  3863  993  2443  3591  3062  2155  710  2814  2343  2809  2536  3180  4952  4973  2446  4359  3197  3080  986  1758  90  1734  4948  2802  3937  297]
        [29  4825  1175  4883  289  4927  2087  3916  2060  310  1719  3992  2810  2359  2380  2377  4223  2218  3299  138  3867  4618  1475  861  2914  2315  819  2632  2482  16  1453  1803  784]
        [30  1846  161  4526  1409  2486  3411  516  4017  43  307  982  4685  4069  3564  2893  1172  2080  2853  2041  1732  1816  3405  83  3529  3499  368  4696  4375  482  4595  2515  3936]
        [31  3728  2328  3543  4472  3701  4387  3719  1936  189  4723  3599  2055  4641  3364  2313  2797  1701  4802  154  3236  972  192  1695  4660  4442  1158  1618  3620  4136  3499  4236  3622]
      ]
  ]


end

;; setup Radix workload from the SPLASH2 testbench for given network size
;; these values were obtained through analysis of the testbenches using PinTool
to set-Radix


  if network-size = 2
  [
    set read-workload
    [[0  501521  1206542  1335202  707538]
      [1  236172  1260530  1051076  728173]
      [2  675169  1130905  28240  191512]
      [3  98046  1401476  730308  291291]]

    set write-workload
    [[0  730011  730036  600602  674270]
      [1  29510  540043  238158  490604 ]
      [2  394906  791706  280125  765747]
      [3  442590  661344  819524  76638 ]]
  ]

  if  network-size = 3
  [
    set read-workload
    [
      [0  159402  51587  115093  210357  219232  118035  365312  189910]
      [1  162001  187542  2267  65649  377493  33901  291840  65194]
      [2  166961  20408  38517  357839  85253  63100  38354  180196]
      [3  126531  32076  63689  279096  355235  205224  22179  375214]
      [4  90227  21423  339344  298038  31119  354084  241239  4221]
      [5  73312  85972  107469  267545  176897  202312  303325  341522]
      [6  286367  150708  255914  41042  140579  298271  223688  122500]
      [7  52921  365310  267788  341641  57449  313972  159792  88002]
    ]

    set write-workload
    [
      [0  12331  148783  178884  101291  104016  45096  209294  115687                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      ]
      [1  153215  28548  63048  42293  208572  140180  106192  118014                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      ]
      [2  53034  6558  131161  984  138574  153545  209089  145419                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          ]
      [3  150397  10280  119812  145941  160531  36207  63746  195865                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      ]
      [4  147854  147929  37977  182901  69116  191916  82769  125577                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      ]
      [5  23745  19785  122639  148876  49044  7744  110314  169920                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      ]
      [6  188720  137800  80886  122082  205562  83420  35929  196951                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      ]
      [7  109135  128039  100159  160659  164361  54334  15580  9992]]
  ]

  if  network-size = 4
  [
    set read-workload
    [
      [0  108901  89931  103509  38964  29089  104655  456  51381  57877  9523  41550  83923  94598  15109  5224  51600]
      [1  105921  28663  64523  107553  73551  66223  21893  64672  21081  29517  95253  60162  1083  29868  64384  95785]
      [2  37796  108036  11950  62982  10292  54597  91255  20407  35653  88785  56631  77823  3100  87111  90115  32322]
      [3  15225  34350  5254  35059  2220  72195  100457  103700  32610  50427  24919  24746  19421  76361  34395  8072]
      [4  86711  73260  79489  57480  88045  16950  35677  13641  39362  15103  3981  86202  45322  29546  10942                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        