breed [banks bank]
breed [consumers consumer]

undirected-link-breed [loanlinks loanlink]

directed-link-breed [interbankloans interbankloan]

globals [totalloss]

banks-own [
  bank-reserves ;;the amount of money a bank has
  initial-reserves ;;initial reserves a bank has - used to calculate total losses of money from bank failure
  bank-prestige ;;this is a simplified property that denotes multiple characteristics of a bank. First of all, it denotes 
                ;;a bank's standards for borrowers, basically whether or not a bank is willing to take on a high risk customer. The higher the value, the less likely.
                ;;Secondly, it is a measure of the bank's desirability to borrowers. They will seek a bank that is known to be more stable. 
                ;;Finally, it denotes the level of connectivity the bank has with other banks, both in terms of loans taken and loans granted. This is because
                ;;prestigious banks tend to be well-connected with others. 
]

consumers-own [
  reliability ;;a consumer's ability to pay back loans
  loan ;;the amount a consumer has taken out in loans from a bank
  consumer-standard ;;the higher this value, the more prestigious a bank the borrower will attempt to acquire a loan from. This value is presumed to be independent of one's ability to pay
                    ;;back a loan, thus it is not correlated with reliability in this model. 
]

to setup
  ca
  setup-banks
  setup-consumers
  network-banks
end

to setup-banks
  create-banks bank-count [set bank-reserves random reserve-amount + 10
    set shape "circle" 
    set color red 
    set bank-prestige random 100 
    move-to one-of patches
    set initial-reserves bank-reserves] ;;Although the banks are randomly placed spatially, this is only for the sake of appearance. It does not affect the model's performance as everything is done
                            ;;with a network topology.
end

to setup-consumers
  create-consumers 100 + bank-count [set reliability random 100 - neg-reliability-mod ;;this is here to simulate the notion that the pool of lenders as a whole has gotten more unreliable, which is the case according to some economists
    set consumer-standard random 100
    set shape "sheep" 
    set color blue
    move-to one-of patches] ;;Same comment with banks applies here. For the sake of simplicity, spatial distance is not a consideration for consumers when choosing a bank.
                            ;;Given that the banks heavily implicated in the financial crisis were large and ubiquitous, this is not an unreasonable assumption.
end

to network-banks
ask banks [create-interbankloan-to one-of other banks]
repeat bank-connectivity [ask banks [if random 100 < bank-prestige [create-interbankloan-to one-of other banks]]] 
;;Every bank has at least one loan with another bank. An overall measure of connectivity is then used to determine how many additional loans exist between banks. 
end


to go
if not any? banks [stop]
if not any? consumers [stop]
ask banks [if bank-reserves <= 0 [bank-failure]] ;;banks fail when they're out of reserves
ask consumers [seek-loan]
ask consumers [check-loan]
tick
make-plot
make-plot2
if layout? [layout]
end

to seek-loan
  if count my-loanlinks = 0 [network-with-bank]
end


to network-with-bank ;;this is the command that will make a consumer try to create a loan with a bank, depending on their preferences
  if count banks > 0 [
  if loan = 0 [ask my-links [die]
  ifelse any? banks with [bank-prestige >= ([consumer-standard] of myself)] 
  [ask one-of banks with [bank-prestige >= ([consumer-standard] of myself) ] [if random 100 + ([reliability] of myself) > (bank-prestige - bank-risk-taking) [create-loanlink-with myself]]
  set loan loan-amount]
  [set consumer-standard consumer-standard - 2]]] ;;standards lower over time as consumers realize they will not be able to find their ideal bank
  end

to check-loan
  if count my-links > 0 [if random 100 > 75 [ifelse random 100 > reliability + 30 [if count my-links > 0 [ask one-of loanlink-neighbors [loseloan]set loan 0 die]]
    [ask loanlink-neighbors [regainloan] ask my-links [die] if random 100 > 50 [die]]]] ;;the random chance of dying is to gradually remove repeat-borrowers from the pool. 
                                                                                        ;;it isn't realistic that someone would take out repeated loans
end

to regainloan
  set bank-reserves bank-reserves + loan-amount + 1 ;;extra 1 here is to represent repaid interest on a loan
end

to loseloan
set bank-reserves bank-reserves - loan-amount
end   

to bank-failure
ask out-link-neighbors [set bank-reserves bank-reserves - interbank-loan-amount set totalloss totalloss + initial-reserves] ;;As a bank fails, all banks it was taking loans from suffer as the loan                                                                                                                ;;is defaulted upon
  die
end

to make-plot
  set-current-plot "bank count"
  set-current-plot-pen "default"
  plot count banks
end

to make-plot2
  set-current-plot "total losses"
  set-current-plot-pen "default2"
  plot totalloss
end

;;Following code is cannibalized from Preferential Attachment for the sake of making the network here look more aesthetically pleasing.   
;;
to layout
  ;; the number 3 here is arbitrary; more repetitions slows down the
  ;; model, but too few gives poor layouts
  repeat 3 [
    ;; the more turtles we have to fit into the same amount of space,
    ;; the smaller the inputs to layout-spring we'll need to use
    let factor sqrt count turtles
    ;; numbers here are arbitrarily chosen for pleasing appearance
    layout-spring turtles links (1 / factor) (7 / factor) (1 / factor)
    display  ;; for smooth animation
  ]
  ;; don't bump the edges of the world
  let x-offset max [xcor] of turtles + min [xcor] of turtles
  let y-offset max [ycor] of turtles + min [ycor] of turtles
  ;; big jumps look funny, so only adjust a little each time
  set x-offset limit-magnitude x-offset 0.1
  set y-offset limit-magnitude y-offset 0.1
  ask turtles [ setxy (xcor - x-offset / 2) (ycor - y-offset / 2) ]
end

to-report limit-magnitude [number limit]
  if number > limit [ report limit ]
  if number < (- limit) [ report (- limit) ]
  report number
end
@#$#@#$#@
GRAPHICS-WINDOW
405
25
844
485
16
16
13.0
1
10
1
1
1
0
0
0
1
-16
16
-16
16
0
0
1
ticks

BUTTON
20
15
83
48
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

BUTTON
94
16
157
49
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL

SLIDER
20
240
192
273
loan-amount
loan-amount
0
20
10
1
1
NIL
HORIZONTAL

SLIDER
20
275
195
308
interbank-loan-amount
interbank-loan-amount
1
50
20
1
1
NIL
HORIZONTAL

SLIDER
19
169
191
202
bank-connectivity
bank-connectivity
1
10
5
1
1
NIL
HORIZONTAL

SLIDER
20
98
192
131
bank-count
bank-count
15
40
40
1
1
NIL
HORIZONTAL

BUTTON
20
55
116
88
redo-layout
layout
T
1
T
OBSERVER
NIL
NIL
NIL
NIL

SWITCH
230
20
333
53
layout?
layout?
0
1
-1000

SLIDER
19
205
191
238
reserve-amount
reserve-amount
20
50
30
1
1
NIL
HORIZONTAL

SLIDER
21
310
193
343
bank-risk-taking
bank-risk-taking
0
100
35
1
1
NIL
HORIZONTAL

MONITOR
230
65
332
110
NIL
count banks
17
1
11

PLOT
20
355
195
505
bank count
ticks
banks
0.0
10.0
0.0
10.0
true
false
PENS
"default" 1.0 0 -16777216 true
"banks" 1.0 0 -16777216 true

SLIDER
20
135
192
168
neg-reliability-mod
neg-reliability-mod
0
100
0
1
1
NIL
HORIZONTAL

PLOT
210
355
395
505
total losses
NIL
NIL
0.0
10.0
0.0
10.0
true
false
PENS
"default" 1.0 0 -16777216 true
"default2" 1.0 0 -16777216 true

@#$#@#$#@
WHAT IS IT?
-----------
This model attempts to explain the 2007-2010 financial crisis by exploring the interaction between banks and consumers (borrowers). A common hypothesis is that the financial crisis occurred largely because of a burst in the housing bubble. A large number of borrowers were granted loans who should not have been. "Sub-prime mortgages" were used by banks when it was clear many of the people who took out these mortgages would be unable to pay them back. Therefore, when the inevitable happened and some borrowers defaulted on their loans, available credit went down as banks lost money. 

Furthermore, this model tests the roll of bank interconnectivity in the financial collapse. Not only do banks vary in their willingness to take on high risk borrowers, they also vary in how many interbank loans they have taken or given. Another curious phenomenon of the financial crisis was how quickly the failure of one bank led to a cascading effect that took down other banks, in the form of loss of reserves due to interbank loans that were lost when a borrowing bank failed. 


HOW IT WORKS
------------
Borrowers attempt to seek loans from banks. They search for banks with a certain level of prestige - these banks are better connected and generally only serve more reliable borrowers. Banks deny or grant loans based on the borrower's perceived reliability at repaying the loan. If a borrower is unable to secure a loan, they lower their standards for how prestigious a bank they wish to borrow from. 

When a loan has been established between a consumer and a bank, there is then a chance that the consumer either pays it back, with interest, or defaults on the loan, which causes the bank to lose a portion of its reserves. Once a bank's reserves go below 0, then the bank fails and all banks that it has taken an interbank loan from lose money from their reserves as well. 


HOW TO USE IT
-------------
There are many setup settings for the model, but it should not seem intimidating. The number of banks can be set with bank-count, but it is generally only useful for simulating greater levels of bank connectivity. 

The property neg-reliability-mod dictates how unreliable on a whole the population will be. A higher value leads to a population of consumers who are unlikely to be able to pay off their loans.

Bank-connectivity dictates how interconnected and thus interdependent the network of banks is. At the minimum, each bank will have granted one interbank loan to one other bank. Higher levels of interconnectivity represent more loans granted. 

Reserve-amount helps determine how much money each bank has in its reserves to start. 

Loan-amount determines how large the loans granted by the banks are. This can have the effect of exacerbating the negative effect of defaulting on loans.

If you think of bank-connectivity as a measure of the breadth of interdependent networking between banks, then you can think of interbank-loan-amount as determining the magnitude or importance of these interdependent links. The larger this amount, the more money a bank will lose when a bank it has granted a loan to fails. 

Bank-risk-taking is an attempt to simulate the notion that banks might be more receptive to giving out bad loans (sub-prime mortgages). This helps explore the idea that a lack of regulation caused banks to give out loans they should not have. At a higher level of risk-taking, banks are more likely to give out loans they don't believe will be paid back. 

The layout switch exists for purely aesthetic reasons. It uses the same code as the Preferential Attachment model for making the network easier to look at. 

THINGS TO NOTICE
----------------
When watching the plots that show the number of remaining banks and the total amount of money lost in the financial crisis, pay attention to the slope of the line. The model fairly well matches with empirical data that suggests there is an accelerating or cascading effect of bank failures. 

Also pay attention to how different configurations of interbank loans generated at setup can create different results despite the value for bank connectivity remaining the same. The sheer number of connections is important, but so is the layout of those connections. 


THINGS TO TRY
-------------
By far the most interesting variable to play with is the bank-connectivity. Try varying this setting at different static amounts of neg-reliability-mod or bank-risk-taking. Why does this have such a large effect on bank failures compared changes to the commonly theorized causes of the financial crisis?


EXTENDING THE MODEL
-------------------
This model is a simplification of the financial crisis and does not take into account how hedge funds work or the specific intricacies of interbank loans. One notable way to make the model more accurate would be to split up the bank's reserves into assets and liabilities and introducing the idea of external shocks to the system rather than having bad borrowers being the only catalysts for bank failure.  


NETLOGO FEATURES
----------------
The procedure for determining what bank a consumer aims at borrowing from as well as whether the bank will grant a loan to that consumer relies heavily upon the "myself" function to compare properties between banks and consumers. 

Directed links are necessary to represent interbank loans. Undirected links are inadequate for that task because they cannnot represent who is the lender and who is the borrower. This is not important for loans between banks and consumers because all of the loans go in one direction anyway. 


RELATED MODELS
--------------
There are multiple models in the library that are relevant to this one, including a Bank Reserves model. However, these economic models tend to be in the unverified folder of the Social Science folder. 


CREDITS AND REFERENCES
----------------------
This section could contain a reference to the model's URL on the web if it has one, as well as any other necessary credits or references.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
0
Rectangle -7500403 true true 151 225 180 285
Rectangle -7500403 true true 47 225 75 285
Rectangle -7500403 true true 15 75 210 225
Circle -7500403 true true 135 75 150
Circle -16777216 true false 165 76 116

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 4.1.3
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="experiment" repetitions="1" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="75"/>
    <metric>totalloss</metric>
    <metric>count banks</metric>
    <steppedValueSet variable="neg-reliability-mod" first="0" step="5" last="70"/>
  </experiment>
  <experiment name="experiment" repetitions="1" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>count turtles</metric>
    <enumeratedValueSet variable="layout?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="bank-count">
      <value value="40"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="neg-reliability-mod">
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="reserve-amount">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="interbank-loan-amount">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="bank-connectivity">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="loan-amount">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="bank-risk-taking">
      <value value="55"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="experiment" repetitions="1" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>count turtles</metric>
    <enumeratedValueSet variable="layout?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="bank-count">
      <value value="40"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="neg-reliability-mod">
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="reserve-amount">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="interbank-loan-amount">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="bank-connectivity">
      <value value="9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="loan-amount">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="bank-risk-taking">
      <value value="55"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 1.0 0.0
0.0 1 1.0 0.0
0.2 0 1.0 0.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
1
@#$#@#$#@
