globals [
  filled          ;; whether the spot a car is considering parking in is filled
  destin-x2       ;; x-location of a store
  time            ;; how long a customer is in a store
  all-filled      ;; whether all the visible parking spots better than the current one are open
  dist            ;; x-coordinate of a car
  dist-destin     ;; distance between a car and its destination
  dist-vision     ;; global variable of the farthest x-coordinate a car can see
  total-pop       ;; total popularity value of the stores
  num-stores
;;
  avg-dist-0      ;; average distance cars park from store-1
  avg-dist-1      ;; average distance cars park from store-2
  avg-dist-2      ;; average distance cars park from store-3
  avg-dist-3      ;; average distance cars park from store-4
  avg-dist-4      ;; average distance cars park from store-5
;;  
  miss-rate-0     ;; average distance cars park from store-1
  miss-rate-1     ;; average distance cars park from store-2
  miss-rate-2     ;; average distance cars park from store-3
  miss-rate-3     ;; average distance cars park from store-4
  miss-rate-4     ;; average distance cars park from store-5
]

breed [cars car]    
breed [stores store] 

cars-own [
  destin          ;; store where passengers are going
  destin-x        ;; x-coordinate of destin
  time-in-store   ;; amount of time will stay in store
  parked          ;; amount of time parked
  parking-time    ;; amount of time remains parked before leaving
  vision          ;; amount of spots car can see ahead if vision? turned on
]

stores-own [
  popularity      ;; variable that affects liklihood that cars will be coming to that store
  time-in         ;; average amount of time people stay in the store
  total-dist      ;; total amount of distance cars have parked away from the store
  avg-dist        ;; average distance cars park away from the store
  count-parked    ;; amount of cars that have parked on the street in one try (used if side-street-parking? is off)
  count-missed    ;; amount of cars unable to find a spot (used if side-street-parking? is off)
  miss-rate       ;; count-missed / ( count-parked + count-missed )
]

to setup
  clear-all
  setup-cars          
  setup-parking
  setup-stores
  determine-total-pop
  do-plots
end

to setup-cars ;; sets the shape of the cars
  set-default-shape cars "car"
end 

to setup-parking ;; colors the parking spaces
  ask patches with [pycor = 0 and pxcor > -25] [set pcolor yellow]
end

to setup-stores ;; sets the possible store locations white, and then places stores at user defined locations, also sets the shape of stores
  ask patches with [pycor > 0] [set pcolor white]
  create-stores 1 [
    set color red
    setxy (store-1-location - 25) 1
    set time-in time-in-1
  ]
  if add-store-2?[
    create-stores 1 [
      set color blue
      setxy (store-2-location - 25) 1
      set time-in time-in-2
    ]
  ]
  
  if add-store-3?[
    create-stores 1 [
      set color green
      setxy (store-3-location - 25) 1
      set time-in time-in-3
    ]
  ]
  
  if add-store-4?[
    create-stores 1 [
      set color orange
      setxy (store-4-location - 25) 1
      set time-in time-in-4
    ]
  ]
  
  if add-store-5?[
    create-stores 1 [
      set color violet
      setxy (store-5-location - 25)  1
      set time-in time-in-5
    ]
  ]
  set-default-shape stores "house"
end 

to go
  move-cars
  decide-to-park
  parked-time
  new-car-parking
  do-plots
  tick
  if ticks >= 10000 [
    stop
  ]
end


to move-cars ;; moves the cars forward 1 patch along the road
  ask cars [
    if ycor = -1 [
      forward 1
    ]
  ]
end 

to decide-to-park ;; determines if a car parks 
  ask cars [ 
    ;; next line specifies that this only applies to cars on the road
    if ycor = -1 [
      ;; this section determines if the space a car is next to is full
      if patch-at 0 1 != nobody [
        ask patch-at 0 1 [
          ifelse count turtles-here = 0 [
            set filled 0
          ]
          [ set filled 1
          ]
        ]
      ]
      ;; if the spot is full and it is the last spot, either the car fails to find a spot or parks elsewhere
      if filled = 1 and xcor = 25 [
        ;; if there is side parking, the car parks elsewhere
        ifelse side-street-parking? [
          ask destin [  
            set count-parked count-parked + 1
            ;; this is determining whether the car parked to the left of the block or to the right based on which distance is shorter
            ifelse avg-dist-left + 2 * xcor <= avg-dist-right [
              set total-dist total-dist + avg-dist-left + xcor + 25
            ]
            [
              set total-dist total-dist + avg-dist-right - xcor + 25
            ]
          ]
        ]
        [ ;; in this case there is no other parking and the car does not find a spot on the first pass
          ask destin [
            set count-missed count-missed + 1
            set miss-rate count-missed / (count-missed + count-parked)
          ]
        ]
      die
      ]
      
      ;; the biggest factor in whether a car chooses to park is whether it has passed the store it is going to
      ;; this code applies when a car is at or has passed the destination
      ifelse side-street-parking? [ 
        ;; the car will only want to park in spaces that are closer than a side-street alternative on average
        if filled = 0 and xcor >= destin-x and xcor - destin-x <= avg-dist-left + destin-x + 25 and patch-at 0 1 != nobody [
          ;; if the space is open and closer than alternatives on other streets, car parks
          move-to patch-at 0 1
          ;; the distance to the destination and the amount of time consumers stay in the destination determine length of parking
          set parking-time distance destin + time-in-store
          ask destin [
            set count-parked count-parked + 1
            set total-dist total-dist + distance myself
            set avg-dist total-dist / count-parked
            set miss-rate count-missed / (count-missed + count-parked)
          ]
        ]
      ]
      ;; the car will always want to park if their is no side street parking and it has passed the destination
      [ if filled = 0 and xcor >= destin-x and patch-at 0 1 != nobody [
          move-to patch-at 0 1
          set parking-time distance destin + time-in-store
          ask destin [
            set count-parked count-parked + 1
            set total-dist total-dist + distance myself
            set avg-dist total-dist / count-parked
            set miss-rate count-missed / (count-missed + count-parked)
          ]
        ]
      ]
      
      ;; this code applies when a car has not passed its destination
      ;; first a turtle checks to see if better spots are available farther down
      if filled = 0 and xcor < destin-x [
        set dist xcor
        set dist-destin destin-x - xcor
        ;; vison affects how many spots a turtle can see
        ifelse vision? [
           set dist-vision xcor + vision
           ;; these lines ask all the spots closer than the current alternative and visible whether they are open, if any are open, the turtle doesn't park
           ask patches with [pxcor > dist and pxcor <= dist-vision and pxcor < (2 * dist-destin + dist) and pycor = 0] [
             if count turtles-here = 0 [
             set all-filled 0
             ]
           ]
        ]
        ;; when vision? is not on, cars ask all parking spots that are closer whether they are open
        [
          set dist-destin destin-x - xcor
          ask patches with [pxcor > dist and pxcor < (2 * dist-destin + dist) and pycor = 0] [
            if count turtles-here = 0 [
            set all-filled 0
            ]
          ]
        ]
      ]
      
      ;; whether other parking exists effects whether a turtle will choose to park
      ifelse side-street-parking? [
        ;; when side-street-parking? is on, a turtle will only consider a spot when no better alternative exists on the current street or a different street
        if filled = 0 and all-filled = 1 and destin-x - xcor <= avg-dist-right - destin-x + 25 and xcor < destin-x and patch-at 0 1 != nobody [
          move-to patch-at 0 1
          set parking-time distance destin + time-in-store
          ask destin [
            set count-parked count-parked + 1
            set total-dist total-dist + distance myself
            set avg-dist total-dist / count-parked
            set miss-rate count-missed / (count-missed + count-parked)
          ]
        ]
      ]
        ;; when side-street-parking? is off, a turtle will only consider if a better alternative exists on the current street
      [ if filled = 0 and all-filled = 1 and xcor < destin-x and patch-at 0 1 != nobody [
          move-to patch-at 0 1
          set parking-time distance destin + time-in-store
          ask destin [
            set count-parked count-parked + 1
            set total-dist total-dist + distance myself
            set avg-dist total-dist / count-parked
            set miss-rate count-missed / (count-missed + count-parked)
          ]
        ]
      ]
      ;; resets the global variable all-filled for the next car
      set all-filled 1
    ]
  ]
end

to parked-time ;; keeps track of how long cars remain parked and removes them when their time is up
  ask cars [
    if ycor = 0 [
    set parked parked + 1
    if parked >= parking-time [die]
    ]
  ]
end

to new-car-parking ;; generates a new car to start parking
  ;; generates cars based on the popularites of the stores
  if random 100 < total-pop / num-stores [
    create-cars 1 [
       setxy -25 -1 
       facexy -24 -1
       determine-total-pop 
       decide-destin
       create-link-with destin
       ask destin [set destin-x2 xcor]
       set destin-x destin-x2
       ask destin [set time (time-in + random time-in + random time-in) / 2]
       set time-in-store time
       set parking-time 1
       set parked 0
       ifelse random 100 < 70 [
         set vision low-vision
       ]
       [
         set vision long-vision
       ]
    ]
  ]
end

to determine-total-pop ;; sets the total-pop the sum of the individual popularities of the stores
  ifelse add-store-5? [
    set total-pop pop-1 + pop-2 + pop-3 + pop-4 + pop-5
    set num-stores 5
  ]
    [ ifelse add-store-4? [
      set total-pop pop-1 + pop-2 + pop-3 + pop-4
      set num-stores 4
      ]
      [ ifelse add-store-3? [
        set total-pop pop-1 + pop-2 + pop-3
        set num-stores 3
        ]
        [ ifelse add-store-2? [
           set total-pop pop-1 + pop-2
           set num-stores 2
          ]
          [ set total-pop pop-1
            set num-stores 1 
          ]
        ]
      ]
    ]
end

to decide-destin ;; turtle procedure, determines which destination a turtle will choose
  let temp-pop random total-pop
  ifelse temp-pop < pop-1 [
    set destin one-of stores with [color = red]
  ]
    [ ifelse temp-pop < pop-1 + pop-2 [
       set destin one-of stores with [color = blue]
      ]
      [ ifelse temp-pop < pop-1 + pop-2 + pop-3 [
         set destin one-of stores with [color = green]
        ]
        [ ifelse temp-pop < pop-1 + pop-2 + pop-3 + pop-4 [
            set destin one-of stores with [color = orange]
          ]
          [
            set destin one-of stores with [color = violet]
          ]
        ]
      ]
    ]
end

to do-plots ;; plots the average distances cars are parking from stores and how many are unable to find spots
  ask stores with [color = red] [
    set miss-rate-0 miss-rate
    set avg-dist-0 avg-dist
  ]
  if add-store-2? [
    ask stores with [color = blue] [
      set miss-rate-1 miss-rate
      set avg-dist-1 avg-dist
    ]
  ]
  if add-store-3? [
    ask stores with [color = green] [
      set miss-rate-2 miss-rate
      set avg-dist-2 avg-dist
    ]
  ]
  if add-store-4? [
    ask stores with [color = orange] [
      set miss-rate-3 miss-rate
      set avg-dist-3 avg-dist
    ]
  ]
  if add-store-5? [
    ask stores with [color = violet] [
      set miss-rate-4 miss-rate
      set avg-dist-4 avg-dist
    ]
  ]
  set-current-plot "Average Distances"
    set-current-plot-pen "red-store"
      plot avg-dist-0
    set-current-plot-pen "blue-store"
      plot avg-dist-1 
    set-current-plot-pen "green-store"
      plot avg-dist-2 
    set-current-plot-pen "orange-store"
      plot avg-dist-3 
    set-current-plot-pen "violet-store"
      plot avg-dist-4
  set-current-plot "Miss Rates"
    set-current-plot-pen "red"
      plot miss-rate-0
    set-current-plot-pen "blue"
      plot miss-rate-1
    set-current-plot-pen "green"
      plot miss-rate-2 
    set-current-plot-pen "orange"
      plot miss-rate-3 
    set-current-plot-pen "violet"
      plot miss-rate-4
end
@#$#@#$#@
GRAPHICS-WINDOW
5
425
831
504
25
1
16.0
1
10
1
1
1
0
0
0
1
-25
25
-1
1
1
1
1
ticks

BUTTON
5
80
68
113
NIL
setup\\n
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

BUTTON
69
80
132
113
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL

SWITCH
180
10
307
43
add-store-2?
add-store-2?
0
1
-1000

SWITCH
309
10
436
43
add-store-3?
add-store-3?
0
1
-1000

SWITCH
438
10
565
43
add-store-4?
add-store-4?
0
1
-1000

SWITCH
567
10
694
43
add-store-5?
add-store-5?
0
1
-1000

SLIDER
5
10
177
43
store-1-location
store-1-location
1
50
6
1
1
NIL
HORIZONTAL

SLIDER
5
45
177
78
store-2-location
store-2-location
store-1-location
50
16
1
1
NIL
HORIZONTAL

SLIDER
180
45
352
78
store-3-location
store-3-location
store-2-location
50
26
1
1
NIL
HORIZONTAL

SLIDER
355
45
527
78
store-4-location
store-4-location
store-3-location
50
36
1
1
NIL
HORIZONTAL

SLIDER
530
45
702
78
store-5-location
store-5-location
store-4-location
50
46
1
1
NIL
HORIZONTAL

SLIDER
210
240
382
273
time-in-1
time-in-1
20
200
85
1
1
NIL
HORIZONTAL

SLIDER
210
275
382
308
time-in-2
time-in-2
20
200
85
1
1
NIL
HORIZONTAL

SLIDER
211
381
383
414
time-in-5
time-in-5
20
200
85
1
1
NIL
HORIZONTAL

SLIDER
211
311
383
344
time-in-3
time-in-3
20
200
85
1
1
NIL
HORIZONTAL

SLIDER
211
346
383
379
time-in-4
time-in-4
20
200
85
1
1
NIL
HORIZONTAL

SLIDER
385
185
557
218
low-vision
low-vision
0
long-vision
5
1
1
NIL
HORIZONTAL

SLIDER
385
150
557
183
long-vision
long-vision
low-vision
50
10
1
1
NIL
HORIZONTAL

SWITCH
385
115
488
148
vision?
vision?
0
1
-1000

PLOT
5
270
205
420
Average Distances
NIL
NIL
0.0
4.0
0.0
4.0
true
false
PENS
"default" 1.0 0 -16777216 true
"red-store" 1.0 0 -2674135 true
"blue-store" 1.0 0 -13345367 true
"green-store" 1.0 0 -10899396 true
"orange-store" 1.0 0 -955883 true
"violet-store" 1.0 0 -8630108 true

PLOT
5
115
205
265
Miss Rates
NIL
NIL
0.0
2.0
0.0
0.5
true
false
PENS
"default" 1.0 0 -16777216 true
"red" 1.0 0 -2674135 true
"blue" 1.0 0 -13345367 true
"green" 1.0 0 -10899396 true
"orange" 1.0 0 -955883 true
"violet" 1.0 0 -8630108 true

SLIDER
384
240
556
273
pop-1
pop-1
1
100
50
1
1
%
HORIZONTAL

SLIDER
385
275
557
308
pop-2
pop-2
1
100
50
1
1
%
HORIZONTAL

SLIDER
386
312
558
345
pop-3
pop-3
1
100
50
1
1
%
HORIZONTAL

SLIDER
386
345
558
378
pop-4
pop-4
1
100
50
1
1
%
HORIZONTAL

SLIDER
386
380
558
413
pop-5
pop-5
1
100
50
1
1
%
HORIZONTAL

SLIDER
210
150
382
183
avg-dist-left
avg-dist-left
0
100
20
1
1
NIL
HORIZONTAL

SLIDER
210
185
382
218
avg-dist-right
avg-dist-right
0
100
15
1
1
NIL
HORIZONTAL

SWITCH
210
115
376
148
side-street-parking?
side-street-parking?
0
1
-1000

@#$#@#$#@
WHAT IS IT?
-----------
Model of a population of cars trying to find parking spaces along a street. There are stores positioned along the street that the passengers of the cars are trying to go to. The cars attempt to minimize the distance that they park away from the store that they are going to, but they may end up driving past all open spaces in attempting to do this or instead settle for a spot far away in order to insure finding a spot. The model examines which store locations are the best for minimizing customer walking distance. It finds that, given the basic assumptions the modeler believes to be most accurate, positioning a store towards the end of a block minimizes the distance that cars have to park from the store-front. However, if certain assumptions, such as parking availability beyond the block, are not true, then it is much better to be in the middle or the front of the block.  

HOW IT WORKS
------------
At each tick, each car that is not parked moves forward one patch along the road. After moving, each car checks if the spot it can park at is full. If the spot is empty, the car checks whether it wants to park. If there is no side-street parking, the car only checks if there are any open spots closer to the store that it can see. If no such spots exist, the car parks, otherwise it moves on. If _side-street-parking?_ is selected, the car also makes sure that neither side offers better parking (on average) than its current location before deciding to park. When the car parks, it moves onto the patch above it and based on the distance to its destination determines how long it will be parked. When it has parked for its given duration, it leaves (dies). Cars that fail to park on the block also leave and either miss their destinations or park on other streets. Data on the average distance cars are parking from each store is recorded as well as how many cars are unable to find parking for each store.

HOW TO USE IT
-------------
This model can be used to represent a block containing one to five stores. Given the coding, stores must be removed or added sequentially, with higher-numbered stores only being included if all stores below it in number are also included. 

The user can also pick the positioning of each store along the block. The number in the slider corresponds to which parking space the store entrance is located at, with one being the leftmost and fifty being the rightmost. 

When side-street-parking? is turned on, it means that cars will park on another street if they don't find a spot along the block. They will park either to the left or the right of the block with their distance to the block being either avg-dist-left or avg-dist-right respectivally. When side-street-parking? is turned off, cars cannot park on side-streets and if they can't find a spot on the block are recorded as missed.

Vision refers to how many spots ahead a car can see. When "vision?" is turned on, they can only see either as far as the long-vision (20% of the cars) or the low-vision (80%). When "vision?" is off, cars can see the entire block.

time-in-# refers to how long customers stay in a given store on average which affects how long cars remain parked.

pop-# refers to the chance a car will be generated with passengers going to a given store. The exact probability for each store is pop-# divided by the number of stores.

The model ends after 10,000 ticks because an equilibrium has generally been reached before that point.

The two plots report the average distances that cars are parking from each store as well as the proportion of cars headed to each store that are unable to find spots.

THINGS TO NOTICE
----------------
When vision? is on and side-street-parking? is off, the store in the middle does the best and the store at the end does the worst. However, when side-street-parking? is turned back on, the store at the end of the block does the best. Why might that be? 

As well, When vision? is on and side-street-parking? is off, most of the parking occurs near the first and second store, but the stores in the middle do the best, why is that?

What do cars do when they cannot see very far ahead and there is no side--street-parking and what does this create?

THINGS TO TRY
-------------
Changing the variables to create different combinations. Sometimes if you change a variable in a store beween the edge and the middle, the biggest effect occurs to its neigbor on the edge, rather than itself.

EXTENDING THE MODEL
-------------------
An easy extension of this model would be to add code to compute the combined average distance for the stores and run a BehaviorSpace analysis from a societal or landlord perspective.

Harder extensions would be to expand the view to an entire block or multiple blocks in order to actually model cars parking on other streets, or have the number of cars being created become time dependent, to simulate fluctuations in the number of shoppers based on the time of day.

NETLOGO FEATURES
----------------
This model uses global variables as a means of passing information from one type of agent to another. 

RELATED MODELS
--------------
Traffic Basic, Traffic Grid, El Farol

CREDITS AND REFERENCES
----------------------
Author: David Price
Model created as final project for EECS 372 in Spring 2011, at Northwestern University
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
0
Rectangle -7500403 true true 151 225 180 285
Rectangle -7500403 true true 47 225 75 285
Rectangle -7500403 true true 15 75 210 225
Circle -7500403 true true 135 75 150
Circle -16777216 true false 165 76 116

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 4.1.3
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 1.0 0.0
0.0 1 1.0 0.0
0.2 0 1.0 0.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
