patches-own [water]
globals [people-constant water-constant]


to setup
  clear-all
  draw-terrain 

  crt population
  [
    set color orange
    set size 2
    set shape "person"
    
    ; distribute randomly, making sure none of them are in the water
    setxy random-xcor random-ycor
    while [water = true] [ setxy random-xcor random-ycor ]
  ]
  
  set people-constant 0.0
  let counter 1
  while [counter <= population]
  [
    set people-constant (people-constant + (1.0 / counter))
    set counter (counter + 1)
  ]
  
  ; we don't want to count ourselves
  set people-constant (people-constant - 1)

  reset-ticks
end


;; draw a fixed terrain
to draw-terrain
  ask patches
  [
    set pcolor 72  ; green
    set water false
  ]
  
  ; ocean along the top
  ask patches with [pycor > 70]
  [
    set pcolor blue
    set water true
  ]
  
  ; ocean along the bottom
  ask patches with [pycor < 3]
  [
    set pcolor blue
    set water true
  ]
  
  ; lake on the left
  ask patches with [distancexy 25 30 < 13]
  [
    set pcolor blue
    set water true
  ]
    
  ; another lake on the right
  ask patches with [distancexy 70 45 < 10]
  [
    set pcolor blue
    set water true
  ]
  
  ; river going down from the second lake  
  ask patches with [pxcor > 67 and pxcor < 73 and pycor < 45]
  [
    set pcolor blue
    set water true
  ]
  
end



to go
  
  ask turtles
  [

    ;; the default probability of staying at the current
    ;; patch derives from laziness
    let p ((laziness / 100.0) + (get-happiness-adjustment xcor ycor))

    let r (random-float 1)
    ifelse (r > p)
    [
      move-intelligently
    ] [
      set r (random 100)
      if (impulsiveness > r) [ move-randomly ]
    ]
    
  ]
  
  tick
end



;; calculate the people index, which we model to have decreasing
;; marginal utility. we go with a harmonic sum as a rough
;; approximation, i.e. the nth neighbor adds value 1/n
to-report get-people-index [x y]

  let people-index 0.0
  let people-count 0

  ask turtles with [(distancexy x y) < sphere-of-activity]
  [
    set people-count (people-count + 1)
    set people-index people-index + (1.0 / people-count)
  ]
  
  ;; we counted ourselves in the above, so take it back out
  set people-index (people-index - 1)
  
  ;; normalize the people-index against the maximum possible value
  ;; that occurs when everyone is all in the same place. after
  ;; this, people-index will be a number between 0 and 1.
  set people-index (people-index / people-constant)
  
  report people-index
end


;; calculate the water index, which we model to have a sharply
;; decreasing marginal utility. we go with the nth neighboring
;; water patch adding value 1/n^2
to-report get-water-index [x y]
  let water-index 0.0
  let water-count 0
  ask patches with [(distancexy x y) < sphere-of-activity]
  [
    if (water = true)
    [
      set water-count (water-count + 1)
      set water-index water-index + (1.0 / (water-count * water-count))
    ]
  ]
  
  ;; normalize the water-index against the maximum possible value
  ;; after this, water-index will be a number between 0 and 1
  set water-index (water-index / 1.65)

  report water-index  
end


;; reports the probability adjustment of staying, as influenced
;; by being close to water and people.
to-report get-happiness-adjustment [x y]
  let people-index (get-people-index x y)
  let water-index (get-water-index x y)
    
  ;; we define leeway to be the amount of influence either proximity
  ;; to water or proximity to people can have, and we give these
  ;; equal parts of the remaining probability. for example,
  ;; if the default probability of staying is 40%, then we allow
  ;; each of these two factors to increase the probability by up to 
  ;; 30%, so that if both of them were to be maxed out, and the person
  ;; had 100% affinity to both, then the probability
  ;; of staying would be 100%.
  let leeway ((100.0 - laziness) / 200.0)

  report (people-index * (affinity-to-people / 100.0) * leeway) + (water-index * (affinity-to-water / 100.0) * leeway)  
end


;; checks the up/down/left/right squares and moves to the
;; one that makes you the happiest
to move-intelligently
  let x xcor
  let y ycor
  
  ; adjustments in the four directions
  let adj1 -1
  let adj2 -1
  let adj3 -1
  let adj4 -1
  
  
  ;; up, idx = 1
  set ycor (y + 1)
  if (water = false) [ set adj1 (get-happiness-adjustment xcor ycor) ]
  
  ;; down, idx = 2
  set ycor (y - 1)
  if (water = false) [ set adj2 (get-happiness-adjustment xcor ycor) ]
    
  ;; left, idx = 3
  set ycor y
  set xcor (x - 1)
  if (water = false) [ set adj3 (get-happiness-adjustment xcor ycor) ]
  
  ;; left, idx = 4
  set xcor (x + 1)
  if (water = false) [ set adj4 (get-happiness-adjustment xcor ycor) ]


  ;; figure out what the max adjustment is, and how many
  ;; of the four directions tie for max
  let max-adj adj1
  let max-count 1
  
  if (adj2 = adj1) [ set max-count (max-count + 1) ]
  if (adj2 > adj1) [ set max-adj adj2 set max-count 1 ]
  
  if (adj3 = adj2) [ set max-count (max-count + 1) ]
  if (adj3 > adj2) [ set max-adj adj3 set max-count 1 ]
  
  if (adj4 = adj3) [ set max-count (max-count + 1) ]
  if (adj4 > adj3) [ set max-adj adj4 set max-count 1 ]
  
  ;; now, randomly choose one of the directions that tie for
  ;; the max adjustment
  let max-idx (random max-count)
  
  if (adj1 = max-adj)
  [
    if (max-idx = 0) [ set xcor x set ycor (y + 1) stop ]
    set max-idx (max-idx - 1)
  ]
  
  if (adj2 = max-adj)
  [
    if (max-idx = 0) [ set xcor x set ycor (y - 1) stop ]
    set max-idx (max-idx - 1)
  ]  
  
  if (adj3 = max-adj)
  [
    if (max-idx = 0) [ set xcor (x - 1) set ycor y stop ]
    set max-idx (max-idx - 1)
  ]  
     
  if (adj4 = max-adj)
  [
    if (max-idx = 0) [ set xcor (x + 1) set ycor y stop ]
    set max-idx (max-idx - 1)
  ]  
      
end


to move-randomly
  let legal-move false
  let r 0
  
  while [legal-move = false]
  [
    set r (random 360)
    rt r fd 1
    ifelse (water = true)
    [
      set legal-move false
      bk 1 lt r
    ] [
      set legal-move true
    ]
  ]
end
@#$#@#$#@
GRAPHICS-WINDOW
210
10
1028
649
-1
-1
8.0
1
10
1
1
1
0
1
1
1
0
100
0
75
0
0
1
ticks
30.0

SLIDER
20
25
192
58
population
population
0
1000
395
1
1
NIL
HORIZONTAL

BUTTON
44
92
110
125
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
121
92
184
125
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
17
197
193
230
affinity-to-water
affinity-to-water
0
100
100
1
1
NIL
HORIZONTAL

SLIDER
17
263
193
296
affinity-to-people
affinity-to-people
0
100
31
1
1
NIL
HORIZONTAL

SLIDER
16
347
188
380
sphere-of-activity
sphere-of-activity
1
10
10
1
1
NIL
HORIZONTAL

SLIDER
18
406
190
439
impulsiveness
impulsiveness
0
100
6
1
1
NIL
HORIZONTAL

SLIDER
19
454
191
487
laziness
laziness
0
100
6
1
1
NIL
HORIZONTAL

@#$#@#$#@
## WHAT IS IT?

This model is designed to show how populations shift by highlighting the factors influencing people's decisions to move or not move and the effect that these decisions have on the aggregate arrangement of the population. 

## HOW IT WORKS

The rules governing agent behavior in the model are:

1) There is a maximum distance that people can reasonably travel to access water or other people.
2) People like to be near other people.
3) People like to be near water.
4) People can be lazy.
5) People can be spontaneous or illogical.   

## HOW TO USE IT

The relative weight in the decision making process given to the considerations listed above can be regulated through the use of sliders. 

Sphere-of-activity is the radius of patches with which the agents are able to interact, i.e. water or other people within this many patches of them are registered. All people within this radius are treated the same (i.e. once you're within this radius, it doesn't matter if you're right near a person or if he/she is on the edge of the radius).

The affinity-to-water and affinity-to-people sliders indicate how much the person cares about being near water or being near people (%). Somebody with a value of 0 is indifferent to water or people, whereas someone with a value of 100 for both cares deeply about each. We model affinity-to-people as a decreasing function, and specifically assign the utility value 1/n to the nth neighbor within the sphere-of-activity. We model affinity-to-water as a very sharply decreasing function, and specifically assign the utility value 1/(n^2) to the nth patch within the sphere-of-activity that has water.

Impulsiveness is how likely people are (%) to just move randomly for no reason, even if they are happy where they are. When they do move, they truly move randomly (1 unit in a random direction). 

Laziness indicates the propensity for a person not to move at all. A very lazy person will not move even if they are unhappy where they are.

Finally, the population slider simply allows the user to regulate the total population of the model (between 0 and 1000).

## THINGS TO NOTICE

When you press the SETUP button, the model creates a landscape with water features and randomly distibutes the number of people indicated on the POPULATION slider only on the land portions of the landscape. Then at each tick, every person will make a decision to move or not move one unit in a direction that will make them the happiest, according to the settings on the AFFINITY-TO-WATER, AFFINITY-TO-PEOPLE, SPHERE-OF-ACTIVITY, IMPULSIVENESS, and LAZINESS sliders.

For example, if AFFINITY-TO-WATER is set higher than AFFINITY-TO-PEOPLE, IMPULSIVENESS and LAZINESS, then people will be more likely to move toward water than they will be to move toward other people, to move in a random direction or to stay where they are. If water is already within their SPHERE-OF-ACTIVITY, the radius of which gets larger as you increase the slider, then they will stay where they are. 

Similarly, if AFFINITY-TO-PEOPLE is set higher than the other sliders, people will move toward other people. 

If IMPULSIVENESS is the highest setting, people will be most likely to move one unit in a random direction at each tick. 

If LAZINESS is the highest setting, people will be most likely to stay where they are. 

The smaller the SPHERE-OF-ACTIVITY, the closer people have to be to water or other people to make them want to stay where they are.

## THINGS TO TRY

To model a population comprised of people who are driven first by the need for water and second by the need to be near other people, who have only low levels of impulsivity and laziness, and who, once settled are not willing/able to travel far to access water or other people, you can set the sliders to: POPULATION = 395, AFFINITY-TO-WATER = 100, AFFINTY-TO-PEOPLE = 31, SPHERE-OF-ACTIVITY = 1, IMPULSIVENESS = 6, LAZINESS = 6. With these settings, people will gradually move toward other people and toward the water, forming tight-knit clumps scattered across the landscape. As the model progresses, more and more of these clumps of people will make their way toward water. 

However, you increase the SPHERE-OF-ACTIVITY to 10, indicating that people are willing/able to travel farther to access people and water, you will see a much more rapid movement of people to form loose settlements around the water supplies. As the model progresses, these settlements will become increasingly close-knit and remain adjacent to water supplies.

By setting IMPULSIVENESS and LAZINESS to 100, you will see that rather than congregating near other people or near water, people remain scattered randomly across the landscape.


## EXTENDING THE MODEL

Right now, the way this model works, additional neighbors make agents happier, albeit in a decreasing fashion. One way to extend the model would be to change the code so that after a certain point, each additional neighbor actually made agents LESS happy.

Another possible extension would be to add reproduction to the model.

A third possible extension would be to have agents change colors to indicate their level of happiness. 

## NETLOGO FEATURES

This model uses patches as well as affinity to other turtles to influence turtle behavior. It models affinity-to-people as a decreasing function, and specifically assigns the utility value 1/n to the nth neighbor within the sphere-of-activity. It also models affinity-to-water as a very sharply decreasing function, and specifically assigns the utility value 1/(n^2) to the nth patch within the sphere-of-activity that has water.

## RELATED MODELS

This model is related to the models in the "Urban Suite" which also model population movement. It is particularly related to "Urban Suite - Path Dependence," which models how firms search for places to locate. In that model, firms choose locations by taking into account the quality of their current location and the number of firms already located there. However, unlike, the the models in "Urban Suite - Path Dependence" model, this model focuses on individual people's decisions to settle, and these people make descisions as to where to move based on the affordances and limitations of the natural landscape and the need for personal contact. 

Our model is also similar to the "Segregation" model in NetLogo, as this model also uses proximity to other agents as a criteria for happiness. However, unlike in our model, in the "Segregation" model, the color, as well as the proximity of other agents influences agent happiness.


## CREDITS AND REFERENCES
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
0
Rectangle -7500403 true true 151 225 180 285
Rectangle -7500403 true true 47 225 75 285
Rectangle -7500403 true true 15 75 210 225
Circle -7500403 true true 135 75 150
Circle -16777216 true false 165 76 116

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.0RC7
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 1.0 0.0
0.0 1 1.0 0.0
0.2 0 1.0 0.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
