; INDISIM-Denitrification - NetLogo Implementation
; Araujo-Granda P.A., Gras A., Ginovart M. and Moulton V. - 2019

;______________________________ Global Variables __________________________________________________________________________________
extensions [matrix]

globals
[
  world                                                                      ; culture medium continuos size [ litres ]
  steptime                                                                   ; step time [h]
  time_now                                                                   ; simulated time [hours]
  gbacteria gdioxide gbicarbonate gn2o gno2- gno gn2 gsuccinate gamonium gnitrate goxygen ; global variables for graphical and numerical outputs
  output_file                                                               ; global variables for outputs files
  DST DSTb DSTr DSTm                                                        ; Variables for General Standard Deviation
  succinatemedium amoniummedium oxygenmedium nitratemedium nitritemedium monoxidemedium dinitrogenmedium nitrogenmedium dioxidemedium bicarbonatemedium
  Suc Amo Oxy Nrt Nit N2O
  succinate-input amonium-input nitrate-input
  xsc xam xox xntr xnr xmn xdn xn2 xco2 xhco3 bacteriaout xbacteria
  d
  untri
  isuccinate iamonium ioxygen initrate initrite imonoxide idinitrogen initrogen idioxide ibicarbonate ibacteria icbacteria inbacteria
  fsuccinate famonium foxygen fnitrate fnitrite fmonoxide fdinitrogen fnitrogen fdioxide fbicarbonate fbacteria fcbacteria fnbacteria
  insuccinate inamonium innitrate
  outsuccinate outamonium outoxygen outnitrate outnitrite outmonoxide outdinitrogen outnitrogen outdioxide outbicarbonate outbacteria outcbacteria outnbacteria
  c-bacteria n-bacteria c-succinate n-amonium n-nitrate n-nitrite n-monoxide n-dinitrogen n-nitrogen c-co2 c-hco3
  n-obtained n-delivered c-obtained c-delivered
  %_e_c %_e_n yc/c
  scm oxm ntrm
  q
  sc-mant2 sc-mant3 sc-mant4
  ini_biomass rep_biomass fmol_small fmol_big deq_small deq_big

  molecular-weight e-eeq/mol dGpc
  A1 fs1 fe1 yc/c1
  A2 fs2 fe2 yc/c2
  A3 fs3 fe3 yc/c3
  A4 fs4 fe4 yc/c4
  A5 fs5 fe5 yc/c5
  A6 fs6 fe6 yc/c6
  donor acceptor1 acceptor2 acceptor3 acceptor4 acceptor5 acceptor6 b-biomass
  pathway1 pathway2 pathway3 pathway4 pathway5 pathway6
  auto_consum_NO auto_consum_N2O auto_consum_NO2-
  rx_mant-aero rx_mant_anox1 rx_mant_anox2 rx_mant_anox3 rx_mant_anox4
  gamma_cells
  u_succ u_nh4+ u_no3-a u_no3-x u_no2- u_no u_n2o u_o2
  u u0 u1 u3 umax

  ; for Mahcine learning matrix procedure

  biomass_matrix GRI_biomass
  nitrate_matrix GRI_nitrate
  nitrite_matrix GRI_nitrite
  nitrous_oxide_matrix GRI_nitrous_oxide
  Fitness

  ]

; ____________________________ Microorganism variables ____________________________________________________________________________

breed [bacteria bacterium]                                                   ; microorganism definition
bacteria-own
[
  biomass                                                                    ; microorganism biomass [ mmol ]
  biomass-reproduction                                                       ; reproduction treshold biomass [ mmol ]
  ac_mn ac_dn
  R                                                                          ; reproductions counter
  m_aero s_aero_1 s_aero_2                                                   ; aerobic phase counter
  m_anox_1 m_anox_2 m_anox_3 m_anox_4 s_anox_1 s_anox_2 s_anox_3 s_anox_4    ; anaerobic reactions counters
  sc-useful am-useful ox-useful ntr-useful nr-useful mn-useful dn-useful     ; usefuls nutrients variables
  bm
]

; ____________________________ Culture Medium variables ____________________________________________________________________________

patches-own
  [
  succinate-medium                     ; culture medium variable for the electron donor and C-Source in aerobic and anaerobic phases [ mmol ]
  amonium-medium                       ; culture medium variable for the N-Source in aerobic and anaerobic phases [ mmol ]
  oxygen-medium                        ; culture medium variable for the electron aceptor in aerobic phase [ mmol ]
  nitrate-medium                       ; culture medium variable for the electron aceptor in anoxic reaction 1 [ mmol ]
  nitrite-medium                       ; culture medium variable for the electron aceptor in anoxic reaction 2 [ mmol ]
  monoxide-medium                      ; culture medium variable for the electron aceptor in anoxic reaction 3 [ mmol ]
  dinitrogen-medium                    ; culture medium variable for the electron aceptor in anoxic reaction 4 [ mmol ]
  dioxide nitrogen bicarbonate         ; culture medium variables for the denitrification products [ mmol ]
  ]

; ____________________________ Virtual Bioreactor Setting ____________________________________________________________________________

to setup
  clear-all                      ; clear all variables, graphical and numerical outputs
  reset-ticks                    ; reset tick counter
  setup-thermodynamics           ; setup thermodynamic coefficients accord to McCarty 2007, TEEM2
  setup-model                    ; setup general model parameters to obtain the starting points accord to Felgate's (2012) experiment
  setup-time                     ; scaled time setting
  setup-medium                   ; formulation of the initial culture medium
  setup-bacteria                 ; formulation of initial bacterial inoculum
  setup-monitors                 ; monitors setting
  if Output_file? = true
  [
  setup-output                   ; setup output file
  write-outputfile               ; write initial values on output file
  ]
  setup-balance                  ; mass balance setup
  set u0 gbacteria
  set u3 0
  setup-machine.learning
end
;___________________________________ Initial data according to Felgate's Experiments _____________________________________________________

to setup-model
  set world (world-width * world-height)                                                  ; Differential adjustment of reactor volume
  set DSTb 0.15                                                                           ; Standard Deviation for initial biomass distribution
  set DSTr 0.15                                                                           ; Standard Deviation for biomass reproduction
  set DST 0.05                                                                            ; Standard Deviation for general model varaibles
  set DSTm 0.25                                                                           ; Strandard Deviation for culture medium availability
  set nsc 0.28 set nam 0.84 set nx 0.79 set nntr 0.63 set nnr 0.77 set nmn 1 set ndn 0.49 ; Setup of culture medium nutrients availability obtanied from Dab

  if Denitrifying_bacterium = "P._denitrificans"
  [
  set deq_small abs random-normal (0.5) (DST * 0.5)                                       ; define the minimun bacterium size Bergey's reference taking into account the spherical shape
  set deq_big abs random-normal (0.9) (DST * 0.9)                                         ; define the maximun bacterium size Bergey's reference taking into account the spherical shape
  set fmol_small (((deq_small * 1e-4) ^ 3) * (pi / 6)) * 1.1 * 1e15 / molecular-weight    ; small bacteria expresed in fmol
  set fmol_big (((deq_big * 1e-4) ^ 3) * (pi / 6)) * 1.1 * 1e15 / molecular-weight        ; big bacteria expresed in fmol
  set ini_biomass 0.75 * fmol_big                                                         ; [fmol] initial biomass for each bacterium
  set rep_biomass 0.75 * fmol_big                                                         ; [fmol] reproduction treshold biomass for each bacterium

  if Experiment = "Succ_High/Nitrate_Low"                                                   ; setup for experiment C-source sufficient / electron-acceptor limited
  [
  Set Succinate 20 Set Nitrate 4.99825                                                    ; Succiante and Nitrate setup accord to Felgate's (2012) experiment (Succinate High/Nitrate low)
  let paracoccus abs random-normal (0.0033683) (DSTr * 0.0033683)                         ; Initial biomass concentration accord to Felgate's (2012) experiment (succinate High/Nitrate low)
  set Microrganism round ((paracoccus * 1000 * world) / (molecular-weight * ini_biomass)) ; Individual's initial number accord to Felgate's (2012) experiment.
  ]
  if Experiment = "Succ_Low/Nitrate_High"                                                 ; setup for experiment C-source limited / electron-accpetor suficient
  [
  Set Succinate 5 set Nitrate 21.60951                                                    ; Succiante and Nitrate setup accord to Felgate's (2012) experiment (Succinate Low/Nitrate high)
  let paracoccus abs random-normal (0.0668478) (DSTr * 0.0668478)                         ; Initial biomass concentration accord to Felgate's (2012) experiment (succinate low/Nitrate high)
  set Microrganism round ((paracoccus * 1000 * world) / (molecular-weight * ini_biomass)) ; Individual's initial number accord to Felgate's (2012) experiment.
  ]
  ]

  if Denitrifying_bacterium = "A._xylosoxidans"
  [
  set deq_small abs random-normal (0.63) (DST * 0.63)                                        ; define the minimun bacterium size Bergey's reference taking into account the spherical shape
  set deq_big abs random-normal (1.40) (DST * 1.40)                                          ; define the maximun bacterium size Bergey's reference taking into account the spherical shape
  set fmol_small (((deq_small * 1e-4) ^ 3) * (pi / 6)) * 1.1 * 1e15 / molecular-weight       ; small bacteria expresed in fmol
  set fmol_big (((deq_big * 1e-4) ^ 3) * (pi / 6)) * 1.1 * 1e15 / molecular-weight           ; big bacteria expresed in fmol
  set ini_biomass 0.75 * fmol_big                                                            ; [fmol] initial biomass for each bacterium
  set rep_biomass 0.75 * fmol_big                                                            ; [fmol] reproduction treshold biomass for each bacteirum

  if Experiment = "Succ_High/Nitrate_Low"                                                    ; setup for experiment C-source sufficient / electron-acceptor limited
  [
  Set Succinate 20 Set Nitrate 5.15375                                                       ; Succiante and Nitrate setup accord to Felgate's (2012) experiment (Succinate High/Nitrate low)
  let achromobacter abs random-normal (0.07695) (DSTr * 0.07695)                             ; Initial biomass concentration accord to Felgate's (2012) experiment (succinate High/Nitrate low)
  set Microrganism round ((achromobacter * 1000 * world) / (molecular-weight * ini_biomass)) ; Individual's initial number accord to Felgate's (2012) experiment.
  ]
  if Experiment = "Succ_Low/Nitrate_High"                                                    ; setup for experiment C-source limited / electron-accpetor suficient
  [
  Set Succinate 5 set Nitrate 21.74694                                                       ; Succiante and Nitrate setup accord to Felgate's (2012) experiment (Succinate Low/Nitrate high)
  let achromobacter abs random-normal (0.05182) (DSTr * 0.05182)                             ; Initial biomass concentration accord to Felgate's (2012) experiment (succinate low/Nitrate high)
  set Microrganism round ((achromobacter * 1000 * world) / (molecular-weight * ini_biomass)) ; Individual's initial number accord to Felgate's (2012) experiment.
  ]
  ]
end

; ___________________________________ Proceeding to Scaled time ____________________________________________________________________________

to setup-time
  set steptime (Min/steptime / 60)       ; User define the number of minute for each step time using a slider labeled "Min_per_steptime"
end

; ___________________________________ Proceeding to Formulation of Initial culture medium __________________________________________________

to setup-medium
    if Denitrifying_bacterium = "P._denitrificans"
    [
     if Experiment = "Succ_High/Nitrate_Low"
       [
       set Suc random-normal (Succinate) (DST * Succinate)           ; Succinate initial concentration (user define using a slider labeled Succinate) [mM]
       set Amo random-normal (Amonium) (DST * Amonium)               ; Amonium initial concentration (user define using a slider labeled Amonium) [mM]
       set Oxy random-normal (0.2360) (DST * 0.2360)                 ; Oxygen initial concentration (simulated saturated conditions) accord to Felgate's (2012) experiment [mM]
       set Nrt random-normal (Nitrate) (DST * Nitrate)               ; Nitrate initial concentration (user define using a slider labeled Nitrate) [mM]
       set Nit random-normal (0.0255) (DST * 0.0255)                 ; Nitrite initial concentration accord to Felgate's (2012) experiment [mM]
       set N2O random-normal (0.003003178) (DST * 0.003003178)       ; Nitrous oxide initial concentration accord to Felgate's (2012) experiment [mM]
       ]
     if Experiment = "Succ_Low/Nitrate_High"
       [
       set Suc random-normal (Succinate) (DST * Succinate)           ; Succinate initial concentration (user define using a slider labeled Succinate) [mM]
       set Amo random-normal (Amonium) (DST * Amonium)               ; Amonium initial concentration (user define using a slider labeled Amonium) [mM]
       set Oxy random-normal (0.2360) (DST * 0.2360)                 ; Oxygen initial concentration (simulated saturated conditions) accord to Felgate's (2012) experiment [mM]
       set Nrt random-normal (Nitrate) (DST * Nitrate)               ; Nitrate initial concentration (user define using a slider labeled Nitrate) [mM]
       set Nit random-normal (0.011163522) (DST * 0.011163522)       ; Nitrite initial concentration accord to Felgate's (2012) experiment [mM]
       set N2O random-normal (0.000028413312) (DST * 0.000028413312) ; Nitrous oxide initial concentration accord to Felgate's (2012) experiment [mM]
       ]
     ]

    if Denitrifying_bacterium = "A._xylosoxidans"
    [
     if Experiment = "Succ_High/Nitrate_Low"
       [
       set Suc random-normal (Succinate) (DST * Succinate)           ; Succinate initial concentration (user define using a slider labeled Succinate) [mM]
       set Amo random-normal (Amonium) (DST * Amonium)               ; Amonium initial concentration (user define using a slider labeled Amonium) [mM]
       set Oxy random-normal (0.2360) (DST * 0.2360)                 ; Oxygen initial concentration (simulated saturated conditions) accord to Felgate's (2012) experiment [mM]
       set Nrt random-normal (Nitrate) (DST * Nitrate)               ; Nitrate initial concentration (user define using a slider labeled Nitrate) [mM]
       set Nit random-normal (0.36863) (DST * 0.36863)               ; Nitrite initial concentration accord to Felgate's (2012) experiment [mM]
       set N2O random-normal (0.00006263) (DST * 0.00006263)         ; Nitrous oxide initial concentration accord to Felgate's (2012) experiment [mM]
       ]
     if Experiment = "Succ_Low/Nitrate_High"
       [
       set Suc random-normal (Succinate) (DST * Succinate)           ; Succinate initial concentration (user define using a slider labeled Succinate) [mM]
       set Amo random-normal (Amonium) (DST * Amonium)               ; Amonium initial concentration (user define using a slider labeled Amonium) [mM]
       set Oxy random-normal (0.2360) (DST * 0.2360)                 ; Oxygen initial concentration (simulated saturated conditions) accord to Felgate's (2012) experiment [mM]
       set Nrt random-normal (Nitrate) (DST * Nitrate)               ; Nitrate initial concentration (user define using a slider labeled Nitrate) [mM]
       set Nit random-normal (0.00765) (DST * 0.00765)               ; Nitrite initial concentration accord to Felgate's (2012) experiment [mM]
       set N2O random-normal (0.00001818) (DST * 0.00001818)         ; Nitrous oxide initial concentration accord to Felgate's (2012) experiment [mM]
       ]
     ]

   ask patches
   [
     set succinate-medium abs random-normal (Suc) (DST * Suc)            ; Assign the succinate concentration to each cell according to the model's standar deviation
     set amonium-medium abs random-normal (Amo) (DST * Amo)              ; Assign the amonium concentration to each cell according to the model's standar deviation
     set oxygen-medium  abs random-normal (Oxy) (DST * Oxy)              ; Assign the oxygen concentration to each cell according to the model's standar deviation
     set nitrate-medium abs random-normal (Nrt) (DST * Nrt)              ; Assign the nitrate concentration to each cell according to the model's standar deviation
     set nitrite-medium abs random-normal (Nit) (DST * Nit)              ; Assign the nitrite concentration to each cell according to the model's standar deviation
     set dinitrogen-medium abs random-normal (N2O) (DST * N2O)           ; Assign the nitrous oxide concentration to each cell according to the model's standar deviation
     set monoxide-medium 0                                                ; Nitric Oxide initial concentration accord to Felgate's (2012) experiment [mM]
     set nitrogen 0                                                       ; Nitrogen gas (N2) initial concentration accord to Felgate's (2012) experiment [mM]
     set dioxide 0                                                        ; Carbon Dioxide initial concentration accord to Felgate's (2012) experiment [mM]
     set bicarbonate 0                                                    ; Bicarbonate initial concentration [mM]
     set pcolor scale-color blue succinate-medium 0 Succinate
   ]

end

; ____________________________ Proceeding to Formulation the Initial bacterial inoculum __________________________________________________

to setup-bacteria
 create-bacteria Microrganism                                                    ; User define the amount of initial active bacteria using a slider labaled Microrganism
 [
   if Denitrifying_bacterium = "A._xylosoxidans" [set shape "cylinder"]          ; Achrobobacter xylosoxidans is a rod bacteria
   if Denitrifying_bacterium = "P._denitrificans" [set shape "circle"]           ; Paracoccus denitrificans is a spherical bacteria
   set size 0.5                                                                  ; This size is for visual propouse only
   set color green                                                               ; This color is for visual propouse only
   setxy random-xcor random-ycor                                                 ; Random inicial position on culture medium
   set biomass abs random-normal (ini_biomass) (DSTb * ini_biomass)              ; Initial biomass for each microorganism [fmol] according to model's standar deviation
   set biomass-reproduction abs random-normal (rep_biomass) (DSTr * rep_biomass) ; reproduction treshold biomass [fmol] accord to INDISIM general procedures.
 ]
end

; ____________________________ Proceeding to monitors setting __________________________________________________________________________

to setup-monitors
  set time_now ticks * steptime                                                                          ; Simulated time [ h ]
  set gbacteria ((sum [biomass] of bacteria / world) / 1000) * molecular-weight                          ; Biomass concentration [ mg/ml ]
  set gsuccinate (sum[succinate-medium]of patches / world)                                               ; Succinate concentration [ mM ]
  set gamonium (sum [amonium-medium] of patches / world)                                                 ; Amonium concentration [ mM ]
  set goxygen (sum [oxygen-medium] of patches / world)                                                   ; Oxygen concentration [ mM ]
  set gnitrate (sum [nitrate-medium] of patches / world)                                                 ; Nitrate concentration [ mM ]
  set gno2- (sum [nitrite-medium] of patches / world)                                                    ; Nitrite concentration [ mM ]
  set gno (sum [monoxide-medium] of patches / world)                                                     ; Nitric Oxide concentration [ mM ]
  if Experiment = "Succ_Low/Nitrate_High"[set gn2o (sum [dinitrogen-medium] of patches / world) * 1000]  ; Nitrous Oxide concentration [ uM ]
  if Experiment = "Succ_High/Nitrate_Low"[set gn2o (sum [dinitrogen-medium] of patches / world)]         ; Nitrous Oxide concentration [ mM ]
  set gdioxide (sum [dioxide] of patches / world)                                                        ; Carbon dioxide concentration [ mM ]
  set gn2 (sum [nitrogen] of patches / world)                                                            ; Nitrogen concentration [ mM ]
  set gbicarbonate (sum [bicarbonate] of patches / world)                                                ; Bicarbonate concentration [ mM ]

  reactor_balance
end

; ____________________________ Proceeding to setup output file __________________________________________________________________________

to setup-output
  file-close
  if Experiment = "Succ_Low/Nitrate_High"
  [
  if Denitrifying_bacterium = "P._denitrificans" [set output_file (word (word behaviorspace-run-number"_"molecular-weight)"_PD_I_Denitrification_V10_ScLw_NtHg_" ".txt")] ; Outputfile name with .txt extension and Behaviorspace numering
  if Denitrifying_bacterium = "A._xylosoxidans" [set output_file (word (word behaviorspace-run-number"_"molecular-weight)"_AX_I_Denitrification_V10_ScLw_NtHg_" ".txt")] ; Outputfile name with .txt extension and Behaviorspace numering
  ]
  if Experiment = "Succ_High/Nitrate_Low"
  [
  if Denitrifying_bacterium = "P._denitrificans" [set output_file (word (word behaviorspace-run-number"_"molecular-weight)"_PD_I_Denitrification_V10_ScHg_NtLw_" ".txt")] ; Outputfile name with .txt extension and Behaviorspace numering
  if Denitrifying_bacterium = "A._xylosoxidans" [set output_file (word (word behaviorspace-run-number"_"molecular-weight)"_AX_I_Denitrification_V10_ScHg_NtLw_" ".txt")] ; Outputfile name with .txt extension and Behaviorspace numering
  ]
  carefully
  [file-delete output_file] [ ]
  file-open output_file
  file-type "Time_[h] "                                                  ; Simulation time labed in outputfile unit hours
  file-type "Succinate_[mM] "                                            ; Succinate concentration labed in outputfile unit milimolar
  file-type "NH4+_[mM] "                                                 ; Amonium concentration labed in outputfile unit milimolar
  file-type "O2_[mg/l] "                                                 ; Oxygen concentration labed in outputfile unit mg/ml
  file-type "NO3-_[mM] "                                                 ; Nitrate concentration labed in outputfile unit milimolar
  file-type "NO2-_[mM] "                                                 ; Nitrite concentration labed in outputfile unit milimolar
  file-type "NO_[mM] "                                                   ; Nitric Oxide concentration labed in outputfile unit milimolar
  if Experiment = "Succ_Low/Nitrate_High" [file-type "N2O_[uM] "]        ; Dinitrogen concentration labed in outputfile unit micromolar
  if Experiment = "Succ_High/Nitrate_Low" [file-type "N2O_[mM] "]        ; Dinitrogen concentration labed in outputfile unit milimolar
  file-type "N2_[mM] "                                                   ; Nitrogen concentration labed in outputfile unit milimolar
  file-type "CO2_[mM] "                                                  ; Carbon dioxide concentration labed in outputfile unit milimolar
  file-type "Biomass_[mg/ml] "                                           ; Biomass concentration labed in outputfile unit miligrams / mililiter
  file-type "Individuals "                                               ; Individuals into bioreactor labed in outputfile
  file-type "d "                                                         ; Reactor dilution ratio labed in outputfile units (h^-1)
  file-type "Sc_In_[mmol] "                                              ; Inflow Succinate labed in outputfile units milimoles
  file-type "NH4+_In_[mmol] "                                            ; Inflow Amonium labed in outputfile units milimoles
  file-type "NO3-_In_[mmol] "                                            ; Inflow Nitrate labed in outputfile units milimoles
  file-type "Sc_Out_[mmol] "                                             ; Outflow Succinate labed in outputfile units milimoles
  file-type "NH4+_Out_[mmol] "                                           ; Outflow amonium labed in outputfile units milimoles
  file-type "NO3-_Out_[mmol] "                                           ; Outflow  nitrate labed in outputfile units milimoles
  file-type "NO2-_Out_[mmol] "                                           ; Outflow nitrite labed in outputfile units milimoles
  file-type "NO_Out_[mmol] "                                             ; Outflow Nitric Oxide labed in outputfile units milimoles
  if Experiment = "Succ_Low/Nitrate_High" [file-type "N2O_Out_[umol] "]  ; Outflow Nitrous Oxide labed in outputfile units micromoles
  if Experiment = "Succ_High/Nitrate_Low" [file-type "N2O_Out_[mmol] "]  ; Outflow Nitrous oxide labed in outputfile units milimoles
  file-type "N2_Out_[mmol] "                                             ; Outflow Nitrogen gas labed in outputfile units milimoles
  file-type "Ind_Out "                                                   ; Outflow Microorganims labed in outputfile
  file-type "HCO3_[mM] "                                                 ; Bicarbonate concentration labed in outputfile units milimolar
  file-type "HCO3_Out_[mmol] "                                           ; Outflow Bicarbonate labed in outputfile units milimoles
  file-type "CO2_Out_[mmol] "                                            ; Outflow Carbon dioxide labed in outputfile units milimoles
  file-print " "
end

; ____________________________ Proceeding to write values on output file __________________________________________________________________________

to write-outputfile
  setup-monitors
  file-type time_now                                            ; Outputfile Simulated time [h]
  file-type " "
  file-type gsuccinate                                          ; Outputfile Bioreactor Succinate concentration [mM]
  file-type " "
  file-type gamonium                                            ; Outputfile Bioreactor Amonium concentration [mM]
  file-type " "
  file-type goxygen * 32                                        ; Outputfile Bioreactor Oxygen concentration [mg/l]
  file-type " "
  file-type gnitrate                                            ; Outputfile Bioreactor Nitrate concentration [mM]
  file-type " "
  file-type gno2-                                               ; Outputfile Bioreactor Nitrite concentration [mM]
  file-type " "
  file-type gno                                                 ; Outputfile Bioreactor Nitric Oxide concentration [mM]
  file-type " "
  file-type gn2o                                                ; Outputfile Bioreactor Nitrous Oxide concentration [mM][uM]
  file-type " "
  file-type gn2                                                 ; Outputfile Bioreactor Nitrogen gas concentration [mM]
  file-type " "
  file-type gdioxide                                            ; Outputfile Bioreactor Carbon Dioxide concentration [mM]
  file-type " "
  file-type gbacteria                                           ; Outputfile Bioreactor Biomass concentration [mg/ml]
  file-type " "
  file-type count bacteria                                      ; Outputfile Bioreactor microorganism count [individuals]
  file-type " "
  file-type d                                                   ; Outputfile Bioreactor Dilution ratio [h^1]
  file-type " "
  file-type succinate-input                                     ; Outputfile Inflow Succinate quantity [mmol]
  file-type " "
  file-type amonium-input                                       ; Outputfile Inflow Amonioum quantity [mmol]
  file-type " "
  file-type nitrate-input                                       ; Outputfile Inflow Nitrate quantity [mmol]
  file-type " "
  file-type xsc                                                 ; Outputfile Outflow Succinate quantity [mmol]
  file-type " "
  file-type xam                                                 ; Outputfile Outflow Amonium quantity [mmol]
  file-type " "
  file-type xntr                                                ; Outputfile Outflow Nitrate quantity [mmol]
  file-type " "
  file-type xnr                                                 ; Outputfile Outflow Nitrite quantity [mmol]
  file-type " "
  file-type xmn                                                 ; Outputfile Outflow Nitric oxide quantity [mmol]
  file-type " "
  if Experiment = "Succ_Low/Nitrate_High"[file-type xdn * 1000] ; Outputfile Outflow Nitrous oxide quantity [umol]
  if Experiment = "Succ_High/Nitrate_Low"[file-type xdn]        ; Outputfile Outflow Nitrous oxide quantity [mmol]
  file-type " "
  file-type xn2                                                 ; Outputfile Outflow Nitrogen gas quantity [mmol]
  file-type " "
  file-type bacteriaout                                         ; Outputfile Outflow microrganism quantity [individuals]
  file-type " "
  file-type gbicarbonate                                        ; Outputfile Bioreactor Bicarbonate concentration [mM]
  file-type " "
  file-type xhco3                                               ; Outputfile Outflow bicarbonate quantity [mmol]
  file-type " "
  file-type xco2                                                ; Outputfile Outflow Carbon Dioxide quantity [mmol]
  file-type " "
  file-print " "
end

; ____________________________ General mass balance proceedings _____________________________________________________________________________

to setup-balance
  set ibacteria (sum [biomass] of bacteria)                                                ; Initial Biomass [mmol]
  set isuccinate (sum[succinate-medium]of patches)                                         ; Initial Succinate [mmol]
  set iamonium (sum [amonium-medium] of patches)                                           ; Initial Amonium [mmol]
  set ioxygen (sum [oxygen-medium] of patches)                                             ; Initial Oxygen [mmol]
  set initrate (sum [nitrate-medium] of patches)                                           ; Initial Nitrate [mmol]
  set initrite (sum [nitrite-medium] of patches)                                           ; Initial Nitrite [mmol]
  set imonoxide (sum [monoxide-medium] of patches)                                         ; Initial Monoxide [mmol]
  set idinitrogen (sum [dinitrogen-medium] of patches)                                     ; Initial Dinitrogen [mmol]
  set idioxide (sum [dioxide] of patches)                                                  ; Initial Carbon dioxide [mmol]
  set initrogen (sum [nitrogen] of patches)                                                ; Initial Nitrogen [mmol]
  set ibicarbonate (sum [bicarbonate] of patches)                                          ; Initial Bicarbonate [mmol]
end

to reactor_balance
  set fbacteria (gbacteria * 1000 * world / molecular-weight)                              ; Final Biomass [mmol]
  set fsuccinate (gsuccinate * world)                                                      ; Final Succinate [mmol]
  set famonium (gamonium * world)                                                          ; Final Amonium [mmol]
  set foxygen (goxygen * world)                                                            ; Final Oxygen [mmol]
  set fnitrate (gnitrate * world)                                                          ; Final Nitrate [mmol]
  set fnitrite (gno2- * world)                                                             ; Final Nitrite [mmol]
  set fmonoxide (gno * world)                                                              ; Final Monoxide [mmol]
  if Experiment = "Succ_Low/Nitrate_High"[set fdinitrogen ((gn2o * world) / 1000)]         ; Final Dinitrogen [mmol]
  if Experiment = "Succ_High/Nitrate_Low"[set fdinitrogen (gn2o * world)]                  ; FInal Dinitrogen [mmol]
  set fdioxide (gdioxide * world)                                                          ; Final Carbon dioxide [mmol]
  set fnitrogen (gn2 * world)                                                              ; Final Nitrogen [mmol]
  set fbicarbonate (gbicarbonate * world)                                                  ; Final Bicarbonate [mmol]
end

to inflow_balance
  set insuccinate (insuccinate + succinate-input)                                          ; Inflow Succinate [mmol]
  set inamonium (inamonium + amonium-input)                                                ; Inflow Amonium [mmol]
  set innitrate (innitrate + nitrate-input)                                                ; Inflow Nitrate [mmol]
end

to outflow_balance
  set outsuccinate (outsuccinate + xsc)                                                    ; OutFlow Succinate [mmol]
  set outamonium (outamonium + xam)                                                        ; OutFlow Amonium [mmol]
  set outoxygen (outoxygen + xox)                                                          ; OutFlow Oxygen [mmol]
  set outnitrate (outnitrate + xntr)                                                       ; Outflow Nitrtate [mmol]
  set outnitrite (outnitrite + xnr)                                                        ; Outflow Nitrite [mmol]
  set outmonoxide (outmonoxide + xmn)                                                      ; Outflow Monoxide [mmol]
  set outdinitrogen (outdinitrogen + xdn)                                                  ; Outflow N2O [mmol]
  set outnitrogen (outnitrogen + xn2)                                                      ; Outflow N2 [mmol]
  set outdioxide (outdioxide + xco2)                                                       ; Outflow CO2 [mmol]
  set outbicarbonate (outbicarbonate + xhco3)                                              ; outflow HCO3 [mmol]
  set outbacteria (outbacteria + xbacteria)                                                ; OutFlow Baacteria [mmol]
end

to general_mass_balance
  set icbacteria (ibacteria * carbon * 12.011)                                             ; Initial C-Mic [mg]
  set inbacteria (ibacteria * b-nitrogen * 14)                                             ; Initial N-Mic [mg]
  set fcbacteria (fbacteria * carbon * 12.011)                                             ; Final C-Mic [mg]
  set fnbacteria (fbacteria * b-nitrogen * 14)                                             ; Final N-Mic [mg]
  set outcbacteria (outbacteria * carbon * 12.011)                                         ; OutFlow C-Mic [mg]
  set outnbacteria (outbacteria * b-nitrogen * 14)                                         ; OutFlow N-Mic [mg]

  set c-bacteria (fcbacteria - icbacteria) - (0 - outcbacteria)                            ; Microorganism Carbon balance
  set n-bacteria (fnbacteria - inbacteria) - (0 - outnbacteria)                            ; Microorganims Nitrogen balance
  set c-succinate ((fsuccinate - isuccinate) - (insuccinate - outsuccinate)) * 4 * 12.011  ; Succinate carbon balance
  set n-amonium ((famonium - iamonium) - (inamonium - outamonium)) * 1 * 14                ; Amonium nitrogen balance
  set n-nitrate ((fnitrate - initrate) - (innitrate - outnitrate)) * 1 * 14                ; Nitrate nitrogen balance
  set n-nitrite ((fnitrite - initrite) - (0 - outnitrite)) * 1 * 14                        ; Nitrite nitrogen balance
  set n-monoxide ((fmonoxide - imonoxide) - (0 - outmonoxide)) * 1 * 14                    ; Nitric oxide nitrogen balance
  set n-dinitrogen ((fdinitrogen - idinitrogen) - (0 - outdinitrogen)) * 2 * 14            ; Nitrous oxide nitrogen balance
  set n-nitrogen ((fnitrogen - initrogen) - (0 - outnitrogen)) * 2 * 14                    ; Nitrogen gas nitrogen balance
  set c-co2 ((fdioxide - idioxide) - (0 - outdioxide)) * 1 * 12.011                        ; Carbon dixoide carbon balance
  set c-hco3 ((fbicarbonate - ibicarbonate) - (0 - outbicarbonate)) * 1 * 12.011           ; Bicarbonate carbon balance

  set n-obtained abs (n-nitrite + n-monoxide + n-dinitrogen + n-nitrogen + n-bacteria)     ; Total nitrogen obtained
  set n-delivered abs (n-amonium + n-nitrate)                                              ; Total nitrogen delivered
  set c-obtained abs (c-co2 + c-hco3 + c-bacteria)                                         ; Total carbon obtained
  set c-delivered abs (c-succinate)                                                        ; Total carbon delivered

  set %_e_c abs ((c-delivered - c-obtained) / c-delivered ) * 100                          ; Carbon general mass balance porcentual error
  set %_e_n abs ((n-delivered - n-obtained) / n-delivered ) * 100                          ; Nitrogen general mass balance porcentual error
  set yc/c abs (c-bacteria / c-succinate)                                                  ; Population Yield growth
end

; ____________________________ Run Proceeding to Virtual Bioreactor __________________________________________________________________________

to go
  tick                                                                 ; Start tick counter
  stir                                                                 ; Observer procedure to stablish quantities for outflow in batch and continuos culture.
  outflow                                                              ; Observer and patch procedure to make outflow from bioreactor in continuos culture with or without culture medium homogenization
  inflow                                                               ; Observer procedure to simulate input nutrients to bioreactor according to dilution ratio with or without culture medium homogenization
  maintenance-requirements                                             ; Observer procedure to calculate maintenance requirements for aerobic and anaerobic phases.

  ask bacteria
  [
    uptake                                                             ; Agent procedure, each time step the bacteria establish their uptake capacities of all nutrients
    ifelse oxygen-medium >= abs random-normal NOX (DSTr * NOX)          ; Agent procedure, according to oxygen local concentration runs different metabolic pathways
    [mant-aerobic][mant-anoxic1]
    auto_consum
    metabolism_end                                                     ; Agent procedure, after cellular maintenance and biomass synthesis the uptaked nutrient quantities are updated
    bipartition                                                        ; Agent procedure, after execute metabolic pathways if the biomass reached the treshold valeu microorganism reproduce by bipartition
  ]

  ask bacteria [ move ]                                                ; Agent procedure, effect produced by the bioreactor agitation

  do-plotting                                                          ; Observer procedure, to setup graphical outputs
  if Output_file? = true
  [
;    if
;    time_now = 5 or
;    time_now = 6 or
;    time_now = 24 or
;    time_now = 24.5 or
;    time_now = 25 or
;    time_now = 30 or
;    time_now = 48 or
;    time_now = 54 or
;    time_now = 72 or
;    time_now = 78 or
;    time_now = 97 or
;    time_now = 99 or
;    time_now = 102 or
;    time_now = 120
;   [
      write-outputfile
      ]
 ; ]                            ; Observer procedure, write-out the all simulated data to output file.

  general_mass_balance
  to-umax

  get_data_machine.learning

  if time_now >= Simulation_Time [machine.learning file-close stop]                     ; Stop simulation if time is over and close output-file
  if (sum [succinate-medium] of patches / world) < 0 [machine.learning file-close stop] ; Stop simulation if the succinate global concentration is lower to zero and close output-file
  if (sum [amonium-medium] of patches / world) < 0 [machine.learning file-close stop]   ; Stop simulation if the amonium global concentration is lower to zero and close output-file
  if (sum [oxygen-medium] of patches / world) < 0 [machine.learning file-close stop]    ; Stop simulation if the oxygen global concentration is lower to zero and close output-file
  if (sum [nitrate-medium] of patches / world) < 0 [machine.learning file-close stop]   ; Stop simulation if the nitrate global concentration is lower to zero and close output-file
end

; ____________________________ Proceeding to simulate bioreactor agitation over the culture medium ____________________________________________________

to stir
  set succinatemedium (sum [succinate-medium] of patches)   ; Total Succinate in culture medium [mmol]
  set amoniummedium (sum [amonium-medium] of patches)       ; Total Amonium in culture medium [mmol]
  set oxygenmedium (sum [oxygen-medium] of patches)         ; Total Oxygen in culture medium [mmol]
  set nitratemedium (sum [nitrate-medium] of patches)       ; Total Nitrate in culture medium [mmol]
  set nitritemedium (sum [nitrite-medium] of patches)       ; Total Nitrite in culture medium [mmol]
  set monoxidemedium (sum [monoxide-medium] of patches)     ; Total Nitric oxide in culture medium [mmol]
  set dinitrogenmedium (sum [dinitrogen-medium] of patches) ; Total Nitrous oxide in culture medium [mmol]
  set nitrogenmedium (sum [nitrogen] of patches)            ; Total nitrogen gas in culture medium [mmol]
  set dioxidemedium (sum [dioxide] of patches)              ; Total carbon dioxide in culture medium [mmol]
  set bicarbonatemedium (sum [bicarbonate] of patches)      ; Total bicarbonate in culture medium [mmol]

  ifelse time_now >= Time_Shutdown_O2                       ; if simulated time is lower than time_shutdown_O2
  [set d abs random-normal (di) (DST * di)]                 ; Bioreactor works in continuos culture according to dilution ratio
  [
    set d 0                                                 ; the bioreactor works in batch conditions
    if oxygenmedium > 0.2360 * world
    [set oxygenmedium 0.2360 * world]                       ; and the oxygen keep saturated conditions
  ]
  set xsc d * succinatemedium * steptime                    ; succinate outflow according to dilution ratio [mmol]
  set xam d * amoniummedium * steptime                      ; amonium outflow according to dilution ratio [mmol]
  set xox d * oxygenmedium * steptime                       ; oxygen outflow according to dilution ration [mmol]
  set xntr d * nitratemedium * steptime                     ; nitrate outflow according to dilution ration [mmol]
  set xnr d * nitritemedium * steptime                      ; nitrite outflow according to dilution ration [mmol]
  set xmn d * monoxidemedium * steptime                     ; nitric oxide outflow according to dilution ration [mmol]
  set xdn d * dinitrogenmedium * steptime                   ; nitrous oxide outflow according to dilution ration [mmol]
  set xn2 d * nitrogenmedium * steptime                     ; nitrogen gas outflow according to dilution ration [mmol]
  set xco2 d * dioxidemedium * steptime                     ; carbon dioxide outflow according to dilution ration [mmol]
  set xhco3 d * bicarbonatemedium * steptime                ; bicarbonate outflow according to dilution ration [mmol]
  if xmn >= (3.27 * world) [set xmn (3.27 * world)]         ; if nitric oxide quantity is higher than maximun saturated conditions outflow only the saturated conditions
  if xdn >= (58.27 * world) [set xdn (58.27 * world)]       ; if nitrous oxide quantity is higher than maximun saturated conditions outflow only the saturated conditions
  if xn2 >= (104.91 * world) [set xn2 (104.91 * world)]     ; if nitrogen gas quantity is higher than maximun saturated conditions outflow only the saturated conditions
  if xco2 >= (8009.05 * world) [set xco2 (8009.05 * world)] ; if carbon dioxide quantity is higher than maximun saturated conditions outflow only the saturated conditions

  let bactout (sum [biomass] of bacteria)                   ; Total biomass in culture medium [mmol]
  let xbact (d * bactout * steptime)                        ; biomass outflow according to dilution ratio [mmol]
  let prom (count bacteria / bactout)                       ; average individual per biomass in outflow [individual ml /mg]
  set bacteriaout round (xbact * prom)                      ; Microorganism quantity to outflow [individuals]
  let monp (bactout / count bacteria)                       ; average biomass per individual in outflow [mg / ml individual]
  set xbacteria (monp * bacteriaout)                        ; Microorganism biomass concentration to outflow [mmol]

  outflow_balance
end

; ____________________________ Proceeding to simulate reactor OutFlow ________________________________________________________________________________________

to outflow
  set q random 2
  ifelse q = 0 [
   ask patches [
    set succinate-medium (succinatemedium - xsc) / world      ; The succinate outflow quantity is taken from global quantity and then homogenization occurs.
    set amonium-medium (amoniummedium - xam) / world          ; The amonium outflow quantity is taken from global quantity and then homogenization occurs.
    set oxygen-medium (oxygenmedium) / world                  ; Agitation and homogenization of oxygen
    set nitrate-medium (nitratemedium - xntr) / world         ; The nitrate outflow quantity is taken from global quantity and then homogenization occurs.
    set nitrite-medium (nitritemedium - xnr) / world          ; The nitrite outflow quantity is taken from global quantity and then homogenization occurs.
    set monoxide-medium (monoxidemedium - xmn) / world        ; The nitric oxide outflow quantity is taken from global quantity and then homogenization occurs.
    set dinitrogen-medium (dinitrogenmedium - xdn) / world    ; The nitrous oxide outflow quantity is taken from global quantity and then homogenization occurs.
    set nitrogen (nitrogenmedium - xn2) / world               ; The nitrogen gas outflow quantity is taken from global quantity and then homogenization occurs.
    set dioxide (dioxidemedium - xco2 ) / world               ; The carbon dioxide outflow quantity is taken from global quantity and then homogenization occurs.
    set bicarbonate (bicarbonatemedium - xhco3) / world       ; The bicarbonate outflow quantity is taken from global quantity and then homogenization occurs.
    set pcolor scale-color blue succinate-medium 0 Succinate
    ]
  ]
  [
    ask patches [
    set succinate-medium succinate-medium - (xsc / world)     ; The succinate outflow concentration is taken from local quantity without homogenization.
    set amonium-medium amonium-medium - (xam / world)         ; The amonium outflow concentration is taken from local quantity without homogenization.
    set oxygen-medium (oxygenmedium) / world                  ; Agitation and homogenization of oxygen
    set nitrate-medium nitrate-medium - (xntr / world)        ; The nitrate outflow concentration is taken from local quantity without homogenization.
    set nitrite-medium nitrite-medium - (xnr / world)         ; The nitrite outflow concentration is taken from local quantity without homogenization.
    set monoxide-medium monoxide-medium - (xmn / world)       ; The nitric oxide outflow concentration is taken from local quantity without homogenization.
    set dinitrogen-medium dinitrogen-medium - (xdn / world)   ; The nitrous oxide outflow concentration is taken from local quantity without homogenization.
    set nitrogen nitrogen - (xn2 / world)                     ; The nitrogen gas outflow concentration is taken from local quantity without homogenization.
    set dioxide dioxide - (xco2 / world)                      ; The carbon dioxide outflow concentration is taken from local quantity without homogenization.
    set bicarbonate bicarbonate - (xhco3 / world)             ; The bicarbonate outflow concentration is taken from local quantity without homogenization.
    set pcolor scale-color blue succinate-medium 0 Succinate
    ]
  ]
  ask n-of bacteriaout bacteria [ die ]                       ; Choose random bacteria and outflow accord to reactor dilution ratio
end

; __________________________________ Proceeding to simulate reactor Inflow ____________________________________________________________________________________

to inflow
  set succinate-input d * world * (isuccinate / world) * steptime                        ; Succinate inflow to continuos culture according to the slider labed succinate [mmol]
  set amonium-input d * world * (iamonium / world) * steptime                            ; Amonium inflow to continuos culture according to the slider labed amonium [mmol]
  set nitrate-input d * world * (initrate / world) * steptime                            ; Nitrate inflow to continuos culture according to the slider labed nitrate [mmol]

  set succinatemedium (sum [succinate-medium] of patches) + (succinate-input) ; Total Succinate in culture medium after inflow [mmol]
  set amoniummedium (sum [amonium-medium] of patches) + (amonium-input)       ; Total Amonium in culture medium after inflow [mmol]
  set nitratemedium (sum [nitrate-medium] of patches) + (nitrate-input)       ; Total Nitrate in culture medium after inflow [mmol]
  if time_now <= Time_Shutdown_O2 [set oxygenmedium 0.2360 * world]

  inflow_balance

  ifelse q = 0
  [ ask patches
   [ set succinate-medium succinatemedium / world                                ; The succinate inflow quantity is added to global quantity and then homogenization occurs.
     set amonium-medium amoniummedium / world                                    ; The amonium inflow quantity is added to global quantity and then homogenization occurs.
     set oxygen-medium oxygenmedium / world                                      ; The oxygen inflow quantity is added to global quantity and then homogenization occurs.
     set nitrate-medium nitratemedium / world                                    ; The nitrate inflow quantity is added to global quantity and then homogenization occurs.
     set pcolor scale-color blue succinate-medium 0 Succinate
   ]
  ]
  [ ask patches
   [ set succinate-medium succinate-medium + (succinate-input / world)           ; The succinate inflow concentration is added to local quantity without homogenization.
     set amonium-medium amonium-medium + (amonium-input / world)                 ; The amonium inflow concentration is added to local quantity without homogenization.
     set oxygen-medium oxygenmedium / world
     set nitrate-medium nitrate-medium + (nitrate-input / world)                 ; The nitrate inflow concentration is added to local quantity without homogenization.
     set pcolor scale-color blue succinate-medium 0 Succinate
    ]
  ]
end

; ____________________________ Maintenance Requirements for aerobic and anaerobic phases __________________________________________________________________________

to maintenance-requirements

ifelse time_now <= Time_Shutdown_O2
  [                                                                                       ; 1/14 (C4H4O4)2- + 1/4 O2 -> 1/7 CO2 + 1/7 HCO3- + 1/14 H2O
    set scm (energy_maintenance * steptime  * carbon / 4)                                 ; [mol succinate / molBiomass]
    set oxm (scm * abs (matrix:get rx_mant-aero 0 0) / abs (matrix:get rx_mant-aero 0 8)) ; [mol oxygen / molBiomass ]
  ]
  [                                                                                          ; 1/14 (C4H4O4)2- + 1/2 NO3- -> 1/7 CO2 + 1/7 HCO3- + 1/2 NO2- + 1/14 H2O
    set scm (energy_maintenance * steptime * carbon / 4)                                     ; [mol succinate / molBiomass ]
    set ntrm (scm * abs (matrix:get rx_mant_anox1 0 0) / abs (matrix:get rx_mant_anox1 0 8)) ; [mol nitrate / molBiomass ]
  ]
end

; ____________________________ Uptake proceeding according the core model INDISM __________________________________________________________________________

to uptake
  let uptsc (abs (random-normal (uSc) (DSTr * uSc)) * biomass * carbon * steptime)              ; [mmo] succinate by biomass uptake accord to INDISIM
  let uptam (abs (random-normal (uam) (DSTr * uam)) * biomass * carbon * steptime)              ; [mmol] amonium by biomass uptake accord to INDISIM
  let uptox (abs (random-normal (uox) (DSTr * u                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                       