;==================== begin zombie-blokade.nlogo ===============================
; zombie-blokade - a simple game to demonstrate agent-based modeling
;
; By Rik Blok, 2013 <http://www.zoology.ubc.ca/~rikblok/wiki/doku.php>
;
; This is free and unencumbered software released into the public domain.
;
; Anyone is free to copy, modify, publish, use, compile, sell, or
; distribute this software, either in source code form or as a compiled
; binary, for any purpose, commercial or non-commercial, and by any
; means.
;
; Revisions:
;
;   2013-09-05 - initial release by Rik Blok
;-------------------------------------------------------------------------------


__includes
[ "diffuse-in-bounds.nls"      ; Same as diffuse but with boundaries.
  "high-scores.nls"            ; Load/save high scores in a file.
  "output-wrap.nls"            ; Same as output-print but wraps printed output.
  "per-capita-tau-leaping.nls" ; Approximate Gillespie's SSA.
  "random-walk.nls"            ; Random walk for individuals.
  "sigfigs.nls"                ; Like precision but specifies number of significant figures.
  "updownhill-in-bounds.nls"   ; Like uphill/downhill but with boundary constraints.
]


extensions [ sound ]


breed [ calm-humans      calm-human      ] ; relaxed humans
breed [ panicked-humans  panicked-human  ] ; excited humans
breed [ docile-zombies   docile-zombie   ] ; relaxed zombies
breed [ frenzied-zombies frenzied-zombie ] ; excited zombies
breed [ bloks            blok            ]


patches-own
[ human-scent  ; diffusing scent zombies use to track humans
  zombie-scent ; diffusing scent humans use to avoid zombies
]


globals
[ diffusion-const      ; how fast humans/zombies move (> 0)
  excited-advantage    ; how much better is it to be excited vs relaxed? (>= 1)
  excited-scent        ; how much stronger do excited smell? (>= 1)
  excited-speed        ; how much faster do excited move? (>= 1)
  inaccuracy           ; determines dt in tau-leaping algorithm (> 0, < 1)
  infection-rate       ; rate of zombie attacks (> 0)
  is-running?          ; false on first pass through go, otherwise true
;  last-mouse-patch     ; remember which patch mouse was just on
  mouse-adds-bloks?    ; if true, mouse presses will add bloks, otherwise remove them
  relaxation-rate      ; how fast do zombies/humans relax from excited state? (> 0)
  scent-decay-rate     ; how fast scents die (> 0, <= 1)
  scent-dispersal-rate ; how fast scents disperse (> 0, <= 1)
  score                ; keep score to make game funner
  starting-zombies     ; how many starting zombies? (>= 1)
  time-to-cross-patch  ; typical ticks needed to cross one patch given diffusion-const
  turn-excited         ; amount of random turning when excited (>0, <= turn-relaxed)
  turn-relaxed         ; amount of random turning when relaxed (>= turn-excited)
  zombie-color         ; color to distinguish zombies from humans
]


to startup
  reset  ; Might as well setup immediately when model loaded.
end


to reset
; On model load or when user presses reset button.
  clear-all
  reset-ticks
  output-wrap 39 ; Set output box to wrap at column.
  output-wrap "ZOMBIE BLOKADE\n"
  output-wrap 
  ( word "Use white bloks to protect the humans (brown) from zombies (red). "
    "Bloks decay over time so watch out!\n"
  )
  output-wrap 
  ( word "Your score will be the blue area in the plot below -- "
    "the difference between the humans you keep alive versus "
    "the expected amount.\n"
  )
  
  ; Set defaults.
  set-default-shape turtles "person"
  set-default-shape bloks   "box"
  set diffusion-const 0.01 ; how fast do individuals move?
  set excited-advantage 2 ; advantage of being excited (>=1) over relaxed
  set excited-scent 1 ; smell of excited vs relaxed (>= 1)
  set excited-speed 4 ; running speed vs walking speed (=1)
  set inaccuracy 0.1 ; will be automatically adjusted to achieve desired speed
  set infection-rate 1 ; rate of zombie attacks (> 0)
  set is-running? false
;  set last-mouse-patch nobody
  set mouse-adds-bloks? false ; but will toggle to true (and print instructions) below.
  set relaxation-rate 0.25
  set scent-decay-rate 1 ; how fast scents decay (>0, <= 1)
  set scent-dispersal-rate 1 ; how fast scents disperse in space (>0, <= 1)
  set starting-zombies 1 ; how many zombies initially? (>= 1)
  set time-to-cross-patch random-walk-time-patch diffusion-const
  set turn-excited 15  ; how much to randomly turn when excited (<= turn-relaxed)
  set turn-relaxed 180 ; how much to randomly turn when relaxed (>= turn-excited)
  set zombie-color red + 0.75
  read-high-scores "zombie-blokade-high-scores.txt" 10
  toggle-mouse ; set mouse-adds-bloks? to true and print instructions

  ; Create initial populations.
  ; Make a green background.
  ask patches [ set pcolor green - 1 ]
  ; add bloks all around edges.
  ask patches with [ pxcor = min-pxcor or pxcor = max-pxcor or pycor = min-pycor or pycor = max-pycor ]
  [ sprout-bloks 1 [ set color white ]
  ]
  ; Place a zombie in the center.
  ask patch 0 0
  [ sprout-frenzied-zombies starting-zombies
    [ set color zombie-color
      set label "Grr! Argh!" 
      set size 1.5
    ]
  ]
  ; Add humans everywhere else.
  ask patches with [ not any? turtles-here ]
  [ sprout-calm-humans 1
    [ set color brown - 2 + random 7
    ]
  ]
  
  ; Place reactions here.
  ; 4 reactions for state of excitation (relaxed<-->excited for zombies and humans)
  ; where relaxed = calm/docile and excited = panicked/frenzied.
  ; Reaction: calm-human + any-zombie -> panicked-h + any-z @ rate 1
  add-per-capita-event calm-humans  
    task [ count docile-zombies-here + count frenzied-zombies-here ] 
    task [ set breed panicked-humans set size 1.5 ]
  ; Reaction: panicked-human -> calm-h @ relaxation-rate
  add-per-capita-event panicked-humans  
    task [ relaxation-rate ] 
    task [ set breed calm-humans set size 1 ]
  ; Reaction: docile-zombie + any-human -> frenzied-z + any-h @ rate 1
  add-per-capita-event docile-zombies  
    task [ count calm-humans-here + count panicked-humans-here ] 
    task [ set breed frenzied-zombies set size 1.5 ]
  ; Reaction: frenzied-zombie -> docile-z @ relaxation-rate
  add-per-capita-event frenzied-zombies  
    task [ relaxation-rate ] 
    task [ set breed docile-zombies set size 1 ]
  ; 4 reactions for zombie bites.  
  ; Excited humans have advantage against relaxed zombies and vice versa.
  ; Reaction: calm-human + docile-zombie -> docile-z + frenzied-z @ rate infection-rate
  add-per-capita-event calm-humans  
    task [ infection-rate * count docile-zombies-here ] 
    task 
    [ sound:play-drum "CABASA" volume
      set breed frenzied-zombies set color zombie-color set size 1.5 
    ]
  ; Reaction: calm-human + frenzied-zombie -> docile-z + frenzied-z @ rate advantage * infection-rate
  add-per-capita-event calm-humans  
    task [ infection-rate * excited-advantage * count frenzied-zombies-here ] 
    task 
    [ sound:play-drum "CABASA" volume
      set breed docile-zombies set color zombie-color 
    ]
  ; Reaction: panicked-human + docile-zombie -> docile-z + frenzied-z @ rate infection-rate / advantage
  add-per-capita-event panicked-humans  
    task [ infection-rate / excited-advantage * count docile-zombies-here ]
    task 
    [ sound:play-drum "CABASA" volume
      set breed frenzied-zombies set color zombie-color 
    ]
  ; Reaction: panicked-human + frenzied-zombie -> docile-z + frenzied-z @ rate infection-rate
  add-per-capita-event panicked-humans  
    task [ infection-rate * count frenzied-zombies-here ] 
    task 
    [ sound:play-drum "CABASA" volume
      set breed docile-zombies set color zombie-color set size 1 
    ]
  ; Destroy each blok randomly once per 100 ticks.
  ; Reaction: blok --> nothing @ rate 0.01
  add-per-capita-event bloks 
    task [ 0.01 ] 
    task 
    [ sound:play-drum "BASS DRUM 1" volume
      die 
    ]
  ; Bloks also destroyed by excited neighbors @ rate once per 10 ticks.
  ; Reaction: blok + (panicked-human or frenzied-zombie nearby) --> panicked-h or frenzied-h @ rate 0.1
  add-per-capita-event bloks 
    ; use in-radius to only look for zombies "touching" this blok.
    ; radius=0.8 is just bigger than distance to patch corner
    task [ 0.01 * ( count frenzied-zombies in-radius 0.8 + count panicked-humans in-radius 0.8 ) ] 
    task 
    [ sound:play-drum "BASS DRUM 1" volume
      die 
    ]
    
  ; system-dynamics tool gives (non-spatial) expectation to compare results against.
  system-dynamics-setup
  set-plot-x-range  -50 10
  set-plot-y-range 0 count calm-humans + count panicked-humans
  update-plots
end


to play
; Called repeatedly by 'play' button.
  ; Check if game over.
  if not (any? calm-humans or any? panicked-humans)
  [ if is-running? 
    [ set is-running? false
      end-game
    ]
    output-wrap "Press 'reset' to play again."
    stop
  ]

  ; Just for fun, mark last human.
  if (count calm-humans + count panicked-humans) = 1
  [ ask calm-humans     [ set label "I am Legend!" follow-me ]
    ask panicked-humans [ set label "I am Legend!" follow-me ]
  ]

  ; Check if game starting.
  if not is-running? 
  [ set is-running? true
    output-wrap "Try to form a blokade between the humans and zombies!\n"
    output-wrap 
    ( word "Click on a blok to move it away or "
      "click on an empty space to move a random blok there.  "
      "Be careful: every placed blok comes from somewhere else!\n"
    )
  ]
  
  ; Do actions.
  every dt  ; Try to match model ticks and real time.
  [ advance-frame ; Advance model time by dt and choose new dt.
  ]
  
  ; Update plot, but no more than 10 times per second.
  every 0.1
  [ let xmax precision (ticks + 1) 0 
    set-plot-x-range ( xmax - 60 ) xmax ; make sure plot auto scale is off
    update-plots
  ]
  
  handle-mouse-activity
end


to toggle-mouse
; Swap behaviour of mouse clicks: add <--> remove bloks.  
  set mouse-adds-bloks? not mouse-adds-bloks?
  output-wrap 
  ( word "Mouse clicks move bloks "
    ifelse-value mouse-adds-bloks? [ "into" ][ "away from" ]
    " the chosen location. Click 'toggle' in the top right to switch.\n"
  )
end


to advance-frame
; Carry out reactions, spread scents, move individuals, and keep score.
  ; time this "frame".
  reset-timer 
  ; do local reactions (relaxation, excitation, attacks, etc.)
  let new-dt per-capita-tau-leap inaccuracy dt
  ; Don't permit dt=0 because it screws up random-walk.
  if new-dt = 0
  [ ;set new-dt ( 1 - inaccuracy ) * time-to-cross-patch
    set new-dt dt * 1.1
  ]
  set dt new-dt
  if dt > 0.1 [ set dt 0.1 ]

  let is-in-bounds? task [ not any? bloks-here ]
  spread-scents is-in-bounds?
  follow-scents is-in-bounds?
  
  tick-advance dt
  set score score + score-increment
  if score < 0 [ set score 0 ]
  
  ; Try to tune simulation speed to 1 tick per second (ie. timer = dt)
  ; regardless of speed slider and computer speed.
  if timer > 1.1 * dt  ; too slow, make bigger jumps
  [ set inaccuracy inaccuracy * 1.1
  ]
  if timer < 0.9 * dt  ; too fast, run slower with smaller jumps
  [ set inaccuracy inaccuracy * 0.9
    wait dt - timer
  ]
  ; don't allow simulation to run too slow (no cheaters! :) )
  if inaccuracy > 0.25 [ set inaccuracy 0.25 ]
  
  ; Advance non-spatial system-dynamics model by dt.
  system-dynamics-go
end


to spread-scents
; Spread scents for tracking.
; Parameters:
[ is-in-bounds? ; task that reports true if scent can spread into patch
]
  ask calm-humans      [ set human-scent human-scent + dt ]
  ask panicked-humans  [ set human-scent human-scent + dt * excited-scent ]
  ask docile-zombies   [ set zombie-scent zombie-scent + dt ]
  ask frenzied-zombies [ set zombie-scent zombie-scent + dt * excited-scent ]
  diffuse-in-bounds  "human-scent"  (scent-dispersal-rate * dt)  is-in-bounds?
  diffuse-in-bounds  "zombie-scent" (scent-dispersal-rate * dt)  is-in-bounds?
  ask patches 
  [ set human-scent  human-scent  * (1 - scent-decay-rate * dt)
    set zombie-scent zombie-scent * (1 - scent-decay-rate * dt)
  ]
end


to follow-scents 
; Face towards/away from scents as desired and move (with some randomness).
; Parameters:
[ is-in-bounds? ; task that reports true if agent can move into patch
]
  ; Relaxed agents avoid overcrowding.
  let relaxed-is-in-bounds? task [ not any? bloks-here and count turtles-here < 2]
  ; Excited agents are too busy chasing/fleeing to care much about crowding.
  let excited-is-in-bounds? task [ not any? bloks-here and count turtles-here < 8]
  ; Humans want to be together...
  ask calm-humans
  [ set heading towards-uphill-in-bounds task [ human-scent ]  relaxed-is-in-bounds?
  ]
  ; ...except when they're fleeing zombies.
  ask panicked-humans
  [ set heading towards-downhill-in-bounds task [ zombie-scent ]  excited-is-in-bounds?
  ]
  ; Zombies sniff out and chase humans.
  ask docile-zombies
  [ set heading towards-uphill-in-bounds task [ human-scent ]  relaxed-is-in-bounds?
  ]
  ask frenzied-zombies
  [ set heading towards-uphill-in-bounds task [ human-scent ]  excited-is-in-bounds?
  ]
  
  ; Movement.  Add some randomness to direction and move forward.
  let relaxed-jump random-walk-jump-size diffusion-const dt
  let excited-jump relaxed-jump * excited-speed
  ; Excited move less randomly than relaxed.
  let tight-angle random-walk-cone-from-per-tick turn-excited dt
  let wide-angle  random-walk-cone-from-per-tick turn-relaxed dt
  random-walk-in-cone-bounds calm-humans      relaxed-jump wide-angle  is-in-bounds?
  random-walk-in-cone-bounds panicked-humans  excited-jump tight-angle is-in-bounds?
  random-walk-in-cone-bounds docile-zombies   relaxed-jump wide-angle  is-in-bounds?
  random-walk-in-cone-bounds frenzied-zombies excited-jump tight-angle is-in-bounds?
end


to-report score-increment
; Calculate and report score change. 
  ; Difference between actual vs. expected curves.
  let score-inc ( count calm-humans + count panicked-humans ) - (ch + ph) * ( count patches - count bloks )
  ; If running too slow, penalize player (should discourage slow speed slider cheat).
  if timer > dt [ set score-inc score-inc * ( dt / timer ) ]
  ; Score is time integral (blue area in plot).
  report score-inc * dt
end

to handle-mouse-activity
; Respond to mouse activity.
  if mouse-inside? and mouse-down? ; click
  [ let mouse-patch patch mouse-xcor mouse-ycor
;    if mouse-patch != last-mouse-patch
;    [ 
    ask mouse-patch
      [ if-else mouse-adds-bloks?
        [ if not any? turtles-here
          [ ask one-of bloks [ move-to mouse-patch ]
            sound:play-drum "HI AGOGO" volume
          ]
        ]
        [ if any? bloks-here 
          [ ask bloks-here [ move-to one-of patches with [ not any? turtles-here ]]
            sound:play-drum "LOW AGOGO" volume
          ]
        ]
      ]
;      set last-mouse-patch mouse-patch ; don't repeat actions on this patch
;    ]
  ]
end


to end-game
; Game is over.  Wrap up.
  set score precision score 0
  output-wrap ( word "Congratulations!  Your score is " score "!\n" )
  if is-high-score? score [ add-high-score score "You got a high score!  What is your name?" ]
  output-print "High scores:"
  output-print high-scores-list 25
end


to-report volume
; Convert sound level (dB) to power.  
; Relationship: level = level_0 + 10 log_10 ( power / power_0 )
; where level = 0 is off and
; level = 1...16 dB corresponds to power = 4...127.
  if sound-level = 0 [ report 0 ] ; sound off
  ; else
  report round ( 127 * 10 ^ ( ( sound-level - 16 ) / 10 ) )
  
end
;==================== end zombie-blokade.nlogo =================================
@#$#@#$#@
GRAPHICS-WINDOW
274
21
632
400
14
14
12.0
1
10
1
1
1
0
1
1
1
-14
14
-14
14
0
0
1
ticks
120.0

BUTTON
4
228
95
281
NIL
reset
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

OUTPUT
4
21
275
201
10

BUTTON
94
228
185
281
NIL
play
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

MONITOR
184
228
275
281
score
precision score 0
17
1
13

PLOT
4
280
275
400
humans
ticks
NIL
0.0
0.0
0.0
0.0
false
true
"" ""
PENS
"actual" 1.0 1 -13345367 true "" "plotxy ticks ( count calm-humans + count panicked-humans )"
"expected" 1.0 1 -7500403 true "" "plotxy ticks ( ch + ph ) * ( count patches - count bloks )"

SLIDER
4
196
275
229
sound-level
sound-level
0
16
8
4
1
dB
HORIZONTAL

BUTTON
542
10
632
47
toggle
toggle-mouse
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

MONITOR
448
10
543
47
mouse moves bloks
ifelse-value mouse-adds-bloks? [ \"IN\" ][ \"AWAY\" ]
17
1
9

@#$#@#$#@
## Zombie Blokade

A [NetLogo] game by [Rik Blok](http://www.zoology.ubc.ca/~rikblok/wiki/doku.php).

Oh no!  Patient zero has turned into a zombie!  Can you slow down the inevitable plague?  Place or remove bloks to isolate zombies from humans.  Act fast: proximity to humans drives zombies into a frenzy and panics humans, making the outbreak harder to contain!

Bloks impede zombies and humans.  Unfortunately, bloks degrade naturally over time.  Also, frenzied zombies and panicked humans can break through bloks, destroying them.  Be careful where you try to place bloks; they can't be dropped on people (living or undead).

## Movement

Zombies are attracted by the scent of humans.  When calm, humans tend to congregate but when panicked they simply flee zombies.  Neither humans or zombies can pass through bloks.  (But panicked/frenzied individuals can break them.)  Use the mouse to build a blokade to keep the humans and zombies apart.  

## How it works

Zombie Blokade contains a stochastic, spatial, SI (susceptible-infected) [epidemic model](https://en.wikipedia.org/wiki/Epidemic_model).  There are four kinds of agents:

  * CH = Calm Human (relaxed)
  * PH = Panicked Human (excited)
  * DZ = Docile Zombie (relaxed)
  * FZ = Frenzied Zombie (excited)

Changes occur when nearby agents interact according to these [reactions](https://en.wikipedia.org/wiki/Reaction_mechanism).

Calm humans encountering a zombie may panic (if they survive):

  * CH + DZ &rarr; PH + DZ @ rate 1
  * CH + FZ &rarr; PH + FZ @ rate 1

But eventually they'll calm back down:

  * PH &rarr; CH @ relaxation-rate

Likewise, docile zombies will frenzy when they cross paths with a human:

  * DZ + CH &rarr; FZ + CH @ rate 1
  * DZ + PH &rarr; FZ + PH @ rate 1

But they too will relax back to their docile state eventually:

  * FZ &rarr; DZ @ relaxation-rate

Unfortunately, the living don't always survive zombie attacks.  Encounters send attacking zombies into a frenzy and victims turn into docile zombies.  In these interactions, excited (panicked/frenzied) have an advantage over relaxed (calm/docile) individuals:

  * CH + DZ &rarr; DZ + FZ @ rate infection-rate
  * CH + FZ &rarr; DZ + FZ @ rate infection-rate x excited-advantage
  * PH + DZ &rarr; DZ + FZ @ rate infection-rate / excited-advantage
  * PH + FZ &rarr; DZ + FZ @ rate infection-rate

Bloks are not permanent.  They degrade naturally with an expected lifespan of 100 seconds:

  * blok &rarr; nothing @ rate 0.01

And finally, excited (panicked/frenzied) individuals can break through and destroy nearby bloks in their fury:

  * blok + PH(nearby) &rarr --> PH(nearby) @ rate 0.1
  * blok + FZ(nearby) &rarr --> FZ(nearby) @ rate 0.1

## Scoring 

Together, the above reactions fully describe the transitions between states.  A simple "tau-leaping" method is used to decide which reactions occur over time.  For comparison, [NetLogo]'s [System Dynamics Modeler](http://ccl.northwestern.edu/netlogo/docs/systemdynamics.html) represents the same reactions assuming a large, "well stirred" population.

You can change the outcome by adding/removing bloks to keep humans and zombies separated. The number of humans you keep alive is plotted over time, compared against the system dynamics expectation.  Your score is the total (blue) area between the curves.

## Things to try

The blue curve shows the actual outcome and the grey curve shows the expected outcome if the population is "well stirred".  Can you check that they're consistent?  How would you change the model code to "stir" the population?

Maybe dropping bloks on individuals should be allowed.  But bloks are heavy so they'd kill living humans they landed on.  Zombies are already (un)dead so they'd just be trapped until the blok was removed.  How would you add this to the code?

If only the humans could find some weapons to fight back!  A shotgun or a chainsaw would sure help.  I wonder how adding weapons would change the game?

## References

[[Gibson2000]] Gibson, Michael A. and Bruck, Jehoshua. [Efficient Exact Stochastic Simulation of Chemical Systems with Many Species and Many Channels](http://dx.doi.org/10.1021/jp993732q). J. Phys. Chem. A, 104(9): 1876-1889. doi:[10.1021/jp993732q](http://dx.doi.org/10.1021/jp993732q). 2000. 

[[NetLogo]] Wilensky, U. NetLogo. [http://ccl.northwestern.edu/netlogo/](http://ccl.northwestern.edu/netlogo/). Center for Connected Learning and Computer-Based Modeling, Northwestern University. Evanston, IL. 1999.

[Brown2009]: http://dx.doi.org/10.1098/rstb.2009.0055
[Gibson2000]: http://dx.doi.org/10.1021/jp993732q
[NetLogo]: http://ccl.northwestern.edu/netlogo/
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.0.4
@#$#@#$#@
@#$#@#$#@
1.0 
    org.nlogo.sdm.gui.AggregateDrawing 12 
        org.nlogo.sdm.gui.StockFigure "attributes" "attributes" 1 "FillColor" "Color" 225 225 182 136 120 60 40 
            org.nlogo.sdm.gui.WrappedStock "ch" "count calm-humans / ( count patches - count bloks )" 0   
        org.nlogo.sdm.gui.StockFigure "attributes" "attributes" 1 "FillColor" "Color" 225 225 182 546 121 60 40 
            org.nlogo.sdm.gui.WrappedStock "ph" "count panicked-humans / ( count patches - count bloks )" 0   
        org.nlogo.sdm.gui.StockFigure "attributes" "attributes" 1 "FillColor" "Color" 225 225 182 143 467 60 40 
            org.nlogo.sdm.gui.WrappedStock "dz" "count docile-zombies / ( count patches - count bloks )" 0   
        org.nlogo.sdm.gui.StockFigure "attributes" "attributes" 1 "FillColor" "Color" 225 225 182 536 479 60 40 
            org.nlogo.sdm.gui.WrappedStock "fz" "count frenzied-zombies / ( count patches - count bloks )" 0   
        org.nlogo.sdm.gui.RateConnection 3 208 139 371 139 534 140 NULL NULL 0 0 0 
            org.jhotdraw.standard.ChopBoxConnector REF 1  
            org.jhotdraw.standard.ChopBoxConnector REF 3  
            org.nlogo.sdm.gui.WrappedRate "ch * ( dz + fz + ph )" "excite-humans" REF 2 REF 4 0   
        org.nlogo.sdm.gui.RateConnection 3 534 140 371 140 208 140 NULL NULL 0 0 0 
            org.jhotdraw.standard.ChopBoxConnector REF 3  
            org.jhotdraw.standard.ChopBoxConnector REF 1  
            org.nlogo.sdm.gui.WrappedRate "ph * relaxation-rate" "relax-humans" REF 4 REF 2 0   
        org.nlogo.sdm.gui.RateConnection 3 215 488 369 492 524 497 NULL NULL 0 0 0 
            org.jhotdraw.standard.ChopBoxConnector REF 5  
            org.jhotdraw.standard.ChopBoxConnector REF 7  
            org.nlogo.sdm.gui.WrappedRate "dz * ( ch + ph )" "excite-zombies" REF 6 REF 8 0   
        org.nlogo.sdm.gui.RateConnection 3 524 497 369 492 215 488 NULL NULL 0 0 0 
            org.jhotdraw.standard.ChopBoxConnector REF 7  
            org.jhotdraw.standard.ChopBoxConnector REF 5  
            org.nlogo.sdm.gui.WrappedRate "fz * relaxation-rate" "relax-zombies" REF 8 REF 6 0   
        org.nlogo.sdm.gui.RateConnection 3 201 172 365 319 530 467 NULL NULL 0 0 0 
            org.jhotdraw.standard.ChopBoxConnector REF 1  
            org.jhotdraw.standard.ChopBoxConnector REF 7  
            org.nlogo.sdm.gui.WrappedRate "ch * infection-rate * dz" "calm-docile-attack" REF 2 REF 8 0   
        org.nlogo.sdm.gui.RateConnection 3 166 172 169 313 172 455 NULL NULL 0 0 0 
            org.jhotdraw.standard.ChopBoxConnector REF 1  
            org.jhotdraw.standard.ChopBoxConnector REF 5  
            org.nlogo.sdm.gui.WrappedRate "ch * infection-rate * excited-advantage * fz" "calm-frenzied-attack" REF 2 REF 6 0   
        org.nlogo.sdm.gui.RateConnection 3 574 173 570 320 566 467 NULL NULL 0 0 0 
            org.jhotdraw.standard.ChopBoxConnector REF 3  
            org.jhotdraw.standard.ChopBoxConnector REF 7  
            org.nlogo.sdm.gui.WrappedRate "ph * infection-rate / excited-advantage * dz" "panicked-docile-attack" REF 4 REF 8 0   
        org.nlogo.sdm.gui.RateConnection 3 538 173 373 314 209 455 NULL NULL 0 0 0 
            org.jhotdraw.standard.ChopBoxConnector REF 3  
            org.jhotdraw.standard.ChopBoxConnector REF 5  
            org.nlogo.sdm.gui.WrappedRate "ph * infection-rate * fz" "panicked-frenzied-attack" REF 4 REF 6 0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 1.0 0.0
0.0 1 1.0 0.0
0.2 0 1.0 0.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
