breed [ circles circle ]                                   ;; Host cells 
breed [ flags flag ]                                       ;; Key host cells
breed [ triangles triangle ]                               ;; Pathogens A
breed [ stars star ]                                       ;; Pathogens B
turtles-own [energy max-energy start-energy birth-energy]  ;; All turtles have energy and a maximum amount of energy they can take on.
patches-own [countdown patch-energy]                       ;; Countdown makes it run. Patch-energy is new to the program.
circles-own [circle-energy-to-survive]                     
flags-own [flag-energy-to-survive]
triangles-own [triangle-energy-to-survive]
stars-own [star-energy-to-survive]
globals [pathB cumulative-energy-deficit]
                                                         
to setup
  clear-all
  set pathB 1  ;; Turns the pathB switch in the code to "on" in the setup. The user must still physically turn on the switch at the interface screen.
  
  ask patches 
  [ set pcolor green  set patch-energy 100]   ;; New code here with patch-energy.
    
 set-default-shape circles "circle"   ;; Host cells
  ask patches 
   [ sprout-circles 1 [ set color blue set start-energy 300 set birth-energy 90 set max-energy 300 set circle-energy-to-survive 40] ]  ;; START-ENERGY HAD BEEN 120 FOR ALL CELLS
  
 set-default-shape flags "flag"       ;; key host cells such as heart cells
  ask patches with [(pycor < 3) and (pycor > -3 ) and (pxcor < 3) and (pxcor > -3)]
   [sprout-flags 1 [ set color orange set start-energy 300 set max-energy 300 set flag-energy-to-survive 30] ]  ;; Can store lots of energy; not prone to apoptosis since irreplaceable.
  ask circles with [ (pycor < 3) and (pycor > -3 ) and (pxcor < 3) and (pxcor > -3)] [ die ] ;; This clears out the circles underneath the flags.  ;; FLAG MAX-ENERGY HAD BEEN 500
 
 set-default-shape triangles "triangle"  ;; Pathogens A (these are like large extracellular pathogens)
  ask circles with [ (pycor <= start-upper-pathogen-pycor) and (pycor > start-lower-pathogen-pycor) and (pxcor <= start-upper-pathogen-pxcor) and (pxcor > start-lower-pathogen-pxcor) ] [ die ]
   ;; This clears out the circles underneath where the triangles will be.
  ask patches with [ (pycor <= start-upper-pathogen-pycor) and (pycor > start-lower-pathogen-pycor) and (pxcor <= start-upper-pathogen-pxcor) and (pxcor > start-lower-pathogen-pxcor)]
   [sprout-triangles 1 [ set color red set start-energy 300 set birth-energy 60 set max-energy 300 set triangle-energy-to-survive 40] ]  
   ;; Low energy storage compared to host cells is a vulnerability; same energy to survive as host cells.
   ;; The pathogens' energy to survive can be modified by the slider (see below).
 
  ask turtles
  [ set energy start-energy ]  ;; Even though start-energy is mentioned above for each breed, this code is needed to actually impart the energy.
                                
  reset-ticks
end
  
to go 
  set cumulative-energy-deficit 326700 - ((sum [energy] of circles) + (sum [energy] of flags)) + cumulative-energy-deficit  
      ;; There are 1089 potential host cells and key host cell x 300 max energy units per cell.
      ;; Note that the host starts with a deficit of 3000 because the of the initial infection (e.g. a bite wound).
      ;; The host must recover to its full 1089 cells.
      ;; Also changed the formula from 323700 to 326700 in the instantaneous-energy-deficit monitor on the interface.
  stop-adding-energy
  keep-turtles-still
  invoke-pathogen-harm
  invoke-pathogen-take-energy
  invoke-pathogen-low-energy
  invoke-pathogen-wander
  
    if auto-infect-with-pathogenB? and pathB = 1  ;; This is a switch in the code that permits turning off the secondary infection (pathogen B) after one tick. 
    [if count circles <= 1000 ;;and mean [energy] of circles <= 90  ;; I added the mean energy statement, but this could have been instantaneoud-energy-deficit.
      [invoke-auto-infection-pathogenB] ] ;; This code for pathogenB, the secondary infection, is farther down.
    
  invoke-pathogenB-harm
  invoke-pathogenB-take-energy
  invoke-pathogenB-low-energy
  invoke-pathogenB-wander
     
  invoke-local-directed
  invoke-local-stressor
  invoke-regional-stressor
  invoke-amputation
  invoke-amputationB
  invoke-hostcell-wander
  
  invoke-auto-systemic-stressor
  invoke-auto-systemic-stressor-slow
  invoke-auto-systemic-stressor-cycle
  invoke-auto-systemic-stressor-hi-cycle
  
  replicate   
  check-death
  tick
end

to stop-adding-energy            ;; Limits the amount of energy each cell can store, but there can be temporary overshooting of the amount.
ask turtles 
  [ if energy >= max-energy
    [set energy max-energy ]]
end

to keep-turtles-still      ;; This sets the action. The switches on the interface screen allow movement to adjacent vacant spacesfor either pathogens or host cells.
    ask turtles [
        right random 360
        forward 0                ;; This gets the turtles, especially the flags, to move in place; therefore no visible movement but it lets time occur (ticks go).
 
 ;; All turtles derive energy from the patches as patch-energy at each tick and lose a little less energy (metabolic energy) with each tick.
 set energy (random-normal 1 0.02) * (energy + ((patch-energy * 0.14) * ((100 - systemic-host-stressor) / 100))) ;; The systemic host stressor is here.    
   ;; set energy (random-normal 1 0.02) * (energy + (14 * (100 - systemic-host-stressor) / 100)) ;; This was old code where the turtles were simply given energy at each tick.
 set energy (random-normal 1 0.02) * (energy - 10)   ;; Every tick costs the turtles about 10 units of energy.  
;; This is the energy gain and loss per tick; the random-normal (mean of 1 and SD of 0.02) gives a bit of randomness to the whole program.
;; I don't want too much randomness since all turtles are bathed in the same fluid of patch-energy and are superficially identical
;; There's additional randomness in the pathogen damage routine and in the host defense routines.
;; The systemic host stressor represents the acute-phase response: fever, iron & zinc sequestration, anorexia/nutrient restriction, anemia/hypoxia, systemic acidosis, etc.

 ifelse show-energy?             ;; This is for the switch on the interface screen. Best to have it on.
     [ set label round energy ]  ;; "round" rounds the energy value on the screen to a readable integer.
     [ set label ""]             ;; If set to "No", no energy values show on the screen   
     
    ]
end

;; PATHOGEN PHENOTYPE SLIDERS & SWITCH 

to invoke-pathogen-harm  ;; This acts like a toxin to harm adjacent host cells.
  ask triangles [ ask circles in-radius 1 [ set energy (random-normal 1 0.02) * (energy - pathogen-harm-hostcells) ]]
  ask triangles [ ask flags in-radius 1 [ set energy (random-normal 1 0.02) * (energy - pathogen-harm-hostcells) ]]  
end

to invoke-pathogen-take-energy  ;; This gives the pathogen energy when it's adjacent to a host cell.
  ask triangles [ if any? circles in-radius 1 [ set energy (random-normal 1 0.02) * (energy + pathogen-take-up-energy) ]]  
  ask triangles [ if any? flags in-radius 1 [ set energy (random-normal 1 0.02) * (energy + pathogen-take-up-energy) ]] 
end

to invoke-pathogen-low-energy  ;; Lowering the "pathogen-survive-low-energy" number lets them survive better. They still need 120 energy units to reproduce (2 * birth-energy).
  ask triangles [ set triangle-energy-to-survive pathogen-survive-low-energy ]
  ask triangles [ set max-energy pathogen-survive-low-energy + 260 ]  ;; As a minor handicap for enhanced survival, I've also reduced the max-energy they can store.
end
  
to invoke-pathogen-wander  ;; In this model, letting the pathogens move to free spaces helps them only if they are more lethal than host cells; otherwise they're more vulnerable.
  ifelse pathogen-wander?
  [ ask triangles 
  [ let empty-patches neighbors4 with [ not any? turtles-here ]
    if any? empty-patches
    [ let target one-of empty-patches
      face target
      move-to target ]] ]
  [
  ]
end

;; PATHOGEN B PHENOTYPE SLIDERS & SWITCH 

to invoke-auto-infection-pathogenB    ;; The parameters needed to start this secondary infection are coded earlier in the program.  
 ask circles with [ (pycor <= start-upper-pathogenB-pycor) and (pycor > start-lower-pathogenB-pycor) and (pxcor <= start-upper-pathogenB-pxcor) and (pxcor > start-lower-pathogenB-pxcor)]    
     [die]  ;; This clears out the host cells at the site before pathogen B comes along (otherwise the pathogens and the host cells are on the same patch and most pathogens die).
 ask patches with [ (pycor <= start-upper-pathogenB-pycor) and (pycor > start-lower-pathogenB-pycor) and (pxcor <= start-upper-pathogenB-pxcor) and (pxcor > start-lower-pathogenB-pxcor)]
  [sprout-stars 1 [ set color red set start-energy 300 set birth-energy 60 set max-energy 300 set star-energy-to-survive 40] ] 
 ask stars
  [ set energy start-energy ] 
 set pathB 0   ;; This turns off the infection with pathogen B after a single round.  
end  
 
to invoke-pathogenB-harm  ;; This acts like a toxin to harm adjacent host cells
  ask stars [ ask circles in-radius 1 [ set energy (random-normal 1 0.02) * (energy - pathogenB-harm-hostcells) ]]
  ask stars [ ask flags in-radius 1 [ set energy (random-normal 1 0.02) * (energy - pathogenB-harm-hostcells) ]]  
end

to invoke-pathogenB-take-energy  ;; This gives pathogen B energy when adjacent to host cell
  ask stars [ if any? circles in-radius 1 [ set energy (random-normal 1 0.02) * (energy + pathogenB-take-up-energy) ]]  
  ask stars [ if any? flags in-radius 1 [ set energy (random-normal 1 0.02) * (energy + pathogenB-take-up-energy) ]] 
end

to invoke-pathogenB-low-energy  ;; Lowering the "pathogenB-survive-low-energy" number lets them survive better
  ask stars [ set star-energy-to-survive pathogenB-survive-low-energy ]
  ask stars [set max-energy pathogenB-survive-low-energy + 160 ]  ;; I've linked the max-energy they can use to the survival energy threshold as a minor handicap
end
  
to invoke-pathogenB-wander  ;; In this model, letting the pathogens move to free spaces helps them only if they are more lethal than host cells; otherwise they're more vulnerable.
  ifelse pathogenB-wander?
  [ask stars 
  [ let empty-patches neighbors4 with [ not any? turtles-here ]
    if any? empty-patches
    [ let target one-of empty-patches
      face target
      move-to target ]] ]
  [
  ]
end 
    
;; HOST DEFENSES: SLIDERS & SWITCHES (Systemic stressor slider is already incorporated above)  

to invoke-local-directed  ;; Host cells directly harm adjacent pathogens (as if injecting a toxin).
  ask circles [ ask triangles in-radius 1 [ set energy (random-normal 1 0.02) * (energy - local-host-directed) ]]   
  ask circles [ ask stars in-radius 1 [ set energy (random-normal 1 0.02) * (energy - local-host-directed) ]]
end 

to invoke-local-stressor ;; Harms all adjacent cells if the host cell is in contact with a pathogen as if depriving the area of nutrients or causing acidosis.
  ask triangles [ ask circles in-radius 1 [ ask turtles in-radius 1 [ set energy (random-normal 1 0.02) * (energy - local-host-stressor) ]]]
  ask stars [ ask circles in-radius 1 [ ask turtles in-radius 1 [ set energy (random-normal 1 0.02) * (energy - local-host-stressor) ]]]
end

to invoke-regional-stressor  ;; Harms all cells within 5 patches from conflict interface as if depriving the region of blood supply due to clotting & edema.
ask triangles [ ask circles in-radius 2 [ ask turtles in-radius 5 [ set energy (random-normal 1 0.02) * (energy * ((100 - regional-host-stressor) / 100)) ]]]
ask stars [ ask circles in-radius 2 [ ask turtles in-radius 5 [ set energy (random-normal 1 0.02) * (energy * ((100 - regional-host-stressor) / 100)) ]]]
;;  Not sure why the effect is so dramatic with a "regional-host-stressor" setting of 1 or 2
end

to invoke-amputation  ;;  This code harms all cells in the lower right region as if using a tourniquet (or hypersensitive reaction in plants).
 ask turtles with [ (pycor < 0) and (pxcor > 10)] [ set energy (random-normal 1 0.02) * (energy * ((100 - regional-amputation) / 100)) ] 
end

to invoke-amputationB  ;;  This code harms all cells in the lower left region as if using a tourniquet (or hypersensitive reaction in plants).
 ask turtles with [ (pycor < 0) and (pxcor < -10)] [ set energy (random-normal 1 0.02) * (energy * ((100 - regional-amputationB) / 100)) ] 
end
    
to invoke-hostcell-wander  ;; Lets the host cells move to free spaces.
  ifelse hostcell-wander?  ;; This slightly helps the host IF the host cells can harm the pathogens either directly or by local stressing.
  [ask circles 
  [ let empty-patches neighbors4 with [ not any? turtles-here ]
    if any? empty-patches
    [ let target one-of empty-patches
      face target
      move-to target ]] ]
  [
  ]
  end

to invoke-auto-systemic-stressor  ;; An automatic device that quickly turns the systemic stressor on and then off, etc. Lots of overshooting with this strategy.
  ifelse auto-systemic-stressor?  ;; However, once the pathogen invades enough that there are less than 925 host cells, it won't turn on anymore.
  [ if count circles <= 1030 ;;and count circles > 950  ;; This "and" allows one to get bigger cycles instead of just hovering around and overshooting 925.
    [ set systemic-host-stressor 60 ]
    if count circles <= 925                                     
    [ set systemic-host-stressor 0 ]]
  [ ]
 if (count triangles + count stars) = 0  ;; These 2 lines allow the host to recover after eliminating the pathogens (otherwise the stressor stays at 60).
 [set systemic-host-stressor 0]     
end

to invoke-auto-systemic-stressor-slow  ;; An automatic device that slowly turns the systemic stressor on and finally has strong 60 to 0 cycling at the end.
  ifelse auto-systemic-stressor-slow?  
  [ if count circles <= 1030   
    [ set systemic-host-stressor 10 ]
    if count circles <= 1020                                    
    [ set systemic-host-stressor 20 ]
 if count circles <= 1010                                    
    [ set systemic-host-stressor 30 ]
 if count circles <= 1000                                    
    [ set systemic-host-stressor 40 ] 
 if count circles <= 990                                    
    [ set systemic-host-stressor 50 ]  
 if count circles <= 980                                    
    [ set systemic-host-stressor 60 ]    
 if count circles <= 925                                    
    [ set systemic-host-stressor 0 ]] 
 [ ]
if (count triangles + count stars) = 0  ;; These 2 lines allow the host to recover after eliminating the pathogens (otherwise the stressor stays at the final level).
 [set systemic-host-stressor 0]
end

to invoke-auto-systemic-stressor-cycle  ;; An automatic device that cycles stress slowly
  ifelse auto-systemic-stressor-cycle?  
  [ if count circles <= 1030 
    [ set systemic-host-stressor 30 ]
    if count circles <= 1010                                     
    [ set systemic-host-stressor 60 ]
    if count circles <= 990 
    [ set systemic-host-stressor 30 ]
    if count circles <= 970                                     
    [ set systemic-host-stressor 0 ]
    if count circles <= 950                                    
    [ set systemic-host-stressor 30 ]
    if count circles <= 930                                     
    [ set systemic-host-stressor 60 ]
    if count circles <= 910                                    
    [ set systemic-host-stressor 30 ]
    if count circles <= 890                                   
    [ set systemic-host-stressor 0 ]
    if count circles <= 870                                    
    [ set systemic-host-stressor 30 ]
    if count circles <= 850                                    
    [ set systemic-host-stressor 60 ]
    if count circles <= 830                                    
    [ set systemic-host-stressor 30 ]
    if count circles <= 810        
    [ set systemic-host-stressor 0 ]]
  [ ]
if (count triangles + count stars) = 0  ;; These 2 lines allow the host to recover after eliminating the pathogens (otherwise the stressor stays at the final level).
 [set systemic-host-stressor 0]
end

to invoke-auto-systemic-stressor-hi-cycle  ;; An automatic device that quickly cycles from 60 to 0. This minimizes overshooting.
  ifelse auto-systemic-stressor-hi-cycle?  
  [ if count circles <= 1030 
    [ set systemic-host-stressor 60 ]
    if count circles <= 1015                                     
    [ set systemic-host-stressor 0 ]
    if count circles <= 1000                                     
    [ set systemic-host-stressor 60 ]
    if count circles <= 985                                     
    [ set systemic-host-stressor 0 ]
    if count circles <= 970                                     
    [ set systemic-host-stressor 60 ]
    if count circles <= 955                                     
    [ set systemic-host-stressor 0 ]
    if count circles <= 940                                     
    [ set systemic-host-stressor 60 ]
    if count circles <= 925                                     
    [ set systemic-host-stressor 0 ]]
  [ ]
if (count triangles + count stars) = 0  ;; These 2 lines allow the host to recover after eliminating the pathogens (otherwise the stressor stays at the final level).
 [set systemic-host-stressor 0]
end


to replicate   
   ask triangles                                                                   ;; Pathogens replicate if they have 2x birth-energy (60 units) and a vacant patch for the hatchling to move to.
    [ if energy >  2 * birth-energy and any? neighbors4 with [not any? turtles-here]
      [ set energy energy / 2                                                      ;; This cuts the energy in half before hatching.
       hatch 1 [set energy birth-energy                                            ;; There's a little lost energy since mother may have more than 2x birth-energy,
         let target one-of neighbors4 with [not any? turtles-here]                 ;;   but daughter cell (hatchling) only gets the birth-energy.
        face target
        move-to target   ] ] ]
     
   ask stars [                                                                       ;; Pathogen B replicate if they have 2x birth-energy (60 units) and a vacant patch for the hatchling to move to.
     if energy >  2 * birth-energy and any? neighbors4 with [not any? turtles-here]
      [ set energy energy / 2                                                        ;; This cuts the energy in half before hatching.
       hatch 1 [set energy birth-energy                                              ;; There's a little lost energy since mother may have more than 2x birth-energy,
         let target one-of neighbors4 with [not any? turtles-here]                   ;;   but daughter cell (hatchling) only gets the birth-energy.
        face target
        move-to target   ] ] ] 
     
   ask circles [                                                                       
     if energy >  2 * birth-energy and any? neighbors4 with [not any? turtles-here]  ;; Host cells replicate if they have 2x birth-energy (90 units) and a vacant patch for the hatchling to move to.
      [ set energy energy / 2                                                        ;; This cuts the energy before hatching.
       hatch 1 [set energy birth-energy                                              ;; There's a little lost energy since mother may have more than 2x birth-energy,
         let target one-of neighbors4 with [not any? turtles-here]                   ;;   but daughter cell (hatchling) only gets the birth-energy.
        face target
        face target
        move-to target   ] ] ]                                                       ;; Flags (key host cells) don't replicate
end

  to check-death
   ask circles
    [ if energy <= circle-energy-to-survive [ die ]]
   ask flags
    [ if energy <= flag-energy-to-survive [ die ]]
   ask triangles
    [ if energy <= triangle-energy-to-survive [ die ]]  
   ask stars
    [ if energy <= star-energy-to-survive [ die ]]         
 end
  
  
  
@#$#@#$#@
GRAPHICS-WINDOW
226
48
665
508
16
16
13.0
1
10
1
1
1
0
0
0
1
-16
16
-16
16
1
1
1
ticks
30.0

BUTTON
7
10
70
43
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

MONITOR
682
53
826
98
count circles (host cells)
count circles
3
1
11

MONITOR
682
152
848
197
count triangles (pathogens)
count triangles
3
1
11

SWITCH
20
50
154
83
show-energy?
show-energy?
0
1
-1000

MONITOR
682
102
843
147
count flags (key host cells)
count flags
17
1
11

PLOT
939
337
1139
487
Pathogens
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"pathogens" 1.0 0 -2674135 true "" "plot count triangles"

PLOT
939
21
1139
171
Host cells
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -11221820 true "" "plot count circles"

PLOT
939
179
1139
329
Key host cells
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"keyhostcells" 1.0 0 -955883 true "" "plot count flags"

MONITOR
1144
22
1271
67
Sum host cell energy
sum [ energy] of circles
1
1
11

MONITOR
1146
181
1296
226
Sum key host cell energy
sum [energy] of flags
1
1
11

MONITOR
1148
340
1283
385
Sum pathogen energy
sum [ energy ] of triangles
1
1
11

MONITOR
1145
73
1279
118
Mean host cell energy
mean [energy] of circles
1
1
11

MONITOR
1146
231
1303
276
Mean key host cell energy
mean [energy] of flags
1
1
11

MONITOR
1147
391
1289
436
Mean pathogen energy
mean [energy] of triangles
1
1
11

TEXTBOX
684
413
834
431
NIL
11
0.0
1

OUTPUT
678
300
933
375
10

BUTTON
263
10
389
46
Go until recovery
go\nif count triangles = 0\nand count stars = 0\n[ output-print (word \"Pathogens eliminated in \" ticks \" ticks.\" )]\nif count flags <= 20\n[ output-print (word \"Host killed in \" ticks \" ticks (too few key host cells).\")]\nif count circles <= 700\n[ output-print (word \"Host killed in \" ticks \" ticks (too few host cells).\") ]\nif count triangles >= 200\n[ output-print (word \"Pathogens overrun host.\") ]\nif ((count triangles = 0 \nand count stars = 0)\n and\n(count circles + count flags) = 1089)\nor count flags <= 20\nor count circles <= 700 \n[ output-print (word round cumulative-energy-deficit \": Cumulative energy deficit\") stop ]\n
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

BUTTON
92
10
241
46
Go until pathogens killed
go\nif count triangles = 0\nand count stars = 0\n[ output-print (word \"Pathogens eliminated in \" ticks \" ticks.\")]\nif count flags <= 20\n[ output-print (word \"Host killed in \" ticks \" ticks (too few key host cells).\")]\nif count circles <= 700\n[ output-print (word \"Host killed in \" ticks \"ticks (too few host cells).\")]\nif count triangles >= 200 \n[ output-print (word \"Pathogens overrun host.\")]\nif (count triangles = 0\nand count stars = 0)\nor count flags <= 20\nor count circles <= 700\n[output-print (word round cumulative-energy-deficit \" : Cumulative energy deficit\") stop]\n
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

TEXTBOX
681
381
831
493
Pathogen B phenotypes
11
0.0
1

TEXTBOX
17
420
167
476
NIL
11
0.0
1

TEXTBOX
679
277
829
295
Result:
11
0.0
1

TEXTBOX
23
471
173
501
An initial stressor-level of 29-30 is close to steady-state.\n
11
0.0
1

SLIDER
4
366
188
399
regional-host-stressor
regional-host-stressor
0
10
0
0.5
1
NIL
HORIZONTAL

SLIDER
4
332
176
365
local-host-stressor
local-host-stressor
0
50
10
1
1
NIL
HORIZONTAL

SLIDER
5
436
180
469
systemic-host-stressor
systemic-host-stressor
0
100
0
1
1
NIL
HORIZONTAL

SLIDER
4
296
176
329
local-host-directed
local-host-directed
0
50
0
1
1
NIL
HORIZONTAL

SLIDER
2
112
189
145
pathogen-harm-hostcells
pathogen-harm-hostcells
0
50
20
1
1
NIL
HORIZONTAL

SLIDER
3
147
189
180
pathogen-take-up-energy
pathogen-take-up-energy
0
50
20
1
1
NIL
HORIZONTAL

SLIDER
3
182
205
215
pathogen-survive-low-energy
pathogen-survive-low-energy
10
40
40
1
1
NIL
HORIZONTAL

SWITCH
3
216
164
249
pathogen-wander?
pathogen-wander?
1
1
-1000

SWITCH
232
614
381
647
hostcell-wander?
hostcell-wander?
0
1
-1000

TEXTBOX
25
98
175
116
Pathogen phenotypes
11
0.0
1

TEXTBOX
26
282
176
300
Host defenses
11
0.0
1

SWITCH
5
512
193
545
auto-systemic-stressor?
auto-systemic-stressor?
1
1
-1000

SLIDER
234
510
416
543
start-lower-pathogen-pxcor
start-lower-pathogen-pxcor
11
15
14
1
1
NIL
HORIZONTAL

SLIDER
234
546
417
579
start-lower-pathogen-pycor
start-lower-pathogen-pycor
-12
-1
-10
1
1
NIL
HORIZONTAL

SLIDER
457
546
633
579
start-upper-pathogen-pycor
start-upper-pathogen-pycor
-10
1
-5
1
1
NIL
HORIZONTAL

SLIDER
457
510
633
543
start-upper-pathogen-pxcor
start-upper-pathogen-pxcor
13
16
16
1
1
NIL
HORIZONTAL

TEXTBOX
289
582
627
618
The location and number of pathogens can be set by the sliders above. The default is 2 columns of 4 pathogens along the surface.
11
0.0
1

SLIDER
5
400
177
433
regional-amputation
regional-amputation
0
30
1
1
1
NIL
HORIZONTAL

SLIDER
508
613
690
646
start-lower-pathogenB-pxcor
start-lower-pathogenB-pxcor
-17
-14
-15
1
1
NIL
HORIZONTAL

SLIDER
508
647
691
680
start-lower-pathogenB-pycor
start-lower-pathogenB-pycor
-12
-1
-11
1
1
NIL
HORIZONTAL

SLIDER
694
613
882
646
start-upper-pathogenB-pxcor
start-upper-pathogenB-pxcor
-17
-13
-15
1
1
NIL
HORIZONTAL

SLIDER
694
648
883
681
start-upper-pathogenB-pycor
start-upper-pathogenB-pycor
-10
1
-4
1
1
NIL
HORIZONTAL

SLIDER
679
403
851
436
pathogenB-harm-hostcells
pathogenB-harm-hostcells
0
50
20
1
1
NIL
HORIZONTAL

SLIDER
679
439
852
472
pathogenB-take-up-energy
pathogenB-take-up-energy
0
50
20
1
1
NIL
HORIZONTAL

SLIDER
679
476
870
509
pathogenB-survive-low-energy
pathogenB-survive-low-energy
10
40
40
1
1
NIL
HORIZONTAL

SWITCH
678
514
845
547
pathogenB-wander?
pathogenB-wander?
1
1
-1000

MONITOR
682
202
833
247
count stars (pathogen B)
count stars
17
1
11

TEXTBOX
870
492
1020
510
NIL
11
0.0
1

PLOT
939
497
1139
647
Pathogen B
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot count stars"

MONITOR
1149
501
1295
546
Sum pathogen B energy
sum [energy] of stars
1
1
11

MONITOR
1150
551
1302
596
Mean pathogen B energy
mean [energy] of stars
3
1
11

SLIDER
232
650
404
683
regional-amputationB
regional-amputationB
0
30
0
1
1
NIL
HORIZONTAL

SWITCH
678
552
868
585
auto-infect-with-pathogenB?
auto-infect-with-pathogenB?
1
1
-1000

SWITCH
5
547
199
580
auto-systemic-stressor-slow?
auto-systemic-stressor-slow?
1
1
-1000

MONITOR
1147
440
1282
485
Max pathogen energy
max [energy] of triangles
3
1
11

MONITOR
1148
125
1307
170
Instanteous energy deficit
326700 - ((sum [energy] of circles) + (sum [energy] of flags))
0
1
11

SWITCH
5
581
199
614
auto-systemic-stressor-cycle?
auto-systemic-stressor-cycle?
1
1
-1000

SWITCH
5
616
215
649
auto-systemic-stressor-hi-cycle?
auto-systemic-stressor-hi-cycle?
1
1
-1000

MONITOR
1149
285
1303
330
Cumulative energy deficit
cumulative-energy-deficit
0
1
11

TEXTBOX
1307
290
1457
327
*
30
104.0
1

TEXTBOX
1312
141
1462
161
*
16
104.0
1

@#$#@#$#@
## NOTE

Note from Feb 14, 2014: This is an early version of the host-versus-pathogen (infection) model, but it has 4 types of automatic-systemic stressor (fast, slow, cycle, and hi-cycle) which were tested at a setting of 60 (set within the code). The subsequent version omitted all of the auto-systemic-stressors except the fast version, which subsequently became known as auto-systemic-stressor-hostcell# (to be distinguished from a later auto-systemic-stressor-pathogen#).

The text of this Info section  and the code has not been cleaned up in this version.

-Edmund K. LeGrand


## HOST-VERSUS-PATHOGENS MODEL: USE OF STRESSORS FOR DEFENSE ##

## WHAT IS IT?

This is a host-versus-pathogens model that explores how the host can use stressors (harmful conditions) to fight pathogens. The struggle, from the host's viewpoint, is built around harming the pathogens relatively more than the host. In this program energy is used as the currency for survival and replication by the host cells and pathogens.  

The program models life-threatening infections, where the host uses stressors to fight an infection but may be killed by being overwhelmed by the pathogens or by its own vigorous defense (sepsis). The host has a number of potential defenses, from invoking: 1) direct toxicity to the pathogen, 2) localized stressful conditions for all nearby cells, 3) regional stressful conditions, to 4) systemic stressors which harm the entire host including the pathogen cells contained within. In the program the pathogen phenotype can be varied according to 1) the amount of harm pathogen cells can do to adjacent host cells, 2) the amount of energy they can obtain when adjacent to host cells, 3) the amount of energy they need to survive, and 4) whether or not they can wander to adjacent vacant spaces. The pathogens can be thought of as a large extracellular pathogens (comparable in size and energy requirements as a host cell) or as infected cells taken over by pathogens. Similarly, the pathogens can be viewed as tumor cells.

**Levels of Host Defense**

1) The "local-host-directed" defense mechanism models the standard view of immune defenses where defense is directed specifically at the pathogen with little, if any, host damage (e.g., phagocytosis with killing within the phagolysosome, and direct pathogen killing via complement, antimicrobial peptides, antibodies, etc.). The only cost to the host is the energy involved in creating this defense. In this program there is no additional cost for "local-host-directed" defense since these costs are already incorporated into the routine functioning of the host cells. Note that in this program there are no specialized immune cells; rather, any host cell has the ability to respond to the pathogens.

2) The "local-host-stressor" defense component is typically ignored in discussions of host defense. However, this program was developed to show how even non-specific self-harm can be utilized in host defense. In inflammatory sites all local cells, including pathogens, can be harmed by a variety of non-specific stressors such as: lactic acidosis, hypoxia, reduced energy-providing nutrients such as glucose and  glutamine, reduced iron and zinc, and even slightly increased local heat (beyond that caused by vasodilation and fever) (Refs). Inflammatory cells utilize both glucose and glutamine, key sources of nutrition for pathogens and host cells, in what appears to be a deliberately wasteful manner (Refs). The waste of glucose contributes to the local lactic acidosis. In the program, "local-host-stressors" reduces energy from all cells adjacent to any host cell which is in contact with a pathogen.

3) Regional stressors are modeled by two mechanisms. The first is similar to "local-host-stressor" in that it reduces energy from all cells within 5 spaces of the contact between a host cell and a pathogen. This roughly models the reduced blood flow in zones of inflammation due to thrombosis (a feature co-induced by inflammatory mediators), extracellular fibrin formation, and edema (vascular leakage leading to tissue compression) (Alcock J, Brainard AH. Hemostatic containment - an evolutionary hypothesis of injury by innate immune cells. _Med. Hypotheses_ 2008, 71:960-8). An additional regional stressor, "regional-amputation", reduces energy to a 6 x 16 space region in the lower right corner of the screen/host. This is comparable to the host's abandonment of a region of tissue through amputation. In plants the hypersensitive response involves programmed cell death of host cells in a zone around an infection to deprive the pathogen of nutrients. 

4) "Systemic-host-stressor" models endogenous stressors of the acute-phase response, such as fever, nutrient restriction, anemia/hypoxia, etc. which affect all cells within the host. By analogy, it models exogenous systemic therapies which have substantial host toxicity.

It is important to note that in this program, except for "local-host-directed", _the defense offered by every host-induced stressor harms host cells as much as pathogens.  Any efficacy is due to the pathogens being relatively more vulnerable to the stressor than the host cells or the host as a whole._  The primary vulnerability of the pathogens is that upon replication, they must split their energy between the mother and daughter cells, such that both cells have less energy reserves to weather stressful events. There are 3 outcomes (besides stalemate) in this program: 1) the host wins by eliminating all pathogens, 2) the pathogen wins by overrunning and killing the host, or 3) both lose when the host dies because of too vigorous a defense (sepsis).

## HOW IT WORKS

The screen starts with 1081 host cells (1056 cyan circles and 25 orange flags) and with 8 pathogens (red triangles) along the surface (though the number and location of the pathogens can be modified with sliders). These cells (turtles) represent an organism with a small infection at its surface. Each cell sits on one of the 1089 green spaces (patches), representing extracellular space. The cells derive approximately 14 units of energy per time interval (tick) and expend approximately 10 units of energy per tick. The randomness of the model is based on slight randomness in the amount of energy received and spent by each cell at each tick.

The central 25 orange flags represent "key host cells" which are unable to replicate, but which can store more energy than the cyan circles ("host cells", which are able to replicate if they have enough energy and are adjacent to a vacant space). The key host cells represent a critical organ such as the heart or brain. In this program the key host cells have no function, but enough of them must survive for the host to survive (here more than 20 key host cells must survive). They are centrally located so as to be less vulnerable to pathogen attack from the surface of the organism. 

Vulnerability of the cells depends on energy, related to differences in their parameters.

**Parameters of host cells (cyan circles):**

start-energy = 120 units  (energy the program starts with)
birth-energy = 90 units  (energy that daughter cells start with in adjacent vacant space)
max-energy = 300 units   (energy above this value can't be stored)
energy-to-survive = 40 units
Replication can occur if there is a vacant space in one of the 4 neighboring spaces and
  if they have at least twice the birth-energy (i.e., 180 units). Replication              splits the energy between the mother and daughter cells.
At least 700 of the original 1056 host cells must be present for the host to survive.
There is an on-off switch which allows host cells to move to vacant adjacent spaces without replicating.

**Parameters of key host cells (orange flags):**

start-energy = 120 units  (energy the program starts with)
max-energy = **500** units   (energy above this value can't be stored)
energy-to-survive = **30** units 
**No replication is possible.**
At least 21 of the original 25 key host cells must be present for the host to survive.

**Parameters of pathogens (red triangles):**

start-energy = 120 units  (energy the program starts with)
birth-energy = **60** units  (energy that daughter cells start with in adjacent vacant space)
max-energy = **200** units   (energy above this value can't be stored)
energy-to-survive = 40 units. To enhance virulence this can be lowered with a slider        down to 10 units, with the maximum energy stored being 160 more units. 
Replication can occur if there is a vacant space in one of the 4 neighboring spaces and
   if they have at least twice the birth-energy (i.e., **120** units). Replication          splits the energy between the mother and daughter cells.

Depending on the pathogen attributes (see below), pathogens can reduce energy of adjacent host cells and/or can gain energy when adjacent to host cells. A switch allows pathogen cells to wander in vacant spaces without replicating.


**Rationale for selected parameters**

The energy-to-survive for the key host cells is set lower than for the host cells since they are irreplaceable. This is consistent with the resistance to apoptosis of cardiomyocytes and neurons. Also to compensate for their irreplaceability, they are allowed to store more energy.

The energy-to-survive and start-energy of the pathogens is set the same as for host cells (though the pathogens' energy-to-survive can be lowered to enhance their virulence). This permits demonstration that other differences in susceptibility to stressors can be used for host defense. Additionally, the pathogens could be modeled as infected host cells or host-derived tumor cells having similar basic physiology as host cells.

Pathogens are adapted for rapid replication, hence they require less energy to replicate and have less need for energy storage than host cells. This rapid replication is their main vulnerability in this program.

Starting with 8 pathogens along the edge is the recommended starting point. It is harder to get an infection started with smaller numbers of pathogens, especially since chance plays a larger role. Additionally, with smaller numbers of pathogens there is more contact with host cells and exposure to their pathogen-harming defenses. 

The host wins when the pathogen is eliminated and it has sufficient host cells and key host cells to survive. It is possible to adjust the stressor levels such that the pathogen numbers can be kept under control indefinitely, but the host will never have its full number of host cells and thus will remain impaired and vulnerable to other threats, besides the threat of pathogen re-emergence.  


**_Pathogen Phenotypes and Host Defenses_** 

**Pathogen Phenotypes** 

**pathogen-harm-hostcell** This slider harms host cells and key host cells that are in contact with the pathogen by reducing energy to those cells. A setting of 10 or 20 is recommended for starters. 

**pathogen-take-up-energy** This slider adds energy to pathogens that are in contact with a host cell, but does not harm the host cells. it is recommended that the setting of this slider be kept the same as that for "pathogen-harm-hostcell" since in real-life a pathogen would tend to harm the adjacent host cells and gain energy in the process. If this slider is used while "pathogen-harm-hostcell" is set to 0, the number of pathogens will be unchanged, with all cells getting to their maximum (or temporarily more, based on the how the program is coded). "pathogen-harm-hostcell" must be set to at least 2 for these high-energy pathogens to be able to break out and spread.

**pathogen-survive-low-energy** With this slider, the lower value of 10 provides a little more pathogenicity by letting the pathogens survive down to 10 units of energy, rather than the default of 40 units (same as the host cells). This can be demonstrated at settings of "pathogen-harm-hostcell" = 20, "pathogen-take-up-energy" = 20, "pathogen-wander?" = on, "local-host-directed" = 10, and "local-host-stressor" = 10. At the default setting of 40, the host typically wins (pathogens destroyed), while at 10, the pathogen typically wins (pathogens overrun the host).

**pathogen-wander?** This switch allows pathogens to randomly move to vacant adjacent spaces. Without it being on, only the pathogen daughter cells can to an adjacent vacant space upon replication. The ability to wander lets virulent pathogens become more aggressive, while less virulent pathogens become more vulnerable because of their more frequent contact with more aggressive host cells.   

When unopposed (all host defenses set at 0) the full extent of pathogenicity leads to the pathogens winning (overrunning the host to the point of the host dying) in 65-70 ticks.

**Host Defenses**

The host defenses are ordered from most specific to least specific.

**local-host-directed** This slider removes energy units from adjacent pathogens. A recommended setting is half of the "pathogen-harm-hostcell" setting.  

**local-host-stressor** This slider removes energy units from all cells within a radius of 1 from the host cell in contact with a pathogen. A recommended setting is that it be the same as "local-host-directed". Interestingly, this defense is comparable to or slightly more effective than "local-host-directed".

**regional-host-stressor** When a host cell is within 2 spaces of a pathogen, this slider reduces energy by a percentage for all cells in a radius of 5 spaces from the host cell. This creates fairly wide zones of destruction. It is especially effective at the start of the infection.

**regional-amputation** This slider reduces energy by a percentage for all cells in the lower right region of the host. While it is highly effective early on, once pathogens have spread from this region, it has little value.

**systemic-host-stressor** This slider reduces energy by a percentage for all cells in the host. This slider allows modeling of sepsis, the condition where the host defense, rather than the pathogen, has killed the host. In this program a failure of the other defenses simply allows overgrowth of the pathogens leading to host death. However, this slider has the ability to kill the host while only a few pathogens are present. This slider can reverse a losing battle, but it is difficult to use effectively, and it is suggested that the action be slowed down with the slider at the top of the screen.     

**auto-systemic-stressor** This switch was developed from a generally winning strategy of using the "systemic-host-stressor" slider. The host is prevented from killing itself (sepsis) while trying to maximize the harm that is done to the pathogens. Once there is a significant threat (when the number of host cells drops from 1056 to 1040 AND the mean energy of the host cells is at least 180) the "systemic-host-stressor" slider moves to 60. This quickly drains the cells of energy, with the pathogens being relatively more affected since they've spent their spare energy on replication. When the number of host cells drops to 975, the slider moves back to 0, allowing all cells to recover. This will cycle again and again. Sometimes the pathogens can be eliminated with one bout of "fever", and other times it may take several cycles. However, if the pathogen is so aggressive that the host never gets more than 975 cells (or if the mean energy never reaches 180), the auto-stressor will not turn on. Then the host will die because of pathogen overgrowth. 

Because of the cell-by-cell calculations involved in running the program and creating a tick of time, there can be considerable lag and overshooting in the auto-stressor. Setting the lower safe zone closer to the required 700 surviving host cells (thus killing more pathogens as well as host cells) would lead to frequent overshooting and host death (sepsis). 

_In the course of playing with the "systemic-host-stressor" slider, it became clear that it is best to rapidly apply the systemic stress. It is anxiety-producing to slowly increase the systemic stress while watching the pathogens continue to spread, thus making success that much more difficult. Likewise, a rapid reprieve from systemic stress is useful when one watches more and more host cells dying. Even with the instantaneoous stress and relief from stress in this program, there is still a lag time for the effects to be apparent. This was not apparent to me before using the program._

**hostcell-wander?** When on, this switch allows host cells to wander to a vacant adjacent cell. It has limited utility. If host cells are set to be aggressive relative to the pathogens, then it will have some efficacy.


## THINGS TO NOTICE

The primary vulnerability of the pathogens in this program is that they replicate rapidly. In doing so, they convert one well-energized cell into two cells with low energy levels. This reflects real-life since replication requires acquiring the material and energy resources to duplicate oneself. Additionally, DNA is vulnerable during replication, as are newly forming and folding proteins. The reasonable strategy during stressful conditions is to inhibit growth and reproduction. 

The only movement of cells in this program is reproduction into vacant spaces (only one cell is permitted in a space). Key host cells, in not replicating, never move. One of the pathogen strategy choices allows pathogen cells to move to empty adjacent spaces even when not replicating. Under most situations movement of pathogens can enhance pathogenicity by increasing the spread of the infection. However, if the pathogens are less toxic than the host cells, the increased contact with host cells will tend to be harmful to the motile pathogens. Likewise for movement of host cells when that option is turned on.

None of the host strategies directly monitor the pathogen number. The auto-host-stressor strategy monitors mean energy of host cells and host cell number before applying a fixed systemic-stressor-level (60 or 0). It is not clear how host strategies could be improved by monitoring pathogen number (except to know that all pathogens were killed). Mean pathogen energy would be useful in deciding whether or not systemic stressors would be effective. In this program it would be useful to automatically decide on the strategy of applying regional-amputation. Obviously, this drastic strategy is only effective before the pathogen has spread beyond the affected region.
 
## THINGS TO TRY

For a given pathogen phenotype, one can be an active participant by adjusting the sliders for the various host defenses. Most notably one can play with the systemic-host-stressor switch to try to use the stressors of the acute-phase response (fever, etc.). The auto-systemic-stressor switch was developed to simplify host defense without killing the host from excessive stress. The settings were developed through trial-and-error and have not been optimized, so it should be possible to improve upon it.

The effect of dual infections could be modeled by programming in an additional pathogen with a differing phenotype. For a mouse model of concomitant infection see Lundqvist J, et al. Concomitant infection decreases the malaria burden but escalates relapsing fever borreliosis. _Infect. Immun._ 2010, 78:1924-30.  

## NETLOGO FEATURES

 

## CREDITS AND REFERENCES



## COPYRIGHT AND LICENSE

Feb 14, 2014 to Edmund K. LeGrand and Judy Day 

## EXTENDING THE MODEL
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
8
Circle -7500403 true false 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

monster
false
0
Polygon -7500403 true true 75 150 90 195 210 195 225 150 255 120 255 45 180 0 120 0 45 45 45 120
Circle -16777216 true false 165 60 60
Circle -16777216 true false 75 60 60
Polygon -7500403 true true 225 150 285 195 285 285 255 300 255 210 180 165
Polygon -7500403 true true 75 150 15 195 15 285 45 300 45 210 120 165
Polygon -7500403 true true 210 210 225 285 195 285 165 165
Polygon -7500403 true true 90 210 75 285 105 285 135 165
Rectangle -7500403 true true 135 165 165 270

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.0.4
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="experiment" repetitions="1" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>count turtles</metric>
    <enumeratedValueSet variable="automatic-systemic-stressor-therapy?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pathogen-strategy">
      <value value="&quot;harm-host-cells&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="show-energy?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="stressor-level">
      <value value="0"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="experiment" repetitions="6" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="1000"/>
    <exitCondition>count circles &lt;= 700
or count flags &lt;= 20
or count triangles = 0</exitCondition>
    <metric>count circles</metric>
    <metric>count flags</metric>
    <metric>count triangles</metric>
    <metric>ticks</metric>
    <metric>mean [energy] of circles</metric>
    <metric>mean [energy] of flags</metric>
    <metric>mean [energy] of triangles</metric>
    <enumeratedValueSet variable="automatic-systemic-stressor-therapy?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pathogen-strategy">
      <value value="&quot;harm-host-cells&quot;"/>
      <value value="&quot;siphon-energy&quot;"/>
      <value value="&quot;move-and-harm&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="stressor-level">
      <value value="0"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="experiment" repetitions="1" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>count turtles</metric>
    <enumeratedValueSet variable="auto-systemic-stressor-slow?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="auto-systemic-stressor?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pathogen-wander?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="regional-host-stressor">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pathogenB-harm-hostcells">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pathogenB-survive-low-energy">
      <value value="40"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="start-lower-pathogenB-pycor">
      <value value="-11"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="systemic-host-stressor">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="regional-amputationB">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="start-lower-pathogen-pxcor">
      <value value="14"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="hostcell-wander?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pathogenB-take-up-energy">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="start-lower-pathogenB-pxcor">
      <value value="-17"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pathogenB-wander?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pathogen-survive-low-energy">
      <value value="40"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="start-upper-pathogen-pycor">
      <value value="-5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pathogen-harm-hostcells">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="start-upper-pathogenB-pycor">
      <value value="-4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="local-host-directed">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="regional-amputation">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="auto-infect-with-pathogenB?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="auto-systemic-stressor-cycle?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="start-upper-pathogen-pxcor">
      <value value="16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="start-upper-pathogenB-pxcor">
      <value value="-15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="start-lower-pathogen-pycor">
      <value value="-10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="show-energy?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="auto-systemic-stressor-hi-cycle?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="local-host-stressor">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pathogen-take-up-energy">
      <value value="20"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="auto-systemic-stressor" repetitions="6" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="1000"/>
    <exitCondition>count circles &lt;= 700
or count flags &lt;= 20
or count triangles = 0</exitCondition>
    <metric>count circles</metric>
    <metric>count flags</metric>
    <metric>count triangles</metric>
    <metric>ticks</metric>
    <metric>cumulative-energy-deficit</metric>
    <enumeratedValueSet variable="auto-systemic-stressor?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pathogen-harm-hostcells">
      <value value="5"/>
      <value value="20"/>
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pathogen-take-up-energy">
      <value value="5"/>
      <value value="20"/>
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pathogen-survive-low-energy">
      <value value="10"/>
      <value value="40"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pathogen-wander?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="systemic-host-stressor">
      <value value="0"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="auto-systemic-stressor-slow" repetitions="6" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="1000"/>
    <exitCondition>count circles &lt;= 700
or count flags &lt;= 20
or count triangles = 0</exitCondition>
    <metric>count circles</metric>
    <metric>count flags</metric>
    <metric>count triangles</metric>
    <metric>ticks</metric>
    <metric>cumulative-energy-deficit</metric>
    <enumeratedValueSet variable="auto-systemic-stressor-slow?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pathogen-harm-hostcells">
      <value value="5"/>
      <value value="20"/>
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pathogen-take-up-energy">
      <value value="5"/>
      <value value="20"/>
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pathogen-survive-low-energy">
      <value value="10"/>
      <value value="40"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pathogen-wander?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="systemic-host-stressor">
      <value value="0"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="auto-systemic-stressor-cycle" repetitions="6" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="1000"/>
    <exitCondition>count circles &lt;= 700
or count flags &lt;= 20
or count triangles = 0</exitCondition>
    <metric>count circles</metric>
    <metric>count flags</metric>
    <metric>count triangles</metric>
    <metric>ticks</metric>
    <metric>cumulative-energy-deficit</metric>
    <enumeratedValueSet variable="auto-systemic-stressor-cycle?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pathogen-harm-hostcells">
      <value value="5"/>
      <value value="20"/>
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pathogen-take-up-energy">
      <value value="5"/>
      <value value="20"/>
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pathogen-survive-low-energy">
      <value value="10"/>
      <value value="40"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pathogen-wander?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="systemic-host-stressor">
      <value value="0"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="auto-systemic-stressor-hi-cycle" repetitions="6" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="1000"/>
    <exitCondition>count circles &lt;= 700
or count flags &lt;= 20
or count triangles = 0</exitCondition>
    <metric>count circles</metric>
    <metric>count flags</metric>
    <metric>count triangles</metric>
    <metric>ticks</metric>
    <metric>cumulative-energy-deficit</metric>
    <enumeratedValueSet variable="auto-systemic-stressor-hi-cycle?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pathogen-harm-hostcells">
      <value value="5"/>
      <value value="20"/>
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pathogen-take-up-energy">
      <value value="5"/>
      <value value="20"/>
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pathogen-survive-low-energy">
      <value value="10"/>
      <value value="40"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pathogen-wander?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="systemic-host-stressor">
      <value value="0"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 1.0 0.0
0.0 1 1.0 0.0
0.2 0 1.0 0.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
