globals [ 
  farmers-x    ;;x-coordinate for the top of the farmersmarket
  farmers-y    ;;y-coordinate for the top of the farmersmarket
  supermarket-x    ;;x-coordinate for the top of the supermarket
  supermarket-y    ;;y-coordinate for the top of the supermarket
  percent-locavore
  number-trucks
  overall-emission
  overall-emission-counter
  number-laborers
  overall-economy
  overall-health-risk
]
breed [supermarkets supermarket]
breed [farmersmarkets farmersmarket]
breed [shoppers shopper]
breed [trucks truck]
breed [labors labor]
shoppers-own [
  locavore?     
  health-risk
  locavore-length
  identity
  at-store?
]
;;
;;SETUP Procedure
;;
;;This procedure sets up the world with all the characters and default variable values.
;;half of the population of shoppers starts at the top-left corner while the other half starts
;;at the bottom right corner. This is arbitrary, but helps control the characters and easier to see
;;them if they are not directly on top of one another. 
to setup
  clear-all
  reset-ticks 
  set-default-shape shoppers "person service"
  set-default-shape supermarkets "building store"
  set-default-shape farmersmarkets "house ranch"
  set-default-shape trucks "truck"
  set-default-shape labors "person farmer"
  
  set percent-locavore  (locavore-propensity / 100)   ;;percent of the population that will be locavore at the end of the time period
  set number-trucks floor (population * .05)         ;;there is a new truck (shipment of food to the supermarket) for every 5% of the population non-locavore
  set overall-emission (number-trucks * 5845)        ;;every truck averages 5845 tons of CO2 emissions. because the world starts with everyone non-locavore this is default. 
  set number-laborers 1    ;;we will assume the farmer is running the show on his own
  set overall-economy 1    ;;because there are no new employees the economy is at 1
  
  create-shoppers (population / 2)
    [setxy -16 14
    set size 1.5
    set color 37
    set heading 135
    set health-risk .20   ;;The average american has a 20% risk of hypertension (high-blood pressure)
    set at-store? false
    set locavore? false
    set locavore? (who < population * .0125)     ;;On average only 1.25% of the population is locavore
    if locavore? [
      set color green]]
  create-shoppers (population / 2) 
    [setxy 16 -14
    set size 1.5
    set color 37
    set heading 315
    set health-risk .20    ;;The average american has a 20% risk of hypertension (high-blood pressure)
    set at-store? false
    set locavore? false
    set locavore? (who < population * .0125)     ;;On average only 1.25% of the population is locavore
    if locavore? [
      set color green]]
  
  create-world
end
;;Helper function that creates some of the additional characters in the world. 
to create-world
  set farmers-x -12
  set farmers-y -8
  set supermarket-x 8
  set supermarket-y 10
  
  ask patches 
    [set  pcolor (green + 1)
    
    if (pxcor < farmers-y) and (pycor < farmers-x) 
      [set pcolor green]
    if (pxcor > supermarket-x) and (pycor > supermarket-y)
      [set pcolor blue]] 
  
  create-supermarkets 1 
    [setxy 13 13
    set color 4  
    set size 6]
  
  create-farmersmarkets 1 
    [setxy -13 -14
    set color 104
    set size 5] 
  
  create-trucks number-trucks
   [setxy 9 (floor ((random-float 1) * 4) + 12)    ;;random number between 8 and 16     ;;random number between 9 and 14
     set color white
     set size 2
     set heading 315]
end

;;
;; GO Procedure
;;
;;This is starts the moving behaviors and at every click calculates and stores information
;;about the individual shoppers as well as the entire community. The model is only intended to 
;;last 100 ticks. This is done so that the user gets a realistic idea of how a community could
;;benefit from locally grown foods within a given time period. 
to go
  tick
  if all? shoppers [locavore?]
    [stop]
  
  if ticks >= 100     
    [stop]
  
  ask shoppers 
    [start-moving  
    stop-for-food
    choose-lifestyle
    calculate-health-risk]        
  
  ask trucks
     [trucks-moving
      calculate-environment-impact]  
 calculate-economy   
 total-health-risk 
 ;update-plot-locavore
 update-plot-healthrisk
 update-plot-environment
 update-plot-economy
end 
;This procedure moves the shoppers and relocates them to the road once they are done shopping. 
to start-moving
  if not can-move? 1 [right 180]
  if who >= random population [stop]   ;; delay some of the shoppers from heading out
  if at-store? 
    [ifelse random-float 100 < 50      ;;50% of the shoppers will return to the top-left corner of the world after shopping
      [setxy 16 -14                      
        set at-store? false ]
      [setxy -14 16                   ;;50% of the shoppers will start from the bottom-right corner of the world after shopping
        set at-store? false]]
  forward random 2
end

;;This procedure is similar to the start-moving, but it is for the trucks. They are confined to a small space near the 
;;grocery store and only move up and down. 
to trucks-moving
  ifelse ycor > 14 and ycor > 16
      [forward random 2]
      [right 180
        forward random 2]
end
;;This procedure checks if the shopper is currently at a store (based on its location in the world
;;If it at a store the procedure will stop; if it is not at the store this is where the shopper will decide
;;based on his lifestyle choice whether to go to the farmer's market or to the supermarket
to stop-for-food
  let sx  floor ((random-float 1) * 8) + 9    ;;random number between 8 and 16 - possible x-cors for the supermarket
  let sy  floor ((random-float 1) * 6) + 11   ;;random number between 10 and 16 - possible y-cors for the supermarket 
  let fx  floor ((random-float -1) * 8) - 8   ;;random number between -8 and -16 - possible x-cors for the farmer's market
  let fy  floor ((random-float -1) * 4) - 12   ;;random number between -12 and -16 - possible y-cors for the farmer's market
    
    ifelse at-store? 
    [stop]
    [ifelse locavore?                   
      [if random-float 100 < 10       ;;10% of the locavores will go to the farmers market at every click 
        [setxy fx fy
        set at-store? true]]
      [if random-float 100 < 2         ;;2% of the non-locavores will go to the supermarket at every click
        [setxy  sx sy
          set at-store? true]]]
end

;;Currently only 1.25% of the population are locavores. It is estimated that this number is growing
;;at about 2% per year. For the sake of this model, every 100 tick marks will represent one year. Based
;;on the locavore-propensity the user sets, we can see increase the number of localvores linearly over
;;that one year.  
to choose-lifestyle
  let x count shoppers with [locavore?]
  
  if x < percent-locavore * population   
    [if remainder ticks 10 = 0 
      [if random-float 100 < (percent-locavore * population / 10) and not at-store? 
        [set locavore? true
        set locavore-length locavore-length + 1]]
        if locavore?
        [set color green]]
end

;;Health risks are associated with consumption of processed foods. Appel (1997) clinical-trial 
;;results showed that adopting a diet consisting of locally-grown foods  reduces the risk of high blood 
;;pressure reduces by 55%. In America, the average risk of high blood pressure
;;is 20%. For locavores the risk of high blood pressure is estimated at 10%. For this project we have initated
;;the shoppers individual health risk at 20%. Shoppers who continue eat non-local foods will continue to 
;;increase their health-risk at a modest 2% increase (consistent with American Heart Association estimates for
;;average Americans)
to calculate-health-risk
  if health-risk >= 1 [stop]
  ifelse locavore? 
   [set health-risk health-risk * .55]  
   [set health-risk health-risk * 1.02]
end
to total-health-risk
    set  overall-health-risk mean [health-risk] of shoppers
end
;;The US States Environmental Protection Agency estimates that on average for every 1 mile of 
;;distance traveled to supply food products to grocery stores about 1.18 tons of combined carbon 
;;monoxide, nitrogen oxide, and sulfur dioxide are emmitted into the environment. Locally-foods
;;create 0 emmissions through transport as the foods are grown locally. The average distance travelled
;;to stock a supermarket produce section is estimated to be 5845 miles. For every 5% of the population
;;that eats locally, one truck shipment will no longer be delivered to the competing supermarket.
to calculate-environment-impact
  let x count shoppers with [locavore?]
  let y floor ((x / population) / .05 )
  
  if y = overall-emission-counter [stop] 
  
  set overall-emission (overall-emission - ((y - overall-emission-counter) * 5845) )
  set overall-emission-counter y
  
  if (number-trucks + y != population * .05)
    [if who = random number-trucks [die]
      set number-trucks number-trucks - 1 ]
end 

;;One measure of economic growth is the increase in the employees only as the sale of the products 
;;they produce generates new fresh dollars for the local region. In farmer's markets, all the sale money that
;;is gained goes directly to fresh dollars because the employees and the owners live and buy from that same community
;;as opposed to shippers, transporters, etc. where the money will be used in a different community. 
;;For this model we will assume that for each 10 new people that eat locally-grown foods, 1 new employee will be 
;;hired by the farm. Economic growth will be based on total number of new employees * a multplier. I have chosen a health
;;economic growth multiplier
to calculate-economy
   let x count shoppers with [locavore?]
   let y floor (x / 10)
   
   if y = number-laborers [stop]
     
   set overall-economy  floor( 1.2 * (overall-economy + y - number-laborers))
   set number-laborers y
   
   create-labors (y - number-laborers)
    [setxy -9 (floor ((random-float -1) * 4) - 12)    ;;random number between 8 and 16     ;;random number between 9 and 14
     set color green + 3
     set size 2]
     
end 

;;
;;PLOTS
;;
to update-plot-healthrisk
  set-current-plot "Health-Risk"
  set-current-plot-pen "health"
  plot overall-health-risk
end
to update-plot-environment
  set-current-plot "Environmental Impact"
  set-current-plot-pen "environment"
  plot overall-emission
end
to update-plot-economy
  set-current-plot "Economy Impact"
  set-current-plot-pen "economy"
  plot overall-economy
end
;to update-plot-locavore
;  set-current-plot "Locavores"
;  set-current-plot-pen "locavores"
;  plot count shoppers with [locavore?]
;end
@#$#@#$#@
GRAPHICS-WINDOW
281
10
720
470
16
16
13.0
1
10
1
1
1
0
0
0
1
-16
16
-16
16
1
1
1
ticks
BUTTON
44
135
110
168
setup
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
SLIDER
15
79
251
112
population
population
0
400
286
1
1
NIL
HORIZONTAL
BUTTON
152
135
215
168
go
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
SLIDER
14
193
249
226
locavore-propensity
locavore-propensity
0
100
26
1
1
%
HORIZONTAL
PLOT
753
19
968
218
Health-Risk
time
overall-health-risk
0.0
100.0
0.0
1.0
true
false
PENS
"health" 1.0 0 -16777216 true
MONITOR
87
382
159
427
# of trucks
number-trucks\\n
17
1
11
PLOT
753
256
969
455
Environmental Impact
time
overall-emission
0.0
100.0
0.0
10.0
true
false
PENS
"environment" 1.0 0 -16777216 true
MONITOR
66
255
190
300
Number of Locavores
count shoppers with [locavore?]
17
1
11
MONITOR
67
323
187
368
NIL
overall-economy
17
1
11
PLOT
999
116
1236
337
Economy Impact
time
overall-economy
0.0
100.0
0.0
10.0
true
false
PENS
"economy" 1.0 0 -16777216 true
@#$#@#$#@
WHAT IS IT?
-----------
This section could give a general understanding of what the model is trying to show or explain.

HOW IT WORKS
------------
This section could explain what rules the agents use to create the overall behavior of the model.

HOW TO USE IT
-------------
This section could explain how to use the model, including a description of each of the items in the interface tab.

THINGS TO NOTICE
----------------
This section could give some ideas of things for the user to notice while running the model.

THINGS TO TRY
-------------
This section could give some ideas of things for the user to try to do (move sliders, switches, etc.) with the model.

EXTENDING THE MODEL
-------------------
This section could give some ideas of things to add or change in the procedures tab to make the model more complicated, detailed, accurate, etc.

NETLOGO FEATURES
----------------
This section could point out any especially interesting or unusual features of NetLogo that the model makes use of, particularly in the Procedures tab.  It might also point out places where workarounds were needed because of missing features.

RELATED MODELS
--------------
This section could give the names of models in the NetLogo Models Library or elsewhere which are of related interest.

CREDITS AND REFERENCES
----------------------
This section could contain a reference to the model's URL on the web if it has one, as well as any other necessary credits or references.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250
airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15
arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150
box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75
bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30
building institution
false
0
Rectangle -7500403 true true 0 60 300 270
Rectangle -16777216 true false 130 196 168 256
Rectangle -16777216 false false 0 255 300 270
Polygon -7500403 true true 0 60 150 15 300 60
Polygon -16777216 false false 0 60 150 15 300 60
Circle -1 true false 135 26 30
Circle -16777216 false false 135 25 30
Rectangle -16777216 false false 0 60 300 75
Rectangle -16777216 false false 218 75 255 90
Rectangle -16777216 false false 218 240 255 255
Rectangle -16777216 false false 224 90 249 240
Rectangle -16777216 false false 45 75 82 90
Rectangle -16777216 false false 45 240 82 255
Rectangle -16777216 false false 51 90 76 240
Rectangle -16777216 false false 90 240 127 255
Rectangle -16777216 false false 90 75 127 90
Rectangle -16777216 false false 96 90 121 240
Rectangle -16777216 false false 179 90 204 240
Rectangle -16777216 false false 173 75 210 90
Rectangle -16777216 false false 173 240 210 255
Rectangle -16777216 false false 269 90 294 240
Rectangle -16777216 false false 263 75 300 90
Rectangle -16777216 false false 263 240 300 255
Rectangle -16777216 false false 0 240 37 255
Rectangle -16777216 false false 6 90 31 240
Rectangle -16777216 false false 0 75 37 90
Line -16777216 false 112 260 184 260
Line -16777216 false 105 265 196 265
building store
false
0
Rectangle -7500403 true true 30 45 45 240
Rectangle -16777216 false false 30 45 45 165
Rectangle -7500403 true true 15 165 285 255
Rectangle -16777216 true false 120 195 180 255
Line -7500403 true 150 195 150 255
Rectangle -16777216 true false 30 180 105 240
Rectangle -16777216 true false 195 180 270 240
Line -16777216 false 0 165 300 165
Polygon -7500403 true true 0 165 45 135 60 90 240 90 255 135 300 165
Rectangle -7500403 true true 0 0 75 45
Rectangle -16777216 false false 0 0 75 45
Polygon -7500403 false true 84 117 79 119 91 121 78 115 81 139
butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60
car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58
circle
false
0
Circle -7500403 true true 0 0 300
circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123
cylinder
false
0
Circle -7500403 true true 0 0 300
dot
false
0
Circle -7500403 true true 90 90 120
face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240
face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225
face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183
fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30
flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45
flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240
house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120
house ranch
false
0
Rectangle -7500403 true true 270 120 285 255
Rectangle -7500403 true true 15 180 270 255
Polygon -7500403 true true 0 180 300 180 240 135 60 135 0 180
Rectangle -16777216 true false 120 195 180 255
Line -7500403 true 150 195 150 255
Rectangle -16777216 true false 45 195 105 240
Rectangle -16777216 true false 195 195 255 240
Line -7500403 true 75 195 75 240
Line -7500403 true 225 195 225 240
Line -16777216 false 270 180 270 255
Line -16777216 false 0 180 300 180
leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195
line
true
0
Line -7500403 true 150 0 150 300
line half
true
0
Line -7500403 true 150 0 150 150
pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120
person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105
person farmer
false
0
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Polygon -1 true false 60 195 90 210 114 154 120 195 180 195 187 157 210 210 240 195 195 90 165 90 150 105 150 150 135 90 105 90
Circle -7500403 true true 110 5 80
Rectangle -7500403 true true 127 79 172 94
Polygon -13345367 true false 120 90 120 180 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 180 90 172 89 165 135 135 135 127 90
Polygon -6459832 true false 116 4 113 21 71 33 71 40 109 48 117 34 144 27 180 26 188 36 224 23 222 14 178 16 167 0
Line -16777216 false 225 90 270 90
Line -16777216 false 225 15 225 90
Line -16777216 false 270 15 270 90
Line -16777216 false 247 15 247 90
Rectangle -6459832 true false 240 90 255 300
person service
false
0
Polygon -7500403 true true 180 195 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285
Polygon -1 true false 120 90 105 90 60 195 90 210 120 150 120 195 180 195 180 150 210 210 240 195 195 90 180 90 165 105 150 165 135 105 120 90
Polygon -1 true false 123 90 149 141 177 90
Rectangle -7500403 true true 123 76 176 92
Circle -7500403 true true 110 5 80
Line -13345367 false 121 90 194 90
Line -16777216 false 148 143 150 196
Rectangle -16777216 true false 116 186 182 198
Circle -1 true false 152 143 9
Circle -1 true false 152 166 9
Rectangle -16777216 true false 179 164 183 186
Polygon -2674135 true false 180 90 195 90 183 160 180 195 150 195 150 135 180 90
Polygon -2674135 true false 120 90 105 90 114 161 120 195 150 195 150 135 120 90
Polygon -2674135 true false 155 91 128 77 128 101
Rectangle -16777216 true false 118 129 141 140
Polygon -2674135 true false 145 91 172 77 172 101
plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90
sheep
false
0
Rectangle -7500403 true true 151 225 180 285
Rectangle -7500403 true true 47 225 75 285
Rectangle -7500403 true true 15 75 210 225
Circle -7500403 true true 135 75 150
Circle -16777216 true false 165 76 116
square
false
0
Rectangle -7500403 true true 30 30 270 270
square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240
star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108
target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60
tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152
triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255
triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224
truck
false
0
Rectangle -7500403 true true 105 45 296 187
Polygon -7500403 true true 4 193 4 150 41 134 56 104 92 104 93 194
Rectangle -1 true false 105 60 105 105
Polygon -16777216 true false 62 112 48 141 81 141 82 112
Circle -16777216 true false 24 174 42
Rectangle -7500403 true true 86 185 119 194
Circle -16777216 true false 114 174 42
Circle -16777216 true false 234 174 42
Circle -7500403 false true 234 174 42
Circle -7500403 false true 114 174 42
Circle -7500403 false true 24 174 42
turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99
wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269
x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 4.1
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 1.0 0.0
0.0 1 1.0 0.0
0.2 0 1.0 0.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
