breed [anchors anchor]
breed [people person]
breed [clubs club]

globals [
  center-anchor
  goals-met-in-clubs
  goals-missed-in-clubs
  left-outs
  goals-met-personal-savings
  goals-missed-personal-savings
]

people-own [
  income-risk ;; chance of a drop in income
  security-risk ;; chance of losing personal savings
  in-club? ;; true if currently in a club
  like-to-join? ;; true if trying to join a club
  preferred-group-size ;; the membership size that has allowed the individual to meet the greatest number of goals in the past
  last-best  ;; used in setting PREFERRED-GROUP-SIZE 
  current-met ;; used in setting PREFERRED-GROUP-SIZE
  chosen-group-size ;; the membership size that a creator of a club will employ
  income-risk-estimate  ;; estimate of average risk in population  
  this-round-risk  ;; used in setting RISK-ESTIMATE
  history  ;; used in setting RISK-ESTIMATE
  contribution ;; the amount put into a club or personal savings in a given tick
  savings ;; the amount of accumulated savings since last calculation
]

clubs-own [
  daily-contribution  ;; how much individuals should put in a club daily
  members  ;; how many members are in the group
  money  ;; how much money is in the pot
  desired-members  ;; how many members the group can hold
  status ;; whether the club is full or not full
]

to setup
  ca
  ;; Create the center node anchor
    create-anchors 1 [
      set shape "circle"
      set color black
    ]
   ;; This is a convenience variable that points to the center anchor
  set center-anchor one-of anchors
  set-default-shape people "person"
  set-default-shape clubs "circle"
  ;; initialize people
  create-people 200
  ask people [
    setxy random-xcor random-ycor
    set income-risk random-normal-in-range mean-income-risk income-risk-sd 0 1
    set security-risk random-normal-in-range mean-security-risk .1 0 1
    set contribution 0   
    set preferred-group-size initial-preferred-group-size
    set last-best 0   
    set history (list income-risk)
    set income-risk-estimate income-risk
    set color scale-color red income-risk 1 0
    set in-club? false
  ] 
 set goals-met-in-clubs 0
 set goals-missed-in-clubs 0
 set goals-met-personal-savings 0
 set goals-missed-personal-savings 0
end

to-report random-normal-in-range [ the-mean the-stdev low high ]
  let val random-normal the-mean the-stdev
  if (val < low) or (val > high)
  [ report random-normal-in-range the-mean the-stdev low high ]
  report val 
end
  
to go
ifelse ((count people with [in-club? = true]) = (count people with [like-to-join? = true])) and 
       (count people with [like-to-join? = true] > 0) and 
       (count clubs with [status != "full"] = 0)
[
  save  ;; if everyone who wants to join a club has, and all clubs are full, start savings procedures
]  
[
  ;; people decide whether they would like to be in clubs or save alone
  ask people with [in-club? = false] [decide]    
  ;; one person starts a club
  ask people [set chosen-group-size round random-normal-in-range preferred-group-size 1 2 200]
  if any? people with [(in-club? = false) and (like-to-join? = true) and (chosen-group-size <= count people with [(like-to-join? = true) and (in-club? = false)])]
  [
   ask one-of people with [(in-club? = false) and (like-to-join? = true) and (chosen-group-size <= count people with [(like-to-join? = true) and (in-club? = false)])] 
     [start-club]] 
  ;; people join the club until it is full
  ask people [if count my-links > 0 [set in-club? true]]
  if any? people with [in-club? = false] [
  ask people with [in-club? = false] [maybe-join]]
  ;; clubs that never get filled are abandoned
  if any? clubs with [status != "full"] [
    ask clubs with [status != "full"] 
    [die]]
  ;; if people would like to join but there are not enough remaining people to fill a club, they will not join this round
  set left-outs people with [(in-club? = false) and (like-to-join? = true) and (chosen-group-size > count people with [(like-to-join? = true) and (in-club? = false)])]
  ask left-outs
    [set like-to-join? false]
  ;; people update IN-CLUB? variable
  ask people with [count my-links = 0]
  [set in-club? false]
]
end

to decide ;; person procedure
  ifelse (1 - income-risk-estimate) > (1 - income-risk - security-risk) 
  [
    set like-to-join? true ]
  [
    set like-to-join? false ]
end

to maybe-join
  if any? clubs with [status = "not-full"]
  [
    if (like-to-join? = true) and (in-club? = false)
    [join] 
  ]
  ask clubs 
  [
   repeat 10 
    [layout-spring link-neighbors my-links .2 2 .5]
  ]
  layout-spring clubs ( [my-links] of center-anchor ) .2 14 .5    
end

to start-club ;; person procedure
 ask patch-here 
 [
  let creator myself
   sprout-clubs 1 
   [
    ifelse [ycor] of creator <= 14 
    [
     setxy [xcor] of creator [ycor] of creator + 2]
    [
     setxy [xcor] of creator [ycor] of creator - 2]
    set color yellow
    set size 1.3
    set desired-members [chosen-group-size] of creator
    set daily-contribution 10 / (1 - [income-risk-estimate] of creator)
    set status "not-full"
    create-link-with creator
   ]
 ]
 set in-club? true
end 

to join ;; person procedure
   create-link-with one-of clubs with [status = "not-full"]
   set in-club? true
   ask one-of link-neighbors 
   [
    set members count my-links
    if members >= desired-members 
    [
     set status "full"
    ]
   ]
end

to save
 update-plots   
 ;; personal saving procedure  
  ask people with [in-club? = false] 
  [
    ifelse random-float 1 < (income-risk + security-risk)
     [
      set contribution 0]
     [
      set contribution 10 / (1 - income-risk - security-risk)  ] 
    set savings savings + contribution
    set label round savings 
    set label-color white
    if (ticks mod 30 = 0) and (ticks != 0) 
    [
      calculate-savings
    ]
  ]  
 ;; group saving procedure 
  ask clubs 
  [
   ask link-neighbors 
   [
    ifelse random-float 1 < income-risk 
    [
     set contribution 0]
    [
     set contribution [daily-contribution] of one-of link-neighbors]
   ]
   ;; money in the club increases by the amount of its members contributions
   set money money + sum [contribution] of link-neighbors
   ;; visually show money in club
   set label round money 
   set label-color black
   ;; every 30 days, divide up the money 
   if (ticks mod 30 = 0) and (ticks != 0) 
   [
    split-prize
   ]
  ]
 tick
end

to split-prize
  let prize (money / members)
  ask link-neighbors 
  [
   ;; calculate risk estimate for this round, add that to history, recalculate INCOME-RISK-ESTIMATE
   set this-round-risk 1 - (prize / (30 * [daily-contribution] of one-of link-neighbors))
   if (this-round-risk > 1) or (this-round-risk < 0) [ 
     show this-round-risk 
   ]
   set history fput this-round-risk history
   set income-risk-estimate mean history  
   ;; if the prize is >= goal size: update globals, stay in club
   ifelse [prize] of myself >= 300
   [
    set goals-met-in-clubs goals-met-in-clubs + 1
    set current-met current-met + 1] 
   ;;if the prize is < goal size: update globals, update people variables, leave club
   [
    set goals-missed-in-clubs goals-missed-in-clubs + 1
    set in-club? false
    if current-met >= last-best [
    set preferred-group-size [desired-members] of one-of link-neighbors
    set last-best current-met]
    set current-met 0 
    set contribution 0
    ask my-links
    [
     die]
    ]]  
  ;; club updates MEMBERS and STATUS
  ;; if people have left the club, those remaining update people variables
  set members count my-links
  if members < desired-members 
  [
   set status "not-full"  
   ask link-neighbors 
    [
     if current-met >= last-best 
     [set preferred-group-size [desired-members] of one-of link-neighbors
      set last-best current-met]
     set current-met 0 
     set contribution 0
    ]
   ]
   set money 0
   set label " " 
end

to calculate-savings ;; person procedure
  ifelse savings >= 300
  [
   set goals-met-personal-savings goals-met-personal-savings + 1
  ]
  [
   set goals-missed-personal-savings goals-missed-personal-savings + 1
  ]   
  set contribution 0
  set savings 0
  set label " "
end

to update-plots 
   set-current-plot "Number of People"
   set-current-plot-pen "in-clubs"
   plot count people with [in-club? = true]
   set-current-plot-pen "saving-alone"
   plot count people with [in-club? = false]
   set-current-plot "% Goals Met"
   set-current-plot-pen "in-clubs"
   if (goals-met-in-clubs + goals-missed-in-clubs) != 0 
    [plot 100 * (goals-met-in-clubs / (goals-met-in-clubs + goals-missed-in-clubs))]
   set-current-plot-pen "saving-alone"
   if (goals-met-personal-savings + goals-missed-personal-savings) != 0
    [plot 100 * (goals-met-personal-savings / (goals-met-personal-savings + goals-missed-personal-savings))]
end
 
@#$#@#$#@
GRAPHICS-WINDOW
378
10
797
450
16
16
12.4
1
10
1
1
1
0
0
0
1
-16
16
-16
16
0
0
1
ticks

BUTTON
202
43
268
76
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

BUTTON
279
43
342
76
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL

SLIDER
5
18
177
51
mean-income-risk
mean-income-risk
0
1
0.15
.01
1
NIL
HORIZONTAL

SLIDER
4
59
177
92
mean-security-risk
mean-security-risk
0
1
0.1
.01
1
NIL
HORIZONTAL

PLOT
43
138
340
295
Number of People
NIL
NIL
0.0
10.0
0.0
10.0
true
true
PENS
"in-clubs" 1.0 0 -13791810 true
"saving-alone" 1.0 0 -2674135 true

PLOT
43
297
340
443
% Goals Met
Time
NIL
0.0
10.0
0.0
10.0
true
true
PENS
"in-clubs" 1.0 0 -13791810 true
"saving-alone" 1.0 0 -2674135 true

SLIDER
183
99
372
132
initial-preferred-group-size
initial-preferred-group-size
0
20
7
1
1
NIL
HORIZONTAL

SLIDER
4
99
176
132
income-risk-sd
income-risk-sd
0
1
0.3
.01
1
NIL
HORIZONTAL

MONITOR
17
462
216
507
avg. income risk of club members
mean [income-risk] of people with [in-club? = true]
3
1
11

MONITOR
17
516
214
561
avg. income risk of personal savers
mean [income-risk] of people with [in-club? = false]
3
1
11

MONITOR
268
374
328
419
left-out
count left-outs
0
1
11

MONITOR
227
461
338
506
avg. group size
mean [members] of clubs
3
1
11

@#$#@#$#@
WHAT IS IT?
-----------
This model simulates the use of informal savings clubs as a buffer against risk. Savings and credit cooperatives play a prevalent role in the informal economy of the developing world, and there is a great amount of diversity in the forms these groups take. Some involve loaning funds out to non-members at an interest rate, such that members do see a positive return on their deposits. However there are also well-documented instances of people creating and joining the simple type that is modeled here, in which net returns are often expected to be negative. In asking why this occurs, some have proposed that savings clubs may simply be the best among bad options. I explore this question by giving individuals a choice between saving in a club and on their own. Club members risk losing money if others default, but personal savers face a host of other problems--temptation spending, theft, pressure from family members--all of which are captured in the variable "SECURITY-RISK". By varying these dimensions, we can understand the conditions that incentivize people to join informal savings clubs, and the conditions that allow them to be most effective. 


HOW IT WORKS
------------
In the model, everyone is trying to achieve the goal of saving 300 units in 30 ticks (we can think of this as $300 dollars in 30 days, although real savings goals are likely to be either smaller or over a longer time period).

People are initialized with individual values for INCOME-RISK and SECURITY-RISK, drawn from a random distribution around means set by sliders. Color is scaled on INCOME-RISK--the darker the red the higher the risk. Each person predicts originally that the mean INCOME-RISK for the population is equal to his own. People also start out with a PREFERRED-GROUP-SIZE that is set by a slider. An individual would like to join a club if (1 - INCOME-RISK-ESTIMATE) > (1 - INCOME-RISK - SECURITY-RISK), which is true for everyone starting out because INCOME-RISK-ESTIMATE initially equals INCOME-RISK.  A random person will create a club, set that club's DESIRED-MEMBERS based on his PREFERRED-GROUP-SIZE. He sets the club's DAILY-CONTRIBUTION such that the club's expected payout in 30 days is $300: 10 / (1 - INCOME-RISK-ESTIMATE). People join this club until it's full, and the process repeats until the people who are left over either would not like to join a club (although this wouldn't happen in the first round), or are too few to make a club with anyone's preferred membership size. People for whom the latter is the case are designated "LEFT-OUTS"; they should not be misunderstood, when looking at the number of people saving in clubs and individually, as preferring to save alone. 

For the next 30 ticks, people in clubs contribute "DAILY-CONTRIBUTION" to their clubs (1 - INCOME-RISK)% of the time, and 0 otherwise. People saving alone contribute 10 / (1 - INCOME-RISK - SECURITY-RISK) to their savings (1 - INCOME-RISK - SECURITY-RISK)% of the time, and 0 otherwise. At the end of the 30 ticks, the total money that has been accumulated in each club is split up evenly between that club's members. If the amount meets or exceeds 300, those members are recorded as having met goals. Otherwise, they have missed goals and they leave the club. Personal savers simply look at their accumulated "SAVINGS" and similarly are recorded as successful if they meet 300. Club members recalculate "INCOME-RISK-ESTIMATE" as follows: each assumes that the average risk among his group is equal to 1 - PRIZE / (30 * DAILY-CONTRIBUTION), for this is the risk that would lead to the prize they received. They add this estimate to their history and calculate a new average. The process of forming and joining groups then runs again.

Personal savings are displayed with white labels, and group savings are black labels. 

HOW TO USE IT
-------------
SETUP button -- sets up the model by creating the agents.

GO button -- runs the model.

INITIAL-PREFERRED-GROUP-SIZE slider -- determines the club membership size that people initially prefer.

MEAN-INCOME-RISK -- determines the chance of experiencing a drop in income at any given time for the average person.

MEAN-SECURITY-RISK -- determines the chance of personal savings being depleted for the average person.

INCOME-RISK-SD -- determines the standard deviation of income risk among people.



THINGS TO NOTICE
----------------
Monitors on the interface display the average values for income-risk among club members and personal savers. It is usually the case that the value for personal savers will decline relative to the value for club members, as people who are less risky than the mean would rather save on their own than be jointly responsible with a group of people who are more likely to default than themselves. This adverse selection can have a negative impact on the effectiveness of clubs. 

The "LEFT-OUT" monitor displays the number of people at a given time who estimate that saving in a club would be less risky than saving at home, but weren't able to find an open club to join. 


THINGS TO TRY
-------------
Under certain conditions, the average group size will stay very close to INITIAL-PREFERRED-GROUP-SIZE throughout the run of the model. In other cases, there is downward pressure on the group size. Explore the risk parameters to see when group size evolves. 

Clubs are always preferred to saving alone by most members in this model. See what the sliders have to be set to for everyone to want to join a club.


EXTENDING THE MODEL
-------------------
The model could be a more accurate representation if it were extended to include other options for saving that are likely to exist alongside the type of club modeled here. A rotating credit and savings association works similarly, but one group member takes home all the contributions each month, until everyone has had a turn. To model this, you would need to account for accumulated savings being different each round, and people having preferences for when they would like the lump sum. 



@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
0
Rectangle -7500403 true true 151 225 180 285
Rectangle -7500403 true true 47 225 75 285
Rectangle -7500403 true true 15 75 210 225
Circle -7500403 true true 135 75 150
Circle -16777216 true false 165 76 116

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 4.1.3
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="security-risk" repetitions="2" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="4000"/>
    <metric>count people with [in-club? = true]</metric>
    <metric>goals-met-in-clubs / (goals-met-in-clubs + goals-missed-in-clubs)</metric>
    <metric>goals-met-personal-savings / (goals-met-personal-savings + goals-missed-personal-savings)</metric>
    <steppedValueSet variable="mean-security-risk" first="0" step="0.01" last="1"/>
  </experiment>
  <experiment name="income-risk" repetitions="2" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="4000"/>
    <metric>count people with [in-club? = true]</metric>
    <metric>goals-met-in-clubs / (goals-met-in-clubs + goals-missed-in-clubs)</metric>
    <metric>goals-met-personal-savings / (goals-met-personal-savings + goals-missed-personal-savings)</metric>
    <steppedValueSet variable="mean-income-risk" first="0" step="0.01" last="1"/>
  </experiment>
  <experiment name="income-risk-standard-deviation" repetitions="2" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="4000"/>
    <metric>count people with [in-club? = true]</metric>
    <metric>goals-met-in-clubs / (goals-met-in-clubs + goals-missed-in-clubs)</metric>
    <metric>goals-met-personal-savings / (goals-met-personal-savings + goals-missed-personal-savings)</metric>
    <steppedValueSet variable="income-risk-sd" first="0" step="0.01" last="1"/>
  </experiment>
  <experiment name="preferred-group-size" repetitions="2" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="4000"/>
    <metric>count people with [in-club? = true]</metric>
    <metric>goals-met-in-clubs / (goals-met-in-clubs + goals-missed-in-clubs)</metric>
    <metric>goals-met-personal-savings / (goals-met-personal-savings + goals-missed-personal-savings)</metric>
    <steppedValueSet variable="initial-preferred-group-size" first="0" step="1" last="20"/>
  </experiment>
  <experiment name="risk-differential-new" repetitions="4" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="4000"/>
    <metric>mean [income-risk] of people with [in-club? = true] - mean [income-risk] of people</metric>
    <steppedValueSet variable="income-risk-sd" first="0" step="0.01" last="1"/>
  </experiment>
  <experiment name="simple" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="3000"/>
    <metric>mean [income-risk] of people with [in-club? = true] - mean [income-risk] of people</metric>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 1.0 0.0
0.0 1 1.0 0.0
0.2 0 1.0 0.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
