globals [
  grid-x-b  ;; determine the size of the city blocks
  grid-y-b
  roads  
  intersections  ;; determine where the intersections are
  homes  
  destinations  
  gasstations
  gasstations_ex  ;; for making an interactive patch right next to the actual gas station patch
  gas-time  ;; for counting the time it takes for the gas station to sell out most of its fuel reserve 
  ]

turtles-own [ 
  car
  energy  ;; for counting how much gas is left in the vehicle
  homexy  ;; for storing the location of the home patch with the vehicle
  dest  ;; for storing the location of the desintation 
  fdest  ;; for storing the locaiton of the gas station 
  ]

patches-own [ 
  count-neighbors  ;; for counting the number of corresponding patches around one patch
  count-homes  
  count-gaspatch
  count-intersections
  gas-level  ;; for determining the amount of gas left and the current gas price at the gas station patch
  gas-price
  ]

to setup
  clear-all
  
  setup-globals
  setup-patches
  setup-homes
  setup-gasstation
  setup-dest
  
  set-default-shape turtles "car"
  
  ask patches [
    if pcolor = green [
      sprout 1  ;; sprout a car on each home patch
       ]             
    ]
  
  ask gasstations_ex [
    set gas-level 2000  ;; give 2000 gas "points" to each station at the start
    set gas-price random-normal 3.5 0.5  ;; price of gasoline is determine from a random distribution with mean of 1 and standard deviation of 1
    ]
  
  ask turtles [
    set energy random 300  ;; each vehicle has a random amount of gas at start, with max. value of 300
    set car turtles  
    
    set homexy min-one-of (patches with [pcolor = green]) [distance myself]  ;; location of the closest home patch is stored as the home patch for the vehicle

    set dest one-of destinations  ;; pick one random destination patch as the desintation 
    
    move-to min-one-of (patches with [pcolor = pink]) [distance myself] ;; move the turtles to the closest intersection patch, so they can determine where to turn
    ]
  
  if display_parameter = "Level" [
    ask gasstations_ex [
     set plabel gas-level  ;; create the extension patch for gas station, initialize its refuelling timer and setup label for the level of gas left in the station
     set plabel-color black
     set gas-time 1
   ]
  ]
  
  if display_parameter = "Price" [
   ask gasstations_ex [
     set plabel gas-price  ;; create the extension patch for gas station, initialize its refuelling timer and setup label for the current price at the station
     set plabel-color black
     set gas-time 1 ]
  ]
  
  setup-plot
  update-plot
end

to setup-plot
  set-current-plot "Average Gas Price"  ;; initialize the plot for average gas price
end

to update-plot
  set-current-plot-pen "Mean Price"  ;; update the plot with the current mean gas price 
  plot mean [gas-price] of gasstations_ex
end
  
to setup-globals
  set grid-x-b world-width / grid-size  ;; determine the size of the city blocks
  set grid-y-b world-width / grid-size
end

to setup-patches
  ask patches [  ;; change the city block's color into brown
  set pcolor brown]
  
  set roads patches with [
    ( floor (( pxcor + max-pxcor - floor (grid-x-b - 1)) mod grid-x-b) = 0 ) or  ;; setup the road patches between the city block
    (floor((pycor + max-pycor) mod grid-y-b) = 0 )
    ]
  ask roads [ set pcolor white ]
  
  ask roads [
    set count-intersections count neighbors4 with [pcolor = white]  ;; determine the locations of the intersections
    if count-intersections = 4 [
      set pcolor pink ]  ;; change their color to pink for easier identification
  ]    
end

to setup-homes
  ask patches [ 
    set count-neighbors count neighbors4 with [pcolor = white]  
    ]
  
  ask patches [
    if pcolor = brown and count-neighbors > 0 [  ;; if the patch is at the edges of the city block, 
      if random-float 10 > (10 - home-prob) [    ;; it will randomly turn into a home patch, based on the set probability
        set pcolor green ] 
      ] 
    ]
  
  set homes patches with [
    pcolor = green 
    ]
end

to setup-gasstation
  ask patches [ 
    set count-neighbors count neighbors4 with [pcolor = white] ]
  
  ask patches [
    if pcolor = brown and count-neighbors = 1 [  ;; if the patch is at the edges of the city block,
      if random-float 10 > (10 - gas-prob) [     ;; it will randomly turn into a gas station patch, based on the set probability
        set pcolor grey ] 
      ] 
    ]
  
  set gasstations patches with [
    pcolor = grey 
    ]
  
  ask patches [
    set count-gaspatch count neighbors4 with [pcolor = grey]  ;; turn the road patch right next to the gas station patch
    if pcolor = white and count-gaspatch = 1 [                ;; into an extension patch
      set pcolor red ] 
    set gasstations_ex patches with [ pcolor = red ] 
    ]
  
  ask gasstations_ex [
    set pcolor white ]
end

to setup-dest
  ask patches [
    set count-neighbors count neighbors4 with [pcolor = white]   ;; setup destination patches, with the same methodology
    set count-homes count neighbors with [pcolor = green ]       ;; as the home and gas station patches
    ]
  
  ask patches [
    if pcolor = brown and pcolor != grey and pcolor != green and count-neighbors > 0 and count-homes = 0 [
      if random-float 10 > (10 - dest-prob) [
        set pcolor orange ] 
      ] 
    ]
  
  set destinations patches with [
    pcolor = orange 
  ]
end

to go
  ask turtles [
    
    ifelse energy < 40 [
      ifelse gas-searching-mode = "Mode 1"  ;; if the vehicle has reached critical fuel level, it will 
      [ move-to-gas-mode1 ] [               ;; look for gas stations to fuel up, based on one of the selected
        if gas-searching-mode = "Mode 2"    ;; modes
        [ move-to-gas-mode2 ] ]
      ]
      [ 
      ifelse distance dest < 2    ;; determine whether the car has reached its destination     
        [ ifelse distance homexy < 2 and distance homexy = distance dest  ;; if the destination is a home patch,
          [ set dest one-of destinations ]                       ;; the vehicle will selected a new desintation.
          [ set dest homexy ]                                    ;; if not, it will return to its home patch
        ]  
        [ move ]  ;; the vehicle will move toward the gas station
      ]
  ] 
  
  ask gasstations_ex[  ;; setup interaction between the gas stations and the vehicle, if there is any
    check-gas-level
    update-label 
  ]
  
  update-plot
  tick
end     

to move
  
  if [pcolor] of patch-here = pink [  ;; if the vehicle  is at the intersection,
    face dest                         ;; face directly to the selected destination, 
    if heading <= 45 [                ;; and compare the direction to one of the four possible routes at the intersection.
      set heading 0 ]                 ;; Select the route that gives the least deviation to the direct heading.
    if heading > 315 [
      set heading 0]
    if heading > 45 and heading <= 135 [
      set heading 90 ]
    if heading > 135 and heading <= 225 [
      set heading 180 ]
    if heading > 225 and heading <= 315 [
      set heading 270 ]
  ]

  fd 1 
  
  set energy energy - 1   ;; consume one gas "point" for every step
end

to move-to-gas-mode1
  
  set fdest min-one-of gasstations_ex [distance myself]  ;; select the closest gas station
  if [pcolor] of patch-here = pink [   ;; travel toward the selected gas station using the same logic as above
    face fdest   
    if heading <= 45 [
      set heading 0 ]
    if heading > 315 [
      set heading 0]
    if heading > 45 and heading <= 135 [
      set heading 90 ]
    if heading > 135 and heading <= 225 [
      set heading 180 ]
    if heading > 225 and heading <= 315 [
      set heading 270 ]
  ]
          
  fd 1
  
  set energy energy - 1
  
  if distance fdest < 1                           ;; if the vehicle is right on top of the gas station,
    [ set gas-level (gas-level - (300 - energy))  ;; it will proceed to fuel up by resetting its gas level to full tank, 
      set energy 300                              ;; and deduct the corresponding amount from the gas station's reservior
    ]
end

to move-to-gas-mode2
  if energy < 40 [
    let cheapest-gas count gasstations_ex in-radius (range_of_gasstations_consider)  ;; counts the number of gas station with a radius
    ifelse cheapest-gas < 2
      [ set fdest min-one-of gasstations_ex [distance myself] ]  ;; if there is less than two gas stations within the radius, the vehicle will just travel to the closest one
      [ set fdest min-one-of (gasstations_ex in-radius range_of_gasstations_consider) [gas-price] ]  ;; if there are multiple gas stations around, it will selected the cheapest station within the radius
  ]

  if [pcolor] of patch-here = pink [
    face fdest 
    if heading <= 45 [
      set heading 0 ]
    if heading > 315 [
      set heading 0]
    if heading > 45 and heading <= 135 [
      set heading 90 ]
    if heading > 135 and heading <= 225 [
      set heading 180 ]
    if heading > 225 and heading <= 315 [
      set heading 270 ]
  ]
  
  fd 1
  
  set energy energy - 1
  
  if distance fdest < 1
    [ set gas-level (gas-level - (300 - energy)) 
      set energy 300
    ]
end
        
to check-gas-level
  set gas-time gas-time + 1  ;; count the time between replenishment of the gas stations
  
  if gas-time > 500 and gas-level > 1500 [  ;; if the gas station have not had any visitor for the last 500 ticks, and the gas level is above 1500
    set gas-price ( mean [gas-price] of gasstations_ex - (random-float 0.5) )  ;; it will reset its price to slightly below the average gas price
    set gas-time 0                                                             ;; and reset the counter
  ]
  
  if gas-level < 300 [   ;; if the gas station is selling out most of its fuel,
    set gas-level 2000   ;; its fuel tank is replenished 
    set gas-price ( gas-price - (gas-time - 400) / 1000 )  ;; a new price is determined by how fast did it take to sell out most of reserve
    set gas-time 0]  ;; reset time counter
  
  if random_fluctuation? [  ;; introduce a switch for random fluctuations
    let prob_change random 10
    if prob_change < 2 [  
      set gas-price ( gas-price - random-float 0.5 + random-float 0.5 ) ]
  ]  
end         
        
to update-label  ;; update the label showing the current amount of gas in the reservior or the current price of the station
  ask gasstations_ex [ 
    if display_parameter = "Level" [
      set plabel gas-level ]
    if display_parameter = "Price" [
      set plabel gas-price ]
  ]
end
@#$#@#$#@
GRAPHICS-WINDOW
315
10
835
551
25
25
10.0
1
10
1
1
1
0
1
1
1
-25
25
-25
25
0
0
1
ticks

BUTTON
7
10
70
43
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

SLIDER
8
170
306
203
grid-size
grid-size
10
15
10
1
1
NIL
HORIZONTAL

SLIDER
8
91
306
124
home-prob
home-prob
0.5
4
2
.5
1
NIL
HORIZONTAL

SLIDER
8
130
306
163
gas-prob
gas-prob
0.1
3
0.7
0.2
1
NIL
HORIZONTAL

BUTTON
79
10
142
43
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL

CHOOSER
8
249
146
294
gas-searching-mode
gas-searching-mode
"Mode 1" "Mode 2"
1

SLIDER
8
52
306
85
dest-prob
dest-prob
1
10
4
1
1
NIL
HORIZONTAL

PLOT
8
337
306
550
Average Gas Price
Time
Gas Price
0.0
10.0
0.0
5.0
true
false
PENS
"Mean Price" 1.0 0 -2674135 true

SLIDER
8
210
306
243
range_of_gasstations_consider
range_of_gasstations_consider
5
20
10
1
1
NIL
HORIZONTAL

SWITCH
8
299
306
332
random_fluctuation?
random_fluctuation?
1
1
-1000

CHOOSER
153
249
306
294
display_parameter
display_parameter
"Price" "Level"
1

@#$#@#$#@
WHAT IS IT?
-----------
  This is a basic model of interaction between personal vehicles and gas stations, and the effects of such interaction on the gas prices of the gas stations.  The model takes into account of the layout of the world, which includes realistic features such as roads and city blocks, and attempts to simulate the various factors that affects the gas price.

HOW IT WORKS
------------
  The world is setup to simulate a typical grid design for urban design, with roads running between individual city blocks.  Three categories of patches are spawned within the city blocks, which are homes, gas stations, and destinations.  The density of all three patches

  Each home patch sprouts one turtle, which is then assigned as vehicle.  The location of home patch is stored within each of the corresponding vehicle.  Once the _Go_ button is pressed by the users, the vehicle would randomly determine a destination from the all of the destination patches within the world.  It would then proceed to travel to the destination by following the roads.  Upon reaching its destination, the vehicle would first return to its home patch and then proceed to travel to another destination.
  
  Every step made by the vehicle consumes some gas from the initial amount of gas given to each vehicle, which is randomly determined.   If the gas level of the vehicle falls below a certain critical value, the vehicle would try to find a gas station to fill through one of the two modes of searching. 

  In the first mode, the vehicle would simply find the closest gas station relative to its current location.  As for the second mode, the vehicle first determines how many gas stations are with a certain radius of its current location.  If there is only one gas station within the radius, the vehicle would then travel to this station and proceed to fill up its gas tank.  If there are multiple stations, the vehicle would then travel to the one with the cheapest gas price.

  For the gas station, its gas price is depended on the time it takes for the station to sell most of its gas.  If the gas is sold out in a really short time, the gas price would rise.  However, if there isn_t any visit to the gas station for a certain amount of time, the price of the gas at the station would be reset to a value slightly lower than the mean price of all stations.  A switch function is also added to the program.  It allows the gas price to randomly fluctuate.

HOW TO USE IT
-------------
1. Set RANDOM_FLUCTUATION? Switch to ON to allow the price to varies randomly by a small amount, or to OFF to only allow the price to change by the behavior of the vehicles
2. Adjust the variable parameters (see below), or use the default settings.
3. Press the SETUP button.
4. Press the GO button to begin the simulation
5. Look at the monitor to see movement of the vehicles and changes to the gas station_s reservoirs
6. Look at the Average Gas Price plot to see the change in the mean price of gas 

Adjustable Slider:
_DEST-PROB_ _ control the density of destination patches within the world
_HOME-PROB_ _ control the density of home patches within the world
_GAS-PROB_ _ control the density of gas station patches within the world
_RANGE_OF_GASSTATIONS_CONSIDER_ _ control the radius for the vehicle to compare the availability and price of the gas. 

Chooser Box:
_GAS-SEARCHING-MODE_ _ Determine which mode that the vehicles use to search for a gas station. See "HOW IT WORKS" section for detailed explanation of the two modes.

Switch:
_RANDOM_FLUCTUATION?_ _ Determine whether or not introduction a random fluctuation in the gas price of each station

THINGS TO NOTICE
----------------
- Switch the chooser, "DISPLAY_PARAMETER", between displaying the gasoline price and reservior level.  Notice any correlation between the gas price and the drainage speed.
- Turn the random fluctuation function on.  Notice any changes to the behavior of the vehicles.   

THINGS TO TRY
-------------
- Try changing to "GAS-SEARCHING-MODE", see what effects it has on the movement of the vehciles
- Try lowering the number of gas stations within the world.  Does that affect the rate of change for the gasoline price?  How about having more or less home patches and vehicles in the world?
- Some stations provides higher quality fuel, which attracts more vehicles, but the gasoline price is slightly higher.

EXTENDING THE MODEL
-------------------
- In the real world, residential regions or commerical districts tend form close to each other.  Add a switchable function that allow the home patches and/or destination patches to be located close to each other.
- The driving behavior of the vehicles are heavily influenced by the traffic condition of the roads that are being used.  Improve the current "MOVE" command, so that the speed of the vehicles are affected by how many other vehicles are using the same stretch of road.  Features like traffic lights can also be added to the existing intersection.
- Currently all roads are treated the same by the vehicles.  Such is not true in the real world, as vehicles are more likely to take highway or expressway to increase the speed of travel and fuel efficiency.  Introduce such feature to the Netlogo model.

RELATED MODELS
--------------
"Traffic Grid" model within the Social Science section of the Netlogo build-in library is related to this model, as the same city layour was used.

CREDITS AND REFERENCES
----------------------
The codes for the setup of the city blocks was constructed with the ones from the "Traffic Grid" model.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
0
Rectangle -7500403 true true 151 225 180 285
Rectangle -7500403 true true 47 225 75 285
Rectangle -7500403 true true 15 75 210 225
Circle -7500403 true true 135 75 150
Circle -16777216 true false 165 76 116

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 4.1.3
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="Dest_1510" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="3000"/>
    <metric>mean [gas-price] of gasstations_ex</metric>
    <enumeratedValueSet variable="random_fluctuation?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="grid-size">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="home-prob">
      <value value="1.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dest-prob">
      <value value="1"/>
      <value value="5"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="gas-prob">
      <value value="0.61"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="gas-searching-mode">
      <value value="&quot;Mode 2&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="range_of_gasstations_consider">
      <value value="10"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="HOME" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="3000"/>
    <metric>mean [gas-price] of gasstations_ex</metric>
    <enumeratedValueSet variable="random_fluctuation?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="grid-size">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="home-prob">
      <value value="0.5"/>
      <value value="1.5"/>
      <value value="2.5"/>
      <value value="3.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dest-prob">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="gas-prob">
      <value value="0.61"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="gas-searching-mode">
      <value value="&quot;Mode 2&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="range_of_gasstations_consider">
      <value value="10"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="GAS" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="3000"/>
    <metric>mean [gas-price] of gasstations_ex</metric>
    <enumeratedValueSet variable="gas-searching-mode">
      <value value="&quot;Mode 2&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="random_fluctuation?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="range_of_gasstations_consider">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dest-prob">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="home-prob">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="grid-size">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="gas-prob">
      <value value="0.1"/>
      <value value="1"/>
      <value value="2"/>
      <value value="3"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="RANGE" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="3000"/>
    <metric>mean [gas-price] of gasstations_ex</metric>
    <enumeratedValueSet variable="gas-searching-mode">
      <value value="&quot;Mode 2&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="random_fluctuation?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="range_of_gasstations_consider">
      <value value="5"/>
      <value value="10"/>
      <value value="15"/>
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dest-prob">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="home-prob">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="grid-size">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="gas-prob">
      <value value="1.1"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="MODE" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="3000"/>
    <metric>mean [gas-price] of gasstations_ex</metric>
    <enumeratedValueSet variable="gas-searching-mode">
      <value value="&quot;Mode 2&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="random_fluctuation?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="range_of_gasstations_consider">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dest-prob">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="home-prob">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="grid-size">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="gas-prob">
      <value value="0.7"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="SEARCH" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="3000"/>
    <metric>mean [gas-price] of gasstations_ex</metric>
    <enumeratedValueSet variable="gas-searching-mode">
      <value value="&quot;Mode 1&quot;"/>
      <value value="&quot;Mode 2&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="random_fluctuation?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="range_of_gasstations_consider">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dest-prob">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="home-prob">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="grid-size">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="gas-prob">
      <value value="0.7"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 1.0 0.0
0.0 1 1.0 0.0
0.2 0 1.0 0.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
