;extensions [sound]

globals [
  screen-rf ; 2 values: lab and blue
  time ; button time
  clicked? ; used for photos so that photo is taken (or erased) on mouse-up, not mouse-down
  photo-list ; list of elements in photos, by photo (as sublist)
  photo-width ; constant of separation between photos at bottom of View
  photo-height ; vertical dimension of photo frame
  photographer ; turtle taking photo when user clicks on it
  speed-of-light ; in patches per tick
  proper-spacing ; between machs
  max-photos ; where screen is full
  reference_frame ; screen RF (string): "lab", "blue", "red"
  meteor? ; string: "no meteor" or "speed 1" "speed 2" etc up to "speed 6"
]

breed [machs mach] ; these are the space turtles
machs-own [
  t-init ; initial clock setting: depends on reference frame
  t-curr ; current clock setting at any time during the run
  x-init ; initial position on screen: depends on r.f.
  screen-velocity ; speed at which it moves across the screen with each press of the > or < btn
  screen-dt ; rate at which turtle's clock changes with each press of the > or < btn
  id ; numerical identifier of turtle (= local x-coordinate)
  my-clock ; visible digital clock of local time (= local t-coordinate): a turtle
  my-badge ; a turtle displaying id
  photo? ; boolean; if turtle is in a picture, doesn't move or change time
]

breed [clocks clock] ; is the clock label for turtle's local time
breed [badges badge] ; id for the turtle's number
breed [meteors meteor] ; an extra for finding the speed of light
meteors-own [x-init screen-velocity photo?]

to startup
  setup
end

to setup
  ca
  set-default-shape machs "turtle 2"
  set-default-shape clocks "dot"
  set-default-shape badges "dot"
  set-default-shape meteors "meteor"
  ask patches [set pcolor 101]
  setup-vars
  create-trains
  endow-trains ; "endow" places variables into the turtles; change with different ref frame
  create-meteor
  endow-meteor ; as with trains
  place-on-view
end

to reset-trains [erase-meteor-photos?]; used with <reset> btn; for change of r.f. or meteor, without erasing pictures
  set time 0
  endow-trains
  endow-meteor
  place-on-view
  ; any photos with meteors need to be removed if meteor? has been reset,
  ; because they won't be compatible with new photos.
  if erase-meteor-photos? [
    if any? (turtle-set photo-list) with [breed = meteors] [
      ;create a new list of photos to be discarded:
      let discards []
      foreach photo-list [
        if any? (turtle-set ?) with [breed = meteors] [ set discards lput ? discards ]
      ]
      ; now remove them from the original photo list:
      foreach discards [ 
        set photo-list remove ? photo-list 
        ask (turtle-set ?) [die]
      ]
      update-photo-gallery
    ]
  ]
end

to create-trains
  create-one-train 90 2 red 10
  create-one-train 270 -2 blue 10
end

to create-one-train [ the-heading the-y the-color num-turtles ]
  let idx 0
  repeat num-turtles [
    create-machs 1 [
      set ycor the-y
      set color the-color
      set heading the-heading
      set size 3.5
      set id idx
      create-labels
      set idx idx + 1
      set photo? false
    ]
  ]
end

to endow-trains ; relativity calculations worked out by hand and fed in via this proc.
  if reference_frame = "lab" [
    endow-one-train red -6 2 1 1
    endow-one-train blue 6 2 -1 1
  ]
  if reference_frame = "blue" [
    endow-one-train red -4.15692193816531 3.2 (proper-spacing / 5) 0.6
    endow-one-train blue proper-spacing 0 0 1
  ]
  if reference_frame = "red" [
    endow-one-train red -1 * proper-spacing 0 0 1
    endow-one-train blue 4.15692193816531 3.2 -1 * (proper-spacing / 5) 0.6
  ]
end

to endow-one-train [the-color spacing dtime svel tvel]
  ask machs with [color = the-color] [
    set screen-velocity svel
    set screen-dt tvel
    set t-init id * dtime
    set x-init id * spacing
  ]
end

to setup-vars
  set screen-rf "desktop"
  set time 0
  set clicked? false
  set photo-list []
  set photo-width 8
  set photo-height 13.5
  set max-photos 7
  set speed-of-light 2
  set proper-spacing 6.92820323027551
  set reference_frame "lab"
  set meteor? "no meteor"
end

to create-meteor
  create-meteors 1 [
    set color yellow + 3
    set size 2
    set ycor 0
    set heading 0
    set photo? false
  ]
end

to endow-meteor ; relativistic values worked out by hand.
  ; note that for speeds 5 and 6 only the lab frame is allowed because they result
  ; in complex/contradictory values in other reference frames -- 
  ; faster than light travel impossible
  ask meteors [
    ifelse meteor? = "no meteor" [ht][st]
    if reference_frame = "lab" [
      set x-init -30
      if meteor? = "speed 1" [ set screen-velocity 1 ]
      if meteor? = "speed 2" [ set screen-velocity 1.5 ]
      if meteor? = "speed 3" [ set screen-velocity 1.66666 ]
      if meteor? = "speed 4" [ set screen-velocity 2 ]
      if meteor? = "speed 5" [ set screen-velocity 4 ]
      if meteor? = "speed 6" [ set screen-velocity 7 ]
    ]
    if reference_frame = "blue" [
      if meteor? = "speed 1" [ 
        set screen-velocity 1.3856406460551018 
        set x-init -20.8
      ]
      if meteor? = "speed 2" [
        set screen-velocity 1.57272727272727
        set x-init -18.8727272727273
      ]
      if meteor? = "speed 3" [
        set screen-velocity 1.62941176470588
        set x-init -18.3058823529411
      ]
      if meteor? = "speed 4" [
        set screen-velocity 1.73333333333333
        set x-init -17.3666666666666
      ]
      ifelse meteor? = "speed 5" [
        ht
        set screen-velocity 0
        set x-init -20
      ] [st]
    ]
    if reference_frame = "red" [
      set x-init 0
      if meteor? = "speed 1" [
        set screen-velocity 0
        set x-init -34.6
      ]
      if meteor? = "speed 2" [
        set screen-velocity 0.69
        set x-init -41.5
      ]
      if meteor? = "speed 3" [
        set screen-velocity 0.98571428571429
        set x-init -44.457142857143
      ]
      if meteor? = "speed 4" [
        set screen-velocity 1.73205080756888
        set x-init -51.9205080756888
      ]
    ]
  ]
end

to create-labels ; mach
  ; creates both the clock and the badge for a mach turtle, and ties them for movement
  ; places clocks above upper turtles and below lower ones to leave the meteor path clear
  ; in between them
  let tbadge nobody
  hatch-badges 1 [
    set tbadge self
    set size 0.1
    set ycor ycor - 0.2
    set xcor xcor + x-offset-of-breed
    set label [id] of myself
  ]
  set my-badge tbadge
  create-link-with my-badge [
    hide-link
    tie
  ]
  let tclock nobody
  hatch-clocks 1 [
    set tclock self
    ifelse heading = 90 [set ycor ycor + 1.5] [set ycor ycor - 2]
    set xcor xcor + x-offset-of-breed
    set size 0.1
  ]
  set my-clock tclock
  create-link-with my-clock [
    hide-link
    tie
  ]
end

;=========================== PHOTOS ============================

to-report xcor-of-photo [the-photo] ;the-photo is photo in photo-list
  report min-pxcor + 6 + photo-width * position the-photo photo-list
end

to-report x-offset-of-breed ; turtle; reports placement relative to mach owner
  let tout 0
  if breed = badges [ifelse heading = 270 [set tout 0.3][set tout 0.2]]
  if breed = clocks [ifelse heading = 270 [set tout 1] [set tout 0.5]]
  report tout
end

to update-photo-list ; places them in appropriate spacing along bottom of screen
  foreach photo-list [
    let px xcor-of-photo ?
    foreach ? [
      ask ? [set xcor px + x-offset-of-breed]
    ]
  ]
end

to draw-photo-frame [photo-x] ; uses drawing surface to put a rectangle around photo
  crt 1 [
    setxy (photo-x + 1 - (photo-width / 2)) (min-pycor +  photo-height)
    set heading 90
    set color white
    set pen-size 2
    pd
    fd photo-width - 2
    rt 90
    fd photo-height
    rt 90
    fd photo-width - 2
    rt 90
    fd photo-height
    die
  ]
end

to update-photo-gallery ; labels photos as "Photo 1" "Photo 2" etc. These are patch labels.
  ask patches [set plabel ""]
  cd
  let i 1
  foreach photo-list [
    draw-photo-frame xcor-of-photo ?
    let the-y min-pycor + photo-height - 2
    let the-x 1 + xcor-of-photo ?
    ask patch the-x the-y [set plabel word "Photo " i]
    set i i + 1
  ]
end

to-report make-image [new-x new-y]; mach; reports a list of self & dependents
  ; to be used in creating photos.
  ; must create new turtles rather than stamp the existing ones because labels don't stamp
  let tout []
  hatch 1 [
    set photo? true
    setxy new-x new-y
    ifelse breed = machs [
      create-labels
      set tout (list self my-badge my-clock)
    ][
      set tout (list self)
    ]
  ]
  report tout
end

to take-snapshot ; mach, when user clicks on a mach
  ; are any other machs or meteors lined up (vertically) with the clicked turtle (called "photographer")?
  ; tcond is the condition that will be used to avoid empty photos.
  let tcond any? other machs with [not hidden? and ycor > -10 and round xcor = [round xcor] of myself]
  set tcond tcond or any? meteors with [not hidden? and ycor > -10 and round xcor = [round xcor] of myself]
  if tcond [
    ; clicking sound if files present
;    carefully [sound:play-sound "CAMERA.WAV"][sound:play-drum "Claves" 127]
    ; find a location for the photo
    let px min-pxcor + 6 + length photo-list * photo-width
    ; get a turtleset of everybody -- mach, clock, label, meteor -- who should be present in photo:
    ; ycor > -10 is specified so we don't inadvertently include a photo in another photo, if it happens to line up vertically.
    let photo-turtles (turtle-set machs meteors) with 
      [not hidden? and ycor > -10 and round xcor = [round xcor] of myself]
    ; photo will be a list of all the turtle copies; then the photo-list is a list of these lists
    let this-photo []
    ask photo-turtles [
      ; make-image actually creates the copy of the calling turtle
      set this-photo sentence this-photo make-image px (min-pycor + 5 + ycor )
    ]
    ; places the time-labels on all the turtles, including photos
    update-time-labels
    ; finally, add photo to global list:
    set photo-list lput this-photo photo-list
  ]
end

to camera ; called when user clicks mouse button over a mach
  if length photo-list = max-photos [
    user-message "No more room in photo album."
    stop
  ]
  ; photographer will be the calling turtle: the one user clicks on
  set photographer one-of machs with [not hidden? and not photo? and (distancexy-nowrap mouse-xcor mouse-ycor) < 2]
  ; did user click close enough to a mach?
  if is-turtle? photographer [
    ; action only on mouse-up:
    ifelse mouse-down? [
      set clicked? true
    ][
      if clicked? [
        ask photographer [take-snapshot]
        ; add the label to the photo & place on screen
        update-photo-gallery
        set clicked? false
        stop
      ]
    ]
  ]
end

to remove-snapshot ; mach; done by clicking on a turtle in a photo. called by erase proc, below
  ; tphoto will be the photo clicked on by the mouse
  let tphoto []
  ; i will be index of tphoto in photo0list
  let i 0
  while [i < length photo-list and tphoto = []] [
    let tlist item i photo-list
    ; self is the calling turtle; i.e., has been clicked on by user
    if member? self tlist [set tphoto tlist]
    set i i + 1
  ]
  ; if photo is found, delete it from list and erase contents
  if tphoto != [] [
    set photo-list remove tphoto photo-list
    ask turtle-set tphoto [die]
  ]
end

to erase ; mouse action button (forever)
  let tmach one-of machs with [photo? and distancexy-nowrap mouse-xcor mouse-ycor < 2]
  if is-turtle? tmach [
    ifelse mouse-down? [
      ; action only on mouse-up
      set clicked? true
    ][
      if clicked? [
        ask tmach [remove-snapshot]
        update-photo-list
        ; redo all photo labels and outlines
        ask patches [set plabel ""]
        update-photo-gallery
        set clicked? false
        stop
      ]
    ]
  ]
end

to erase-all
  if length photo-list = 0 [stop]
  if user-yes-or-no? "Are you sure you want to erase all photos?" [
    foreach photo-list [
      ask turtle-set ? [die]
    ]
    set photo-list []
    update-photo-gallery
  ]
end

;============================= MOTION =========================

to update-time-labels ;observer
  ; at every step forward or backward in time, label of each mach's local time
  ; has to be updated, and turned into a string of form :001 (e.g.)
  ask machs [
    ask my-clock [ set label time-string [t-curr] of myself]
  ]
end

to-report time-string [the-time] ; clock
  ; creates a string out of a number; e.g. 1 => :001, and 10 => :010
  let ttime floor the-time
  let tstr (word ttime)
  let l length tstr
  while [l < 3] [
    set tstr word "0" tstr
    set l length tstr
  ]
  report word ":" tstr
end

to-report t-of-time ; mach
  ; relativistic local time, depending on initial time, which depended on location
  report precision (t-init + screen-dt * time) 0
end

to-report x-of-time ; mach, meteor
  ; position on screen of moving bodies
  report precision (x-init + screen-velocity * time) 2
end

to place-on-view ; observer
  ; places all turtles in their appropriate locations as time passes via user clicks on > and < btns
  ; only non-photo objects move with time
  ask meteors with [not photo?] [
    ; use a temp variable to check for leaving screen area:
    let tx x-of-time
    ifelse abs tx < max-pxcor [
      ; inside screen area? yes:
      set xcor tx
      ; if meteor has a faster-than-light speed, it can only be seen in the lab r.f.
      set hidden? (meteor? = "no meteor") or ((reference_frame != "lab") and (member? meteor? ["speed 5" "speed 6"]))
   ] [
      ; inside screen area? no:
      ht
    ]
    
  ]
  ask machs with [not photo?] [
    ; temp variable as with meteor above:
    let tx x-of-time
    set t-curr t-of-time
    ifelse abs tx < max-pxcor [
      set xcor tx
      ; each mach is a complex of three turtles: the mach itself, its clock, and its badge:
      ask (turtle-set self my-clock my-badge) [st]
    ] [
      ask (turtle-set self my-clock my-badge) [ht]
    ]
  ]
  ; be sure to change the time label on my-clock
  update-time-labels
end

to go-forward ; observer
  ; time is a global time variable that is pinned to one of the reference frames.
  ; in lab frame, it is the same for both blue and red; in others, it corresponds to the clock rate of the 
  ; stationary machs
  ; move forward in time:
  set time time + 1
  ; get everybody in the right place:
  place-on-view 
end

to go-backward
  if time = 0 [stop]
  ; move backward in time:
  set time time - 1
  ; get everybody in the right place:
  place-on-view
end

;===================== USER SETTINGS ========================

to set-reference_frame ; observer
  ; give user list of all r.f.'s, with current value noted:
  let tref user-one-of (word "Choose reference frame (currently " reference_frame ")")
    ["lab" "blue" "red"]
  if tref != false [set reference_frame tref]
end

to set-meteor ; observer
  ; warn user that any meteor photos will be erased:
  let can-proceed true
  if any? (turtle-set photo-list) with [breed = meteors] [
    set can-proceed user-yes-or-no? "Changing meteor settings will erase existing meteor photos. Proceed?"
  ]
  ; give user list of all meteor settings, with current value noted:
  if can-proceed [
    let tmet user-one-of (word "Choose meteor setting (currently " meteor? ")")
      ["no meteor" "speed 1" "speed 2" "speed 3" "speed 4" "speed 5" "speed 6"]
    if tmet != false [set meteor? tmet]
  ]
end

to btn-settings ; observer
  ; ask which setting user wants to change:
  let settings-list (list (word "Reference frame (currently " reference_frame ")...")
    (word "Meteor (currently " meteor? ")..."))
  let setting-choice user-one-of "Please select one: " settings-list
  let choice-idx position setting-choice settings-list
  if choice-idx = false [ stop ]
  ; change the setting as appropriate
  if choice-idx = 0 [ set-reference_frame ]
  if choice-idx = 1 [ set-meteor ]
  ; change screen to reflect new setting. This will preserve all photos,
  ;   ... unless meteor settings have changed, and some photos contain meteors.
  ;   ... In that case, meteor photos will be removed, because they are not compatible
  ;   ... with any new meteor photos.
  reset-trains (choice-idx = 1)
end
@#$#@#$#@
GRAPHICS-WINDOW
3
10
663
351
32
-1
10.0
1
10
1
1
1
0
1
0
1
-32
32
-22
8
0
0
1
ticks
30.0

BUTTON
7
358
62
391
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
236
359
299
392
>
go-forward
NIL
1
T
OBSERVER
NIL
>
NIL
NIL
1

BUTTON
169
359
232
392
<
go-backward
NIL
1
T
OBSERVER
NIL
<
NIL
NIL
1

BUTTON
403
356
480
389
NIL
camera
T
1
T
OBSERVER
NIL
C
NIL
NIL
1

BUTTON
484
356
576
389
erase one
erase
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
580
356
662
389
erase all
erase-all
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
66
359
165
392
reset
reset-trains false
NIL
1
T
OBSERVER
NIL
R
NIL
NIL
1

BUTTON
7
408
99
441
settings...
btn-settings
NIL
1
T
OBSERVER
NIL
S
NIL
NIL
1

MONITOR
103
409
217
454
NIL
reference_frame
17
1
11

MONITOR
221
410
303
455
meteor
meteor?
17
1
11

@#$#@#$#@
## WHAT IS IT?

This is a simulator of Special Relativity theory, allowing users to experiment with local times in different reference frames. For full documentation, see the author's web page: http://www.studiodosrios.com, click the "workshops" link, and then click "more..." under "Einstein for Everybody." (full address: http://www.studiodosrios.com/Site/Einstein.html)

This program is best used in conjunction with a course on special relativity, rather than as a replacement for it.

## OUR STORY

In intergalactic space, two trains of Mach Turtles, the most highly evolved creatures ever, fly past one another. They are carrying cargoes of Yttrium, Tortugium, and Praseodymium to and from distant planets.

Although they are moving at high speeds, the distances are vast so travel time is tedious. They entertain themselves by taking photographs of passing objects (such as the train moving in the opposite direction), and then comparing photos when they land.

Each turtle in each train has a number indicating its place in line, and a clock. As far as the turtles know, all the clocks within their own train are synchronized, even though we observers see them as de-synchronized. (We'll get to that point later.)

When they assemble on the remote planet Znrx, the turtles in the blue train look at their photos. Then, as a game, they try to answer the questions in the question section below.

## HOW TO USE IT

Use the forward and backward buttons (">" and "<") on the interface tab to run the model forwards and backwards in time.

To return the turtles to the initial situation, use the `reset` button. **NOTE:** If you use the `setup` button, the entire system is reset, and you will lose any settings you have made, and any pictures you have taken.

To take a picture, turn on the `camera` button, and click on one of the mach turtles. A photo taken by that turtle at the moment you mouse-clicked it, will appear in the photo gallery below. (The gallery will hold up to 7 photos.) You can erase a single picture by clicking the `erase one` button, then clicking on a turtle in the photo you want to erase. `erase all` does just what you expect. 

## QUESTIONS

The questions are deceptive. Think very carefully about how you answer them. (Answers can be obtained on the webpage noted above.)

Answers may require times and distance. Let the time unit be the "tick", which is what a turtle's clock shows. For example, think of the clock reading ":003" as showing "three ticks". Let the distance between from one turtle to the next in the same train be the unit of distance: measured from nose to nose. This distance is called "pace" (short for "carapace"). A speed, for example, might be represented as "two paces per tick."

All answers require photos: do not trust your eyes. Photos are the only evidence, and each question can be answered with two photos (though the same two photos will not answer every question). Be sure you can explain how the photo supports your answer, and remember that the turtles in one train believe that their own clocks are synchronized.

  (Q-1) How fast to the blue turtles think the red turtles are moving?

  (Q-2) Do the blue turtles think the red turtles' clocks are synchronized? 

  (Q-3) How fast do the blue turtles think the red turtles' clocks are ticking? 

  (Q-4) What do the blue turtles conclude about the spacing of the turtles in the red train?  (All turtles realize that the turtles in both trains are numbered sequentially.)

  (Q-5) Looking over the above four questions, how would the red turtles answer them about the blue train?

## INTERPRETATIONS AND FACTS

Earlier it was suggested that the turtles in one train believe that their clocks are synchronized, when it is obvious to the user that they are not. Also you should have found that the turtles in one train do not believe the at the clocks of the other train are synchronized. Since this is the world of relativity, you might therefore conclude that the turtles in one train would not think clocks in _any_ other train are synchronized. If you and your friends were standing side by side watching the screen, these turtles might not even think _your_ clocks are synchronized!

Let's look at the world from the blue turtles' point of view. Click the `settings` button and choose `reference frame.` Select `blue.` Now see the world as the blue turtles see it, with the blue clocks indeed synchronized, and the blue turtles standing still. Note that you can take exactly the same photos as you did in the earlier setting, the "lab" reference frame. (You can do the same for the red reference frame.)

This brings us to a very important principle. The photographs are the facts. (They are known as _events_ in physics.) The pictures you saw on the screen, of the turtles moving this way and that, are _interpretations._ It seems natural to suppose that your interpretation (e.g. the lab viewpoint at the beginning of the run) is the _reality_ of the situation (namely, that the two lines of turtles are both moving, and that their clocks are de-synchroized, that they are such-and-such a distance apart, etc.), but the other interpretations are equally real, because they give rise to the same facts.

The reality is simply the sum total of facts. Notice that all turtles (and you and your friends too) can all agree on those.

## THE SPECIAL SPEED

Discard your photos and return to the `lab` reference frame. (You can do this in a single stroke with the `setup` button.) Now, using the `settings` button, select `meteor`, and notice the different speeds available to you (1 through 6). Here is the big meteor question (to be answered, as always, with photographs):

  (Q-6) Is there a meteor speed at which both trains of turtles will observe as being the same? (Speeds are measured, remember, in paces per tick.) (The turtles -- and you -- will of course agree that the meteor is moving in a single _direction,_ from left to right, which is from higher-numbered to lower-numbered turtles in red frame, and from lower-numbered to higher-numbered in the blue frame.)

The answer to this is yes (your job is to find it and prove it.) We will now call the special speed that you found the _speed of light._

  (Q-7) How fast is the speed of light in the blue reference frame? In the red frame?
  (Q-8) How fast do the trains see each other moving as a proportion of the speed of light?
  (Q-9) If a meteor moves faster than the speed of light, what goes wrong? (lab frame only)

## RELATED MODELS

Further models in the relativity curriculum are under development. These will show galilean relativity, the problem with electromagnetic waves, Einstein's light clock, a simulation of the Michelson-Morley experiment, and, with enough time and encouragement, a general relativity toy.

## CREDITS AND REFERENCES

This model is based on an article by the physicist N. David Mermin, "The Amazing Many-Colored Relativity Engine," in _American Journal of Physics,_ 56, 600 (1988), and reprinted in his book _Boojums All the Way Through_ (Cambridge University Press, 1990). The ability to choose reference frames is not in Mermin's original conception, but the rest is, as are most of the questions.

For further models and curricular ideas, please refer to the author's website: http://www.studiodosrios.com.

You are welcome to use the "contact us" link on that website to send comments, ideas, suggestions, etc.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

meteor
true
0
Circle -7500403 true true 48 73 94
Circle -7500403 true true 36 112 108
Circle -7500403 true true 86 57 127
Circle -7500403 true true 90 105 120
Circle -7500403 true true 119 44 152
Circle -16777216 true false 70 122 44
Circle -7500403 true true 76 124 42
Circle -16777216 true false 128 81 54
Circle -7500403 true true 131 86 54

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
0
Rectangle -7500403 true true 151 225 180 285
Rectangle -7500403 true true 47 225 75 285
Rectangle -7500403 true true 15 75 210 225
Circle -7500403 true true 135 75 150
Circle -16777216 true false 165 76 116

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

turtle 2
true
0
Polygon -10899396 true false 132 70 134 49 130 42 132 17 151 7 171 18 173 42 169 50 172 72
Polygon -10899396 true false 165 195 195 195 240 225 240 240 210 270 195 240
Polygon -10899396 true false 90 195 60 225 60 240 90 270 105 240 120 195
Polygon -7500403 true true 119 60 179 60 209 86 224 120 209 210 180 240 120 240 89 210 74 120 88 84
Polygon -10899396 true false 195 75 225 60 255 60 270 75 285 105 285 135 240 90 225 90 210 90
Polygon -10899396 true false 105 75 75 60 45 60 30 75 15 105 15 135 60 90 75 90 90 90

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 1.0 0.0
0.0 1 1.0 0.0
0.2 0 1.0 0.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
