

;;; DEFINING VARIABLES ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


globals [
  width         ;; width of the pitch
  height        ;; height of the pitch
  red-goal      ;; patches that belongs to the red goal
  blue-goal     ;; patches that belong to the blue goal
  the-ball      ;; the ball
  score-red     ;; amount of goals that the red team has scored since the setup
  score-blue    ;; amount of goals that the blue team has scored since the setup
  red-team      ;; players of the red team
  blue-team     ;; players of the blue team
  seed          ;; current seed number
  previous-seed ;; previous seed number
  
  pass-distance              ;; the maximum distance a player can pass
  move-ball-distance         ;; the distance the ball is moved forward when the player is moving
  trick-ball-distance        ;; the distance the ball is moved forward when the player is doing a trick
  shoot-distance-selfish     ;; the maximum distance a selfish player will shoot
  shoot-distance-teamplayer  ;; the maximum distance a teamplayer will shoot
  speed                      ;; the speed of a player without the ball
  speed-receiver             ;; the speed of the player receiving a pass
  speed-with-ball            ;; the speed of the player with the ball
  speed-ball-pass            ;; the speed of the ball when it is passed
  speed-ball-shot            ;; the speed of the ball when it is shot
  speed-keeper               ;; the movement speed of the keeper in vertical direction
  distance-ballpossession    ;; the distance of the player to the ball for a player to recognize he is in ball possession
  locate-ball-distance       ;; the distance from which a player locates the ball and recognize he is close to the ball
  locate-player-distance     ;; the distance from which a player locates another player and recognize he is close to that player
  defender-distance          ;; the maximum distance the defender wants to stand from the goal
  keeper-reach               ;; the reach of the keeper
] 


breed [players player]   ;; soccer players
breed [keepers keeper]   ;; keeper
breed [balls ball]       ;; the ball
breed [referees referee] ;; referee


patches-own[
 available? 
]


players-own[
  ball?          ;; does the player have the ball?
  teamball?      ;; does the team of the player have the ball?
  nearbygoal?    ;; is the goal nearby?
  oppinfront?    ;; is there an opponent in front?
  oppnearby?     ;; is there an opponent nearby?
  nearbyteam?    ;; is there a free teammate nearby?
  team           ;; the team the player belongs to
  nearbyball?    ;; is the ball nearby?
  inbetween?     ;; is there an opponent between the player and the ball? (is the player standing free?)
  target         ;; the target of the player      
  mygoal         ;; the goal where the player has to score
  teamplayer?    ;; is the player a teamplayer?
  defensive?     ;; is the player a defensive player?
  defending?     ;; is the player currently in a defensive position?  
]


referees-own[
  team    ;; used when neither of the teams have the ball
]


keepers-own[
  team
]


balls-own[                
  owner              ;; the owner of the ball
  target             ;; the target where the ball can be played to (either a player or a patch of the goal)
  closest-blue       ;; the closest player to the ball of the blue team
  closest-red        ;; the closest player to the ball of the red team
  closestfront-blue  ;; the closest player of the blue team in defending position (between the ball and the goal of the defenders)
  closestfront-red   ;; the closest player of the blue team in defending position (between the ball and the goal of the defenders)
]


;;; SETUP THE MODEL ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


to setup ;; setup the field en set values of the variables
  clear-all
  set width 25
  set height 18
  set score-red 0
  set score-blue 0
  
  set pass-distance 10             ;; default 10
  set shoot-distance-selfish 13    ;; default 10
  set shoot-distance-teamplayer 7.5  ;; default 5
  set speed 0.5                    ;; default 0.5
  set speed-with-ball 0.45         ;; default 0.45
  set move-ball-distance 0.75      ;; default 0.75
  set trick-ball-distance 1.2        ;; default 1
  set speed-ball-pass 1            ;; default 1
  set speed-ball-shot 2            ;; default 2
  set distance-ballpossession 0.8  ;; default 0.8
  set locate-ball-distance 10      ;; default 10
  set locate-player-distance 1.5   ;; default 1.5
  set defender-distance 15         ;; default 15
  set speed-keeper 0.1             ;; default 0.1
  set keeper-reach 1.5             ;; default 1.5
  set speed-receiver 0.1           ;; default 0.1
  
  set-default-shape keepers "keeper"
  set-default-shape players "player"
  set-default-shape balls "football"
  set-default-shape referees "keeper"
  prepare-patch
  prepare-teams
  prepare-referee
  prepare-ball
end 


to prepare-patch ;; set the colors of the field
  ask patches [
    set pcolor green
    set available? 0
  ]
  let land define-lines
  ask land [
    set pcolor white
    set available? 2
  ]
  set blue-goal create-blue-goal
  ask blue-goal [
    set pcolor blue - 2
    set available? 1
  ]
  set red-goal create-red-goal
  ask red-goal [
    set pcolor red - 2
    set available? 1
  ]
end


to-report define-lines
  report patches with [(pxcor = width OR pxcor = -1 * width OR pycor = height OR pycor = -1 * height) AND pxcor <= width AND pxcor >= -1 * width AND pycor <= height AND pycor >= -1 * height]
end


to-report create-red-goal
  report patches with [((pxcor <= -1 * width AND pxcor >= -1 * width - 2)) and (pycor >= -3 and pycor <= 3)]
end


to-report create-blue-goal
  report patches with [((pxcor >= width AND pxcor <= width + 2)) and (pycor >= -3 and pycor <= 3)]
end


to prepare-teams
  set previous-seed seed
  if not fixed-seed? [
    set seed-number new-seed
  ]
  random-seed seed-number
  prepare-team1
  prepare-team2
  set seed seed-number
end

to prepare-team1 ;; prepare the first (blue) team
  create-players defense-team-blue [
    set color blue
    set teamplayer? true
    set defensive? true
  ]
  create-players defense-selfish-blue [
    set color blue
    set teamplayer? false
    set defensive? true
  ]
  create-players attack-team-blue [
    set color blue
    set teamplayer? true
    set defensive? false
  ]
  create-players attack-selfish-blue [
    set color blue
    set teamplayer? false
    set defensive? false
  ]  
  ask players with [color = blue] [
    set team "blue"
    set mygoal blue-goal
    set inbetween? true
    set size 2
    setxy (-1 * random width) (random (2 * height - 1) - height + 1)
    set defending? true
  ]
  
  create-keepers 1 [
    setxy (1 - width) 0
    set size 2
    set heading 90
    set color blue
    set team "blue"
  ]
  set blue-team players with [color = blue]
end

to prepare-team2 ;; prepare the second (red) team
  create-players defense-team-red [
    set color red
    set teamplayer? true
    set defensive? true
  ]
  create-players defense-selfish-red [
    set color red
    set teamplayer? false
    set defensive? true
  ]
  create-players attack-team-red [
    set color red
    set teamplayer? true
    set defensive? false
  ]  
  create-players attack-selfish-red [
    set color red
    set teamplayer? false
    set defensive? false
  ] 
  ask players with [color = red] [
    set team "red"
    set mygoal red-goal
    set inbetween? true
    set size 2
    setxy ( random width) (random (2 * height - 1) - height + 1)
    set defending? true
  ]
  
  create-keepers 1 [
    setxy (width - 1) 0
    set size 2
    set heading 270
    set color red
    set team "red"
  ]
  set red-team players with [color = red]
end


to prepare-referee
  create-referees 1 [
    setxy 0 (height + 1)
    set color yellow
    set heading 180
    set size 2
  ]
    create-referees 1 [
    setxy 0 (0 - height - 1)
    set color yellow
    set heading 0
    set size 2
  ]
end


to prepare-ball
  create-balls 1 [
    setxy 0 0
    set owner one-of referees
    set the-ball self
    set closest-red (min-one-of (players with [color = red]) [distance myself] )
    set closest-blue (min-one-of (players with [color = blue]) [distance myself] )   
  ]
end


;;; GO PROCEDURE ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


to go ;; observer procedure
  information
  decision
  move-ball
  set-motion
  check-goal
  check-out
end


;;; GATHERING INFORMATION ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


to information ;; gather information for all the players  
  ask players[
    ifelse (distance the-ball) < distance-ballpossession [ ;; do i have the ball?
      set ball? true ;; i have the ball
      ask one-of balls [set owner myself]
      set nearbyteam?(( other players in-radius pass-distance) with [team = [team] of myself] ) ;; how many team members are nearby? and free?   
      set oppnearby? count ((players in-radius locate-player-distance) with [team != [team] of myself]) ;; how many opponents are nearby?      
      ifelse any? patches in-radius shoot-distance-selfish with [pcolor = [color] of myself - 2] and not teamplayer? [ ;; is the goal nearby?
        set nearbygoal? true
        ]
      [ ifelse any? patches in-radius shoot-distance-teamplayer with [pcolor = [color] of myself - 2] and teamplayer? [ ;;
          set nearbygoal? true ]
        [ set nearbygoal? false ]
      ]
    ]
    [ set ball? false ;; i dont have the ball
      ifelse ([team] of ([owner] of the-ball) = [team] of self) and (self != ([target] of the-ball)) [ ;; does my team have the ball?
        set teamball? true
        ifelse distance (the-ball) < locate-ball-distance [ ;;am i close to the ball?
          set nearbyball? true
          set nearbyteam? count (( other players in-radius 2) with [team = [team] of myself]) ;; am i to close to someone else of my team?
          face the-ball
          ifelse (any? other players with [ball? = false] in-cone (distance the-ball) 20) or (any? other players with [ball? = false] in-radius locate-player-distance)  ;; are there opponents in between me and the ball?
          [ set inbetween? true ]
          [ set inbetween? false ]
          ]
        [ set nearbyball? false ]
      ]
      [ set teamball? false ;; my team does not have the ball
        ifelse team = "blue" 
        [ ifelse xcor < [xcor] of the-ball
          [ set defending? true ]
          [ set defending? false ]
        ]        
        [ ifelse xcor > [xcor] of the-ball
          [ set defending? true ]
          [ set defending? false ]
        ]  
        ifelse [closest-blue] of the-ball = self or [closest-red] of the-ball = self or [closestfront-blue] of the-ball = self or [closestfront-red] of the-ball = self ;; am i the closest to the ball?
        [ set nearbyball? true ]
        [ set nearbyball? false ]
        
      ] 
    ]           
  ]      
end


;;; DECISION TIME ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


to decision
  ask players [
    ifelse ball? [ ;; do you have the ball?
      ifelse teamplayer? [  ;; are you a teamplayer?
        ihavetheball-teamplayer]
      [
        ihavetheball-selfish ]
    ]
    [ ifelse teamball? [   ;; does your team have the ball?
        ifelse nearbyball? [nearby-teammate] ;; are you close by?
        [ ifelse defensive?  ;; are you defensive?
          [ far-teammate-defense]
          [ far-teammate-attack]
        ]
      ]
      [ ifelse nearbyball? [  ;; are you close by?
          nearby-opponent ]
        [ ifelse defensive?  ;; are you defensive?
          [ far-opponent-defense ]
          [ far-opponent-attack ]
        ]
      ]
    ]
  ] 
end


to ihavetheball-selfish ;; player procedure - when the player is selfish and has the ball
  let target-goal mygoal 
  face one-of mygoal
  ifelse nearbygoal? [ ;; is the goal nearby?
    print "shoot"
    shoot ]
  [
    ifelse oppinfront? > 0 or oppnearby? > 0 [ ;; is there an opponent in front?
      ifelse count nearbyteam? with [not inbetween?] > 2  [ ;; are there multiple teammates nearby and free?
        print "pass" 
        pass ]
      [ print "trick"
        trick ]
    ]
    [ print "move"
      i-move ]
  ]    
end


to ihavetheball-teamplayer ;; player procedure
  let target-goal mygoal 
  face one-of mygoal
  ifelse nearbygoal? [ ;; is the goal nearby?
    print "shoot"
    shoot ]
  [
    ifelse oppinfront? > 0 or oppnearby? > 0 [ ;; is there an opponent in front?
      ifelse count nearbyteam? with [not inbetween?] > 0  [ ;; are there multiple teammates nearby and free?
        print "pass" 
        pass ]
      [ print "trick"
        i-move ]
    ]
    [ print "move"
      i-move ]
  ]     
end


;;; ACTIONS ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  
to nearby-opponent ;; player procedure
   let my-team team
   set target one-of balls
   face target
end


to far-opponent-defense ;; player procedure
  ifelse team = "blue" [
    ifelse (distance one-of red-goal < defender-distance)
    [ rt random 360 ]
    [ face one-of red-goal
      set heading heading - 90 + random 180 ]
  ]
  [ ifelse (distance one-of blue-goal > defender-distance )
    [ face one-of blue-goal
      set heading heading - 90 + random 180 ]
    [ rt random 360 ]
  ] 
end


to far-opponent-attack ;; player procedure
  ifelse count players with [(color = [color] of myself) and (defending?)] > 2
  [ rt random 360]
  [ ifelse team = "blue" [
      ifelse (distance one-of red-goal < defender-distance)
      [ rt random 360]
      [ face one-of red-goal
        set heading heading - 90 + random 180 ]
    ]
    [ ifelse (distance one-of blue-goal > defender-distance )
      [ face one-of blue-goal
        set heading heading - 90 + random 180]
      [ rt random 360 ]
    ]
  ] 
end


to far-teammate-attack ;; player procedure
  ifelse team = "blue" [
    ifelse xcor < ([xcor] of the-ball)
    [ face one-of mygoal ]
    [ rt random 360 ]
  ]
  [ ifelse xcor > ([xcor] of the-ball)
    [ face one-of mygoal ]
    [ rt random 360 ]
  ]
end


to nearby-teammate ;; player procedure
  let my-team team
  ifelse nearbyteam? > 0
  [ face one-of ((other players in-radius 2.5) with [team = my-team])
    set heading heading + 180
  ]
  [ ifelse team = "blue" [
      face one-of blue-goal
      ifelse ([ycor] of the-ball) < ycor
      [ set heading heading - 30 ]
      [ set heading heading + 30 ]
    ]
    [ face one-of red-goal
      ifelse ([ycor] of the-ball) < ycor
      [ set heading heading + 30 ]
      [ set heading heading - 30 ]
    ] 
  ]
end


to far-teammate-defense
  rt random 360
end


to i-move ;; player procedure  
  ask the-ball [
    set heading ([heading] of owner)
    setxy ([xcor] of myself) ([ycor] of myself)
    fd move-ball-distance
    set target 0
  ]
end


to trick
  ask the-ball[
    set heading ([heading] of owner)
    setxy ([xcor] of myself) ([ycor] of myself)
    fd trick-ball-distance
    set target 0
    set owner one-of referees
  ]
end


to shoot ;; player procedure
  let target-goal one-of mygoal
  ask the-ball [
    set target target-goal
    face target
  ]
end


to pass ;; player procedure
  let my-team team
  let new-target one-of ((other players in-radius pass-distance) with [team = my-team and not inbetween?])
  if new-target != nobody [
    ask the-ball [
      set target new-target
      face target
      ask target [ set teamball? false ]
      print target
    ]
  ] 
end
  
  
;;; MOVE THE PLAYERS ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


to set-motion ;; observer procedure     let the players walk
  ask players [
    ifelse ball? [
      ifelse ([available?] of patch-ahead 2) < 1 [
        fd speed-with-ball ]
      [ set heading heading + 180
        fd speed-with-ball ]
    ]
    [ ifelse self != [target] of the-ball [
        ifelse ([available?] of patch-ahead 1) < 1 [
          fd speed ]
        [ set heading heading + 180
          fd speed ]
      ]
      [ fd speed-receiver ]
    ]   
  ]
  
  ask keepers[
    ifelse [ycor] of the-ball > ycor [
      if [pycor] of (one-of blue-goal)  > ycor[
        setxy xcor (ycor + speed-keeper) ]
    ]
    [ if [pycor] of (one-of blue-goal) < ycor[
        setxy xcor (ycor - speed-keeper) ]
    ]
  ]
end


;;; MOVE THE BALL ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


to move-ball ;; movement of the ball
  ask the-ball[
    set closestfront-red (min-one-of (players with [color = red and xcor > [xcor] of the-ball]) [distance myself] )
    set closestfront-blue (min-one-of (players with [color = blue and xcor < [xcor] of the-ball]) [distance myself] )
    set closest-red (min-one-of (players with [color = red]) [distance myself])
    set closest-blue (min-one-of (players with [color = blue]) [distance myself])
    if (([available?] of patch-ahead 1.2) < 2) and (([available?] of patch-ahead 0.6) < 2) [
      if target != 0 [
        if (distance target > shoot-distance-selfish + 1) and (distance target > shoot-distance-teamplayer + 1) and (distance target > pass-distance + 1) [  
          set target 0
          set owner one-of referees ]
        ifelse is-patch? target [
          if distance one-of keepers < keeper-reach [set heading heading + 90 + random 180] 
          fd speed-ball-shot ]
        [ fd speed-ball-pass ]
      ]
    ]
  ]
end


;;; CHECK GOAL OR OUT ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


to check-goal ;; check whether a goal is scored
   if [pcolor] of (patch ([xcor] of the-ball) ([ycor] of the-ball) ) = red - 2 [
     set score-red score-red + 1
     goal ]
   if [pcolor] of (patch ([xcor] of the-ball) ([ycor] of the-ball) ) = blue - 2 [
     set score-blue score-blue + 1
     goal ]   
end


to goal
  clear-turtles
  prepare-teams
  prepare-referee
  prepare-ball
end


to check-out ;; check whether the ball is outside the field
  if [pcolor] of (patch ([xcor] of the-ball) ([ycor] of the-ball) ) = green [
    if ([xcor] of the-ball > width + 1) or ([xcor] of the-ball < (-1 * (width + 1)))[
      out ]
  ]
  if [pcolor] of (patch ([xcor] of the-ball) ([ycor] of the-ball) ) = white [
    out ]
end


to out
  ask the-ball [
    set xcor 0
    set ycor 0
    set target 0
  ]
end
@#$#@#$#@
GRAPHICS-WINDOW
218
10
1355
798
30
20
18.48
1
10
1
1
1
0
0
0
1
-30
30
-20
20
0
0
1
ticks
30.0

BUTTON
65
325
128
358
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
17
370
80
403
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

MONITOR
14
412
80
457
NIL
score-red
17
1
11

MONITOR
112
412
183
457
NIL
score-blue
17
1
11

INPUTBOX
35
569
190
629
seed-number
-71588817
1
0
Number

SWITCH
64
527
184
560
fixed-seed?
fixed-seed?
1
1
-1000

MONITOR
39
638
174
683
previous seed number
previous-seed
17
1
11

SLIDER
23
80
195
113
attack-team-blue
attack-team-blue
0
10
4
1
1
NIL
HORIZONTAL

SLIDER
23
116
195
149
attack-selfish-blue
attack-selfish-blue
0
10
3
1
1
NIL
HORIZONTAL

SLIDER
23
10
195
43
defense-team-blue
defense-team-blue
0
10
0
1
1
NIL
HORIZONTAL

SLIDER
23
45
195
78
defense-selfish-blue
defense-selfish-blue
0
10
3
1
1
NIL
HORIZONTAL

SLIDER
23
237
195
270
attack-team-red
attack-team-red
0
10
2
1
1
NIL
HORIZONTAL

SLIDER
23
271
195
304
attack-selfish-red
attack-selfish-red
0
10
2
1
1
NIL
HORIZONTAL

SLIDER
23
167
195
200
defense-team-red
defense-team-red
0
10
6
1
1
NIL
HORIZONTAL

SLIDER
23
202
195
235
defense-selfish-red
defense-selfish-red
0
10
0
1
1
NIL
HORIZONTAL

BUTTON
134
374
197
407
NIL
go\n
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

INPUTBOX
28
684
183
744
sensitivity
20
1
0
Number

@#$#@#$#@
## WHAT IS IT?

This model is about a football game but can also be used for different teamsports with a ball and passing and shooting behaviour.

The model is trying to show how the composition of different types of soccer players influence the result of a match. 

## HOW IT WORKS

there are 4 types of players.

- selfish defender
- selfish attacker
- attacking teamplayer
- defensive teamplayer

they all behave in their own way. The teamplayers tend to pass faster, the selfish players are better with tricks and will shoot at greater distances. The attackers tend to move forward while their team has the ball and the defenders will move backwards when their team does not have the ball. 

At the setup and after each goal the players are randomly placed on their side of the field leading to a different behaviour for every model run.

## HOW TO USE IT

In this model you can change the composition of the team with sliders. You can create traditional teams with 10 players or even vary with the amount of players. there is a max of 10 players of the same type for each team.

With the fixed seed option turned on you can watch the same goal over and over again. You can rewatch the previous goal by entering the previous seed number in the current seed number and turn the fixed seed option on.

## THINGS TO NOTICE

Check the different behaviour of the team when a certain mix of players are in the field.

## THINGS TO TRY

Try different composition of teams to beat a team. 

## CREDITS AND REFERENCES

Designed by: Willemijn Bikker & Sander van Egmond
Written by: Sander van Egmond
For the course Agent Based Modelling at Wageningen University
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

football
true
0
Circle -1 true false 65 65 170
Line -16777216 false 60 150 90 150
Line -16777216 false 90 150 105 135
Line -16777216 false 105 135 105 120
Line -16777216 false 105 120 105 105
Line -16777216 false 105 105 90 90
Polygon -16777216 true false 225 150 210 150 195 165 195 195 210 210 225 195 240 150
Polygon -16777216 true false 60 150 90 150 105 135 105 105 90 90 75 105 60 135
Polygon -16777216 true false 135 150 150 165 165 165 180 150 180 135 165 120 150 120 135 135
Polygon -16777216 true false 105 210 90 225 135 240 150 240 150 210 135 195 120 195 105 210
Polygon -16777216 true false 165 60 195 75 210 90 210 105 195 105 180 105 165 90 165 75

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

keeper
true
0
Circle -6459832 true false 174 99 42
Circle -6459832 true false 84 99 42
Circle -7500403 true true 180 120 60
Rectangle -7500403 true true 90 120 210 180
Circle -7500403 true true 60 120 60
Circle -955883 true false 103 88 95
Polygon -7500403 true true 255 120 240 105 210 135 240 150 255 120
Polygon -7500403 true true 45 120 60 105 90 135 60 150 45 120
Circle -6459832 true false 240 90 30
Circle -6459832 true false 30 90 30

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

player
true
0
Circle -6459832 true false 174 99 42
Circle -6459832 true false 84 99 42
Circle -7500403 true true 180 120 60
Rectangle -7500403 true true 90 120 210 180
Circle -7500403 true true 60 120 60
Circle -955883 true false 103 88 95

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.0.3
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="blind (red) willems (blue) selfish shot sens" repetitions="1" runMetricsEveryStep="false">
    <setup>setup
set shoot-distance-selfish sensitivity</setup>
    <go>go

print sensitivity
print shoot-distance-selfish</go>
    <exitCondition>score-red = 10 or score-blue = 10</exitCondition>
    <metric>score-red</metric>
    <metric>score-blue</metric>
    <enumeratedValueSet variable="defense-team-red">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="defense-selfish-red">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-team-red">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-selfish-red">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="defense-team-blue">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="defense-selfish-blue">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-team-blue">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-selfish-blue">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sensitivity">
      <value value="5"/>
      <value value="7.5"/>
      <value value="10"/>
      <value value="12.5"/>
      <value value="15"/>
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="fixed-seed?">
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="blind (red) willems (blue) selfish shot sens" repetitions="50" runMetricsEveryStep="false">
    <setup>setup
set shoot-distance-selfish sensitivity</setup>
    <go>go

print sensitivity
print shoot-distance-selfish</go>
    <exitCondition>score-red = 10 or score-blue = 10</exitCondition>
    <metric>score-red</metric>
    <metric>score-blue</metric>
    <enumeratedValueSet variable="defense-team-red">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="defense-selfish-red">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-team-red">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-selfish-red">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="defense-team-blue">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="defense-selfish-blue">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-team-blue">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-selfish-blue">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sensitivity">
      <value value="5"/>
      <value value="7.5"/>
      <value value="10"/>
      <value value="12.5"/>
      <value value="15"/>
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="fixed-seed?">
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="blind (red) kuyt (blue) selfish shot sens" repetitions="50" runMetricsEveryStep="false">
    <setup>setup
set shoot-distance-selfish sensitivity</setup>
    <go>go

print sensitivity
print shoot-distance-selfish</go>
    <exitCondition>score-red = 10 or score-blue = 10</exitCondition>
    <metric>score-red</metric>
    <metric>score-blue</metric>
    <enumeratedValueSet variable="defense-team-red">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="defense-selfish-red">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-team-red">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-selfish-red">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="defense-team-blue">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="defense-selfish-blue">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-team-blue">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-selfish-blue">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sensitivity">
      <value value="5"/>
      <value value="7.5"/>
      <value value="10"/>
      <value value="12.5"/>
      <value value="15"/>
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="fixed-seed?">
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="blind (red) huntelaar (blue) selfish shot sens" repetitions="50" runMetricsEveryStep="false">
    <setup>setup
set shoot-distance-selfish sensitivity</setup>
    <go>go

print sensitivity
print shoot-distance-selfish</go>
    <exitCondition>score-red = 10 or score-blue = 10</exitCondition>
    <metric>score-red</metric>
    <metric>score-blue</metric>
    <enumeratedValueSet variable="defense-team-red">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="defense-selfish-red">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-team-red">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-selfish-red">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="defense-team-blue">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="defense-selfish-blue">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-team-blue">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-selfish-blue">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sensitivity">
      <value value="5"/>
      <value value="7.5"/>
      <value value="10"/>
      <value value="12.5"/>
      <value value="15"/>
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="fixed-seed?">
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="kuyt  (red) huntelaar (blue) selfish shot sens" repetitions="50" runMetricsEveryStep="false">
    <setup>setup
set shoot-distance-selfish sensitivity</setup>
    <go>go

print shoot-distance-selfish</go>
    <exitCondition>score-red = 10 or score-blue = 10</exitCondition>
    <metric>score-red</metric>
    <metric>score-blue</metric>
    <enumeratedValueSet variable="defense-team-red">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="defense-selfish-red">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-team-red">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-selfish-red">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="defense-team-blue">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="defense-selfish-blue">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-team-blue">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-selfish-blue">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sensitivity">
      <value value="5"/>
      <value value="7.5"/>
      <value value="10"/>
      <value value="12.5"/>
      <value value="15"/>
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="fixed-seed?">
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="willems  (red) huntelaar (blue) selfish shot sens" repetitions="50" runMetricsEveryStep="false">
    <setup>setup
set shoot-distance-selfish sensitivity</setup>
    <go>go

print shoot-distance-selfish</go>
    <exitCondition>score-red = 10 or score-blue = 10</exitCondition>
    <metric>score-red</metric>
    <metric>score-blue</metric>
    <enumeratedValueSet variable="defense-team-red">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="defense-selfish-red">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-team-red">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-selfish-red">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="defense-team-blue">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="defense-selfish-blue">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-team-blue">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-selfish-blue">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sensitivity">
      <value value="5"/>
      <value value="7.5"/>
      <value value="10"/>
      <value value="12.5"/>
      <value value="15"/>
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="fixed-seed?">
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="willems  (red) kuyt (blue) selfish shot sens" repetitions="50" runMetricsEveryStep="false">
    <setup>setup
set shoot-distance-selfish sensitivity</setup>
    <go>go

print shoot-distance-selfish</go>
    <exitCondition>score-red = 10 or score-blue = 10</exitCondition>
    <metric>score-red</metric>
    <metric>score-blue</metric>
    <enumeratedValueSet variable="defense-team-red">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="defense-selfish-red">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-team-red">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-selfish-red">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="defense-team-blue">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="defense-selfish-blue">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-team-blue">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-selfish-blue">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sensitivity">
      <value value="5"/>
      <value value="7.5"/>
      <value value="10"/>
      <value value="12.5"/>
      <value value="15"/>
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="fixed-seed?">
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="willems  (red) kuyt (blue) teamplayer shot sens" repetitions="50" runMetricsEveryStep="false">
    <setup>setup
set shoot-distance-teamplayer sensitivity</setup>
    <go>go

print shoot-distance-teamplayer</go>
    <exitCondition>score-red = 10 or score-blue = 10</exitCondition>
    <metric>score-red</metric>
    <metric>score-blue</metric>
    <enumeratedValueSet variable="defense-team-red">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="defense-selfish-red">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-team-red">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-selfish-red">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="defense-team-blue">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="defense-selfish-blue">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-team-blue">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-selfish-blue">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sensitivity">
      <value value="2.5"/>
      <value value="5"/>
      <value value="7.5"/>
      <value value="10"/>
      <value value="15"/>
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="fixed-seed?">
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="willems (red) blind (blue) teamplayer shot sens" repetitions="50" runMetricsEveryStep="false">
    <setup>setup
set shoot-distance-teamplayer sensitivity</setup>
    <go>go

print shoot-distance-teamplayer</go>
    <exitCondition>score-red = 10 or score-blue = 10</exitCondition>
    <metric>score-red</metric>
    <metric>score-blue</metric>
    <enumeratedValueSet variable="defense-team-red">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="defense-selfish-red">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-team-red">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-selfish-red">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="defense-team-blue">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="defense-selfish-blue">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-team-blue">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-selfish-blue">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sensitivity">
      <value value="2.5"/>
      <value value="5"/>
      <value value="7.5"/>
      <value value="10"/>
      <value value="15"/>
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="fixed-seed?">
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="willems (red) huntelaar (blue) teamplayer shot sens" repetitions="50" runMetricsEveryStep="false">
    <setup>setup
set shoot-distance-teamplayer sensitivity</setup>
    <go>go

print shoot-distance-teamplayer</go>
    <exitCondition>score-red = 10 or score-blue = 10</exitCondition>
    <metric>score-red</metric>
    <metric>score-blue</metric>
    <enumeratedValueSet variable="defense-team-red">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="defense-selfish-red">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-team-red">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-selfish-red">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="defense-team-blue">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="defense-selfish-blue">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-team-blue">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-selfish-blue">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sensitivity">
      <value value="2.5"/>
      <value value="5"/>
      <value value="7.5"/>
      <value value="10"/>
      <value value="15"/>
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="fixed-seed?">
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="kuyt (red) huntelaar (blue) teamplayer shot sens" repetitions="50" runMetricsEveryStep="false">
    <setup>setup
set shoot-distance-teamplayer sensitivity</setup>
    <go>go

print shoot-distance-teamplayer</go>
    <exitCondition>score-red = 10 or score-blue = 10</exitCondition>
    <metric>score-red</metric>
    <metric>score-blue</metric>
    <enumeratedValueSet variable="defense-team-red">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="defense-selfish-red">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-team-red">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-selfish-red">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="defense-team-blue">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="defense-selfish-blue">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-team-blue">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-selfish-blue">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sensitivity">
      <value value="2.5"/>
      <value value="5"/>
      <value value="7.5"/>
      <value value="10"/>
      <value value="15"/>
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="fixed-seed?">
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="blind (red) huntelaar (blue) teamplayer shot sens" repetitions="50" runMetricsEveryStep="false">
    <setup>setup
set shoot-distance-teamplayer sensitivity</setup>
    <go>go

print shoot-distance-teamplayer</go>
    <exitCondition>score-red = 10 or score-blue = 10</exitCondition>
    <metric>score-red</metric>
    <metric>score-blue</metric>
    <enumeratedValueSet variable="defense-team-red">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="defense-selfish-red">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-team-red">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-selfish-red">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="defense-team-blue">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="defense-selfish-blue">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-team-blue">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-selfish-blue">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sensitivity">
      <value value="2.5"/>
      <value value="5"/>
      <value value="7.5"/>
      <value value="10"/>
      <value value="15"/>
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="fixed-seed?">
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="blind (red) kuyt (blue) teamplayer shot sens" repetitions="50" runMetricsEveryStep="false">
    <setup>setup
set shoot-distance-teamplayer sensitivity</setup>
    <go>go

print shoot-distance-teamplayer</go>
    <exitCondition>score-red = 10 or score-blue = 10</exitCondition>
    <metric>score-red</metric>
    <metric>score-blue</metric>
    <enumeratedValueSet variable="defense-team-red">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="defense-selfish-red">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-team-red">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-selfish-red">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="defense-team-blue">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="defense-selfish-blue">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-team-blue">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attack-selfish-blue">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sensitivity">
      <value value="2.5"/>
      <value value="5"/>
      <value value="7.5"/>
      <value value="10"/>
      <value value="15"/>
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="fixed-seed?">
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 1.0 0.0
0.0 1 1.0 0.0
0.2 0 1.0 0.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
