;: All credit for base model created by Amy Steimke, Boise State University
;: Further edits by Colin McCarthy, Whitman College
;: Last edit: 4/17/17
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;; PARAMETERS ;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; currently: 1 tick = 1 hour,
; each patch = 1 km ( with the exception of dams )

globals [ velocity
   ] ; average velocity of movement (km/hr), calculated in set-velocity procedure


turtles-own [ start-movement              ; true/false, says if they can start migrating
              movement-speed              ; fish's velocity, remains constant thru each run
              prob-death                  ; probability of death on any tick while migrating
              at-dam?                     ; if turtle is within 1 tick of a dam, goes to true
              time-at-dam                 ; total time turtle spends at dams
              probability-of-passage      ; probability of passing through a dam
              matelife                    ; time alive once at end of the model
              partner                     ; numerical value representing partner agent
              probability-of-reproduction ; probability that reproduction is successful

]

patches-own [ dam1   ; x-coordinate of Ice Harbor Dam (first one)
              dam2   ; x-coordinate of Lower Monumental Dam (second dam)
              dam3   ; x-coordinate of Little Goose Dam (third dam)
              dam4   ; x-coordinate of Lower Granite Dam (last one)
              matezone   ; x-coordinate of mating zone
]


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;:::;;;;;;;;;;;;;;;;;::: SETUP ;:;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


to setup
  clear-all
  reset-ticks
  file-close
  set-velocity
  setup-patches
  setup-turtles
  destroy-dams
end


;;; creates river corridor and locations of dams ;;;
to setup-patches
  ask patches
  [ set pcolor 55 ; sets up river banks
    set dam1 1 ; these 4 lines are x-coordinates of the dams
    set dam2 52
    set dam3 97
    set dam4 157
    set matezone 167
    if (pycor < 10) and (pycor > -10) [ set pcolor 105 ] ; sets up river corridor
    if (pxcor = dam1) [ set pcolor 2 ] ; sets up first dam
    if (pxcor = dam2) [ set pcolor 2 ] ; sets up second dam
    if (pxcor = dam3) [ set pcolor 2 ] ; sets up third dam
    if (pxcor = dam4) [ set pcolor 2 ] ; sets up last dam
    if ((pxcor = matezone and pycor < -1) or (pxcor = matezone and pycor > 1)) [set pcolor 15]
   ]
end


;;; sets average velocity for fish ;;;
to set-velocity
  ; velocity equation from Salinger & Anderson, 2006
  let flow (FlowRate-m3/s / 1000) ; flowrate in 10^3 m3/s
  let rivertemp (( RiverTemp-farenheit - 32 ) / 1.8 ) ; change river temperature to Celsius
    ifelse rivertemp <= 16.3
     [ set velocity ((27.3 + ( 2 * rivertemp ) - (1.5 * flow)) / 24)] ; velocity in km/hr for temperatures below 16.3C
     [ set velocity ((100.7 - ( 2.5 * rivertemp) - (1.5 * flow)) / 24)] ; velocity in km/hr for temperatures above 16.3C
end


;;; sets up initial fish variables ;;;
to setup-turtles
  create-turtles initial-number-fish
  [
    choose-sex
    set shape "fish"
    set size 2
    setxy 2 8 - random(17)  ; places fish in starting corridor
    set heading 90
    set probability-of-passage probability-making-thru-dam
    set at-dam? false
    set time-at-dam 0
    set matelife abs(round(random-normal 24 24))
    set probability-of-reproduction probability-successfully-reproducing
    set partner 0
    set start-movement false
    set movement-speed random-normal velocity speed-stdev ; normally distributes speeds based off of calculated velocity and chosen stdev from slider on interface
    if movement-speed <= 0 [set movement-speed movement-speed * -1] ; makes sure movement-speed is not negative, otherwise error occurs while checking for dams and is unrealistic
 ]
end

to choose-sex  ;; turtle procedure for determining sex of agent
  set color one-of [magenta red];
end

;;; procedure for destroying dams in the world, switches are on the interface ;;;
to destroy-dams
    if destroy-dam-2?        ; procedure for destroying 2nd dam
      [ask patches
        [if pxcor = dam2
         [if (pycor < 10) and (pycor > -10) [ set pcolor 105 ]]
        ]
       ]
    if destroy-dam-3?        ; procedure for destroying 3rd dam
      [ask patches
        [if pxcor = dam3
         [if (pycor < 10) and (pycor > -10) [ set pcolor 105 ]]
        ]
       ]
    if destroy-dam-4?        ; procedure for destroying 4th dam
      [ask patches
        [if pxcor = dam4
         [if (pycor < 10) and (pycor > -10) [ set pcolor 105 ]]
        ]
       ]
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;; MOVEMENT ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


to go
  tick
  start-migration
  move-turtles
  fish-die
  mate
  if-else Opportunity-Mating?
  [if not any? turtles with [xcor < matezone + 1 and matelife > 0]
    [stop]       ; stops ticking once all fish are past last dam and have had the chance to mate
  ]
  [if not any? turtles with [xcor < dam4 ]
      [stop]     ; stops ticking once all fish are past last dam ignoring the mating submodel
  ]
  check-for-dams
  dam-learn
end


; procedure tells turtles when they can start moving, and when to stop movement after last dam
to start-migration
  ask turtles
  [
    if random-float 1 < probability-of-starting-migration  ; starts fish migrating
       [set start-movement true]
    if xcor > dam4 + movement-speed                        ; stops fish migrating
        [set start-movement false]
  ]
end


; procedure checks for dams ahead of turtle in the length of their movement-speed (only get stopped once by each dam)
to check-for-dams
  ask turtles
  [
    ifelse any? patches in-cone movement-speed 1 with [pcolor = 2] ; checks for dams ahead of them
    [set at-dam? true]
    [set at-dam? false]
   if at-dam? = true
   [set time-at-dam time-at-dam + 1]  ; calculates how many ticks they are stopped by dams
  ]
end


; procedure calls for fish who have successfully navigated previous dams in a shorter period of time to have a
; higher likelihood of navigating next dam they encounter more quickly
to dam-learn
  if dam-learning? ; switch is located on interface
  [ask turtles
    [ if at-dam? = true and xcor > dam2 and xcor < dam3 and time-at-dam <= 3      ; if second dam crossing took 3 or less ticks, they will get a higher probability for next dam
     [set probability-of-passage probability-of-passage * (random-float 1 + 1) ]  ; will multiply their previous probability anywhere from ~1.01 - 1.99

      if at-dam? = true and xcor > dam3 and xcor < dam4  and time-at-dam <= 5     ; if second & third dam crossings took 5 or less ticks, they will get a higher probability for next dam
     [set probability-of-passage probability-of-passage * (random-float 1 + 1) ]  ; will multiply their previous probability anywhere from ~1.01 - 1.99
    ]

  ]
end

to mate
  if Opportunity-Mating?
  [ask turtles
    [if (xcor >= dam4 and xcor < dam4 + 9)
      [setxy matezone 0]
    if xcor = matezone
      [set matelife matelife - 1
      if partner = 0
        [set partner one-of other turtles with [partner = 0 and color != [color] of myself and xcor = matezone]
          ifelse partner = nobody [set partner 0] [ask partner [ set partner myself ] ]
        ]
      if matelife != nobody
        [if matelife <= 0
          [die]
      ]
      if partner != nobody
      [if partner != 0
        [if color = magenta
          [setxy matezone + 5 8 - random(17)
            set color pink]
      ]
      ]
      ]
    ]

  ]
end


; procedure for movement; each tick turtles move their movement-speed (km/hr)
; if at-dam? is true for turtle, then turtle takes time trying to navigate/find fish ladder to cross dam
to move-turtles
  ask turtles
  [ if start-movement = true
    [ ifelse at-dam? = false
    [ set heading 90 forward movement-speed]
      [if random-float 1 < probability-of-passage ; set on interface tab, probability of fish finding the ladder
        [set heading 90 forward movement-speed]
      ]
    ]
  ]
end


; Procedure for fish to die on each tick with probability set on interface tab
; Mortality factors - fishing, disease, unsuccessful dam crossing
to fish-die
    ask turtles
    [
      ifelse at-dam? = false
              [if xcor <= dam4
                 [set prob-death initial-mortality-constant] ; chance of dying on any turn
                  if xcor > dam4 [set prob-death 0 ] ; no death after the cross 4th dam
              ]
              [ set prob-death initial-mortality-constant * dam-death-multiplier ] ; higher likelihood of death while they are trying to cross dam
    if random-float 1 < prob-death [die] ; death procedure
    ]
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;; END ;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
@#$#@#$#@
GRAPHICS-WINDOW
20
10
957
161
-1
-1
5.28
1
10
1
1
1
0
0
0
1
0
175
-13
13
1
1
1
ticks
60.0

BUTTON
15
210
96
249
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
113
211
193
250
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
17
323
254
356
Probability-of-starting-migration
Probability-of-starting-migration
0
.5
0.015
.005
1
NIL
HORIZONTAL

SLIDER
230
455
357
488
speed-stdev
speed-stdev
0
1
0.2
.01
1
NIL
HORIZONTAL

SLIDER
18
363
225
396
Initial-mortality-constant
Initial-mortality-constant
0
.001
4.0E-4
.00001
1
NIL
HORIZONTAL

MONITOR
543
214
653
259
Fish reaching end
count turtles with [xcor > (dam4)]
17
1
11

SLIDER
19
472
211
505
Probability-making-thru-dam
Probability-making-thru-dam
0
.5
0.05
.01
1
NIL
HORIZONTAL

INPUTBOX
17
260
213
320
Initial-number-fish
57672.0
1
0
Number

PLOT
990
35
1530
505
Number of fish crossing dams
Time
Number of Fish
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"Dam2" 1.0 0 -16777216 true "" "plot count turtles with [xcor >= dam2 AND xcor < dam2 + movement-speed ]"
"Dam3" 1.0 0 -7500403 true "" "plot count turtles with [xcor >= dam3 AND xcor < dam3 + movement-speed ]"
"Dam4" 1.0 0 -955883 true "" "plot (count turtles with [xcor >= (dam4 - movement-speed + 1) and xcor < dam4 - 1] ) "

INPUTBOX
228
374
354
434
RiverTemp-farenheit
55.79
1
0
Number

INPUTBOX
229
212
355
272
FlowRate-m3/s
1688.7
1
0
Number

PLOT
676
367
874
487
Speed distribution
NIL
NIL
0.0
3.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "histogram [movement-speed] of turtles"

SWITCH
373
212
519
245
Destroy-Dam-2?
Destroy-Dam-2?
0
1
-1000

SWITCH
374
247
520
280
Destroy-Dam-3?
Destroy-Dam-3?
0
1
-1000

SWITCH
375
283
521
316
Destroy-Dam-4?
Destroy-Dam-4?
1
1
-1000

MONITOR
545
265
653
310
# of fish alive
count turtles
17
1
11

MONITOR
544
315
655
360
Mean time at dam
mean [ time-at-dam ] of turtles
2
1
11

TEXTBOX
223
192
373
210
Factors affecting fish's speed
11
0.0
1

INPUTBOX
19
403
212
463
Dam-death-multiplier
3.0
1
0
Number

SWITCH
375
345
521
378
Dam-learning?
Dam-learning?
0
1
-1000

MONITOR
546
370
658
415
Mean passage prob
mean [ probability-of-passage ] of turtles
2
1
11

SWITCH
370
455
567
488
Opportunity-Mating?
Opportunity-Mating?
0
1
-1000

SLIDER
15
515
272
548
probability-successfully-reproducing
probability-successfully-reproducing
0
1
1.0
.01
1
NIL
HORIZONTAL

PLOT
995
530
1525
810
Opportunity Mating
Time
Number of fish
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"Offspring" 1.0 0 -16777216 true "" "plot count turtles with [xcor > matezone]"
"Mating Zone" 1.0 0 -7500403 true "" "plot count turtles - count turtles with [xcor > matezone]"

@#$#@#$#@
## WHAT IS IT?

This model simulates salmon migration up a large river body, specifically the Salmon River, but could be tweaked to represent the Columbia. Large dams impede migration on these rivers, so the fish must try to navigate fish ladders and avoid fallback after they get over the dam. Recently, some organizations have suggested removing dams in the Snake River, so this model can help to say to what extent that would help fish populations.

## HOW IT WORKS

Once fish start migrating, they follow a set of rules and probabilities. They are assigned a swimming speed and move upstream, experiencing a small chance of mortality on any turn. When they encounter a dam, they must spend time (and a greater exposure to mortality) as they try to cross it. If the fish learns how to cross ladders, then it is more likely to cross the next one more quickly. 

## HOW TO USE IT

Setup: Sets up the world and agents 

Go: Runs the program

Initial-number-fish: Input a whole number to create starting number of agents

Probability-of-starting-migration: This is the probability that a fish will start their migration on each tick at the first dam in the Snake River. This is to spread the fish out to get a more realistic spread of arrival times at the last dam.

Initial-mortality-constant: This is the probability that on any given tick a fish will die. Mortality factors here include: fishing, predators, disease, or being lost to a smaller tributary. 

Dam-death-multiplier: Additional mortality factors occur while fish are trying to cross a dam. This input multiplies the initial-mortality-constant for as long as the fish is trying to cross.

Probability-making-thru-dam: If fish is at a dam, this is the initial likelihood of crossing the dam on any given tick. More about this in Dam-learning?

Flow-rate: River discharge rate, in m3/s. Historical data can be found online. Typical values are ~3500-4500 for spring, and ~500-1200 for the fall. 

RiverTemp-Fahrenheit: River temperature in Fahrenheit. Typical values in the spring are ~45-50 and in the fall, ~65-70.

Speed-stdev: Changes how great of a spread of swimming velocities the fish have.

Destroy-dam-XX: These three switches allow you to 'destroy' dams 2, 3, and 4. This allows fish to pass freely through where the structures impeded them before.

Dam-learning?: This switch lets you decide if the fish 'learn' how to cross dams. That is, a fish that was more successful in a prior dam crossing is more likely to be more successful in upstream crossings.


## THINGS TO NOTICE

Notice how if probability-of-making-through dam is low that the fish get hung up at each dam crossing? Additionally, there is greater mortality there so more fish die. Also, you'll see a greater spread of fish arriving at the end if dam-learning? is turned on and the output 'Mean time at dam' will be lower. 

## THINGS TO TRY

Play with all of the sliders! Ones that are particularly interesting are: initial-mortality-constant, probability-of-making-through-dam, speed-stdev, destroying dams, and dam-learning?

## EXTENDING THE MODEL

Daily fish counts exist at these dams. So, you could force the model each 24 ticks with the daily counts from Ice Harbor Dam (1st one) and get rid of the probability-of-starting-migration slider. Then, you'd be able to see much more accurately and precisely how the fish arriving at the last dam (Lower Granite) match that dam's actual daily fish count.

Can you find where along the path the fish experience death the most?

You could extend the model to include full migration with the 4 dams downstream on the Columbia River. The tricky part would be estimating which proportion of fish choose to go up the Snake River instead of continuing up the Columbia.

## CREDITS AND REFERENCES

Thanks to Ben Pauli for teaching the class and learning us up real good in NetLogo. Also, thanks to Andrea Leonard, Reggie Walters, Shital Dhakal, and Hamid Dashti for providing input and feedback on model design.

Salinger, D. H., & Anderson, J. J. (2006). Effects of water temperature and flow on adult salmon migration swim speed and delay. Transactions of the American Fisheries Society, 135(1), 188-199.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="experiment" repetitions="2" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>count turtles with [xcor &gt; (dam4)]</metric>
    <enumeratedValueSet variable="initial-mortality-constant">
      <value value="7.5E-4"/>
      <value value="0.001"/>
      <value value="0.00125"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Probability-of-starting-migration">
      <value value="0.05"/>
      <value value="0.1"/>
      <value value="0.15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="probability-making-through-dam">
      <value value="0.075"/>
      <value value="0.15"/>
      <value value="0.225"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="speed-stdev">
      <value value="0.15"/>
      <value value="0.3"/>
      <value value="0.45"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="experiment" repetitions="1" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>count turtles</metric>
    <enumeratedValueSet variable="initial-number-fish">
      <value value="8540"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Destroy-Dam-3?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Destroy-Dam-2?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="RiverTemp-farenheit">
      <value value="47"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-mortality-constant">
      <value value="0.001"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Probability-of-starting-migration">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="FlowRate-m3/s">
      <value value="3960"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="probability-making-through-dam">
      <value value="0.15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="speed-stdev">
      <value value="0.3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Destroy-Dam-4?">
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="experiment" repetitions="1" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>count turtles</metric>
    <enumeratedValueSet variable="RiverTemp-farenheit">
      <value value="47"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="probability-making-through-dam">
      <value value="0.2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Destroy-Dam-4?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Destroy-Dam-3?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Destroy-Dam-2?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Probability-of-starting-migration">
      <value value="0.06"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="FlowRate-m3/s">
      <value value="3960"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="speed-stdev">
      <value value="0.31"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-mortality-constant">
      <value value="9.0E-4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-number-fish">
      <value value="1000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="experiment" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>count turtles</metric>
    <enumeratedValueSet variable="RiverTemp-farenheit">
      <value value="47"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="probability-making-through-dam">
      <value value="0.2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Destroy-Dam-4?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Destroy-Dam-3?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Destroy-Dam-2?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Probability-of-starting-migration">
      <value value="0.06"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="FlowRate-m3/s">
      <value value="3960"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="speed-stdev">
      <value value="0.31"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-mortality-constant">
      <value value="9.0E-4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-number-fish">
      <value value="1000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="experiment" repetitions="10" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>count turtles with [xcor &gt; (dam4)]</metric>
    <enumeratedValueSet variable="RiverTemp-farenheit">
      <value value="47"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="probability-making-through-dam">
      <value value="0.2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Destroy-Dam-4?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Destroy-Dam-3?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Destroy-Dam-2?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Probability-of-starting-migration">
      <value value="0.06"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="FlowRate-m3/s">
      <value value="3960"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="speed-stdev">
      <value value="0.31"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-mortality-constant">
      <value value="9.0E-4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-number-fish">
      <value value="1000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="experiment" repetitions="20" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>count turtles</metric>
    <enumeratedValueSet variable="RiverTemp-farenheit">
      <value value="47"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="probability-making-through-dam">
      <value value="0.2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Probability-of-starting-migration">
      <value value="0.06"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="speed-stdev">
      <value value="0.31"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dam-death-multiplier">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-number-fish">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Destroy-Dam-2?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dam-learning?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="FlowRate-m3/s">
      <value value="3960"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Destroy-Dam-4?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Destroy-Dam-3?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-mortality-constant">
      <value value="6.9E-4"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="TEST" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>count turtles</metric>
    <enumeratedValueSet variable="RiverTemp-farenheit">
      <value value="47"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="probability-making-through-dam">
      <value value="0.2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Probability-of-starting-migration">
      <value value="0.06"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="speed-stdev">
      <value value="0.31"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dam-death-multiplier">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-number-fish">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Destroy-Dam-2?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dam-learning?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="FlowRate-m3/s">
      <value value="3960"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Destroy-Dam-4?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Destroy-Dam-3?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-mortality-constant">
      <value value="6.9E-4"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="experiment" repetitions="100" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>count turtles</metric>
    <enumeratedValueSet variable="RiverTemp-farenheit">
      <value value="47"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="probability-making-through-dam">
      <value value="0.2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Probability-of-starting-migration">
      <value value="0.06"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="speed-stdev">
      <value value="0.31"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dam-death-multiplier">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-number-fish">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Destroy-Dam-2?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dam-learning?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="FlowRate-m3/s">
      <value value="3960"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Destroy-Dam-4?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Destroy-Dam-3?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-mortality-constant">
      <value value="6.9E-4"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="experiment" repetitions="20" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>mean [ time-at-dam ] of turtles</metric>
    <enumeratedValueSet variable="RiverTemp-farenheit">
      <value value="47"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="probability-making-through-dam">
      <value value="0.2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Probability-of-starting-migration">
      <value value="0.06"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="speed-stdev">
      <value value="0.3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dam-death-multiplier">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-number-fish">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Destroy-Dam-2?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dam-learning?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="FlowRate-m3/s">
      <value value="3960"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Destroy-Dam-4?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Destroy-Dam-3?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-mortality-constant">
      <value value="6.9E-4"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
1
@#$#@#$#@
