globals [
  tick-advance-amount               ; how much we advance the tick counter this time through
  max-tick-advance-amount           ; the largest tick-advance-amount is allowed to be
  init-avg-speed init-avg-energy    ; initial averages
  avg-speed avg-energy              ; current average
  avg-energy-green
  avg-energy-orange
  avg-energy-purple
  particle-size
  toggle-red-state
  toggle-green-state
  min-particle-energy
  max-particle-energy
  particles-to-add
  show-wall-hits?
  max-particles
  #particles
  #-green-particles
  #-purple-particles
  #-orange-particles
  villi-slider-moved?
  old-villi-slider-value
  bottom-wall-ycor
  tracker-init-particles
  permeability
  particle-to-watch
]


breed [ particles particle ]
breed [ flows flow ]
breed [ walls wall ]
breed [ edges edge ]



particles-own [
  speed mass energy          ; particles info
  last-collision
  color-type
  absorbed?
  swept?
  large?
]

patches-own [
  is-blood?
  blood-heading

]


walls-own [
  energy
  valve-1?
  valve-2?
  pressure?
  surface-energy



]

to setup
  clear-all
  reset-ticks
  ask patches [set pcolor white]
  set particle-size 2
  set max-tick-advance-amount 0.02
  set show-wall-hits? false
  set particles-to-add 2
  set old-villi-slider-value villi-height
  set villi-slider-moved? true
  set bottom-wall-ycor min-pycor + 3
  set-default-shape walls "cell"
  set max-particles 200
  set #-green-particles 0
  set #-purple-particles 0
  set #-orange-particles 0
  set tracker-init-particles initial-#-small-food-particles
  set permeability 50
  set particle-to-watch nobody

  set min-particle-energy 0
  set max-particle-energy 500  ;(.5 ) * ( max-dist-in-tick-advance-amount  / max-tick-advance-amount ) ^ 2




  redraw-villi?
  draw-blood-stream
  draw-edges
  make-small-particles
  make-large-particles

  set init-avg-speed avg-speed
  set init-avg-energy avg-energy

  update-variables
  do-plotting
end


to go

  if ticks < end-simulation-run-at [
  redraw-villi?

    ask particles with [not absorbed?] [ bounce ]
    ask flows [move-flows]
    ask particles with [not absorbed?] [ move ]
    ask particles with [absorbed?  and pycor != min-pycor] [seep-toward-blood]
    ask particles with [swept?] [sweep-with-blood]
  ;  ask particles with [not absorbed?] [ check-for-collision ]
    ask particles with [not absorbed? and any? walls-here ] [ rewind-to-bounce ]
    ask particles with [not absorbed? and any? walls-here ] [ remove-from-walls ]

    check-watched-particle

  tick-advance tick-advance-amount
  calculate-tick-advance-amount


  update-variables
  do-plotting
  display
  ]
end

to check-watched-particle
  if particle-to-watch != nobody [
  ask particle-to-watch [
      if pycor = max-pycor [rp]
    ]

    ]
end


to update-variables
  let green-particles particles with [ color-type = 55 ]
  let purple-particles  particles with [ color-type = 115 ]
  let orange-particles particles with [ color-type = 25 ]
  set #-green-particles count green-particles
  set #-orange-particles count orange-particles
  set #-purple-particles count purple-particles
  set #particles (#-green-particles + #-purple-particles + #-orange-particles)

  if any? particles [
    set avg-speed  mean [ speed ] of particles
    set avg-energy mean [ energy ] of particles
  ]

  ifelse #-green-particles > 0 [
    set avg-energy-green mean [ energy ] of green-particles
  ]
  [ set avg-energy-green "N/A"]

  ifelse  #-orange-particles > 0 [
    set avg-energy-orange mean [ energy ] of orange-particles
  ]
  [ set avg-energy-orange "N/A"]

  ifelse  #-purple-particles > 0 [
    set avg-energy-purple mean [ energy ] of purple-particles
  ]
  [ set avg-energy-purple "N/A"]

end

to watch-a-particle
  set particle-to-watch one-of particles with [pycor < (max-pycor - 2)]
  if count particles > 0 [
    watch one-of particles

  ]
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;BUILD VILLI   ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


to shove-particles
  ask particles-here with [pycor <= (bottom-wall-ycor + villi-height)] [set ycor (bottom-wall-ycor + villi-height)]
end

to add-wall
  shove-particles
          sprout 1 [
          set breed walls set color gray
          ;set is-blood? false
          initialize-this-wall
        ]
end

to add-hidden-wall
  shove-particles
          sprout 1 [
          set breed walls set color gray
          set is-blood? false
          set hidden? true
          initialize-this-wall
        ]
end


to initialize-blood
          set breed walls set color red
          set size 1.0
          set is-blood? true
          set heading 90
          set shape "square"
          set color [255 0 0 150]
end

to add-blood-up
  shove-particles
  set blood-heading 0
  set is-blood? true
  sprout 1 [initialize-blood ]
end

to add-blood-down
  shove-particles
  set blood-heading 180
  set is-blood? true
  sprout 1 [initialize-blood]
end

to add-blood-right
  shove-particles
  set blood-heading 90
  set is-blood? true
  sprout 1 [initialize-blood]
end


to redraw-villi?
  if old-villi-slider-value != villi-height [set villi-slider-moved? true set old-villi-slider-value villi-height]
  if villi-slider-moved? [
     redraw-villi
     set villi-slider-moved? false
  ]
end

to redraw-villi
  ask walls with [pycor >= (min-pycor + 3)] [die]
  ask patches with [pycor >= (min-pycor + 3)] [set is-blood? false]
   ;; draw horizontal line until reaching a point where a villi is... then call villi build...skip and continue

   let distance-between-villi 5

    ask patches with [pxcor >= min-pxcor and pxcor <= max-pxcor and pycor = max-pycor ]  [add-hidden-wall]


     let this-pxcor min-pxcor
     let this-pycor bottom-wall-ycor
     let villi-width 5
     let this-width-counter 0
     let this-lift false
   repeat (max-pxcor - min-pxcor + 1)  [

    ifelse ((this-pxcor mod 11 >= 0) and (this-pxcor mod 11 <= 4)) and (this-pxcor >= (min-pxcor + 3) and this-pxcor <= (max-pxcor - 5))
         [set this-pycor (bottom-wall-ycor + villi-height) set this-lift true]
         [set this-pycor bottom-wall-ycor]
    ask patches with [pxcor = this-pxcor and pycor = this-pycor] [add-wall]

    ;; draw-vertical wall
    if this-lift [
        ask patches with [pxcor = this-pxcor and pycor >= bottom-wall-ycor and pycor <= this-pycor] [
          if (this-pxcor mod 11 = 0 and pycor < (this-pycor)) [add-wall]
          if (this-pxcor mod 11 = 1 and pycor < (this-pycor - 1)) [add-blood-up]
          if (this-pxcor mod 11 = 2 and pycor < (this-pycor - 2))[add-wall]
          if ((this-pxcor mod 11 = 2 or this-pxcor mod 11 = 1) and (pycor = (this-pycor - 1) ))[add-blood-right]
         if ((this-pxcor mod 11 = 2 ) and ( pycor = (this-pycor - 2)))[add-blood-right]
          if (this-pxcor mod 11 = 3 and pycor <= (this-pycor - 1)) [add-blood-down]
          if (this-pxcor mod 11 = 4 and pycor < (this-pycor)) [add-wall]
        ]
      set this-lift false
      ]
    set this-pxcor this-pxcor + 1
   ]


end

to draw-blood-stream
  let blood-patches patches with [pxcor >= min-pxcor and pxcor <= max-pxcor and pycor < bottom-wall-ycor and pycor >= min-pycor ]
  ask blood-patches [add-blood-right]
  ask n-of 30 blood-patches [
    make-a-floater
  ]
end

to make-a-floater
  let this-color (5 + random-float 20)
  let this-list [0 0 0]
  set this-list lput  this-color this-list
  sprout 1 [
    set breed flows
  set color this-list
  set shape "square"
  ]
end

to draw-edges
  ask patches with [pycor >= (max-pycor) ][
    sprout 1 [
      set breed edges
      set shape "square 3"
      set size 1.05
      set color gray + 2
    ]
  ]
end



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;WALL INTERACTION;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;GAS MOLECULES MOVEMENT;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
to bounce  ; particles procedure
  ; get the coordinates of the patch we'll be on if we go forward 1
  let bounce-patch nobody
  let bounce-patches nobody
  let hit-angle 0
  let this-patch patch-here
  let new-px 0
  let new-py 0
  let visible-wall nobody

  set bounce-patch  min-one-of walls in-cone ((sqrt (2)) / 2) 180 with [ patch-here != this-patch ] [ distance myself ]

  if bounce-patch != nobody [
    set new-px [ pxcor ] of bounce-patch
    set new-py [ pycor ] of bounce-patch
    set visible-wall walls-on bounce-patch

    if any? visible-wall  [
      ifelse (random 100 > permeability or (pycor >= (max-pycor - 1) ) or large? ) [
      if bounce-patch != patch-here [ set hit-angle towards bounce-patch ] ;; new bounce patch code
      ifelse (hit-angle <= 135 and hit-angle >= 45) or (hit-angle <= 315 and hit-angle >= 225) [
        set heading (- heading)
      ][
        set heading (180 - heading)
      ]

      set absorbed? false]
      [set absorbed? true]
    ]


  ]
end


to rewind-to-bounce  ; particles procedure
  ; attempts to deal with particle penetration by rewinding the particle path back to a point
  ; where it is about to hit a wall
  ; the particle path is reversed 49% of the previous tick-advance-amount it made,
  ; then particle collision with the wall is detected again.
  ; and the particle bounces off the wall using the remaining 51% of the tick-advance-amount.
  ; this use of slightly more of the tick-advance-amount for forward motion off the wall, helps
  ; insure the particle doesn't get stuck inside the wall on the bounce.

  let bounce-patch nobody
  let bounce-patches nobody
  let hit-angle 0
  let this-patch nobody
  let new-px 0
  let new-py 0
  let visible-wall nobody

  bk (speed) * tick-advance-amount * .49
  set this-patch  patch-here

  set bounce-patch  min-one-of walls in-cone ((sqrt (2)) / 2) 180 with [ self != this-patch ] [ distance myself ]

  if bounce-patch != nobody [

    set new-px [pxcor] of bounce-patch
    set new-py [pycor] of bounce-patch
    set visible-wall walls-on bounce-patch

    if any? visible-wall with [not hidden?] [
      set hit-angle towards bounce-patch

      ifelse (hit-angle <= 135 and hit-angle >= 45) or (hit-angle <= 315 and hit-angle >= 225) [
        set heading (- heading)
      ][
        set heading (180 - heading)
      ]


    ]
  ]
  fd (speed) * tick-advance-amount * 0.75
end

to move  ; particles procedure
  if patch-ahead (speed * tick-advance-amount) != patch-here [ set last-collision nobody ]
  fd (speed * tick-advance-amount * 0.75)
  if ycor >  (bottom-wall-ycor + villi-height) [set xcor xcor - .05]
end


to move-flows
  set heading 90
  fd (7 * tick-advance-amount * 0.75)

end

to seep-toward-blood
  let flex-threshold .2 + random-float .05
  let all-blood patches with [is-blood?]
  let target-blood-patch min-one-of all-blood [distance myself  ]
 ; show distance target-blood
  let final-heading towards target-blood-patch
  let heading-difference ((final-heading - heading) / 20)
  set heading heading + heading-difference
  let blood-near-me all-blood with [distance myself < flex-threshold]
  ifelse not any? blood-near-me [fd (2 * tick-advance-amount) ][set swept? true]
end

to sweep-with-blood
  let old-heading heading
  let heading-difference ((old-heading - heading) / 20)
  if (is-blood? and swept?)  [set heading blood-heading fd (3 * tick-advance-amount) set heading heading + heading-difference]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;GAS MOLECULES COLLISIONS;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;from GasLab

to calculate-tick-advance-amount
  ; tick-advance-amount is calculated in such way that even the fastest
  ; particles will jump at most 1 patch delta in a ticks tick. As
  ; particles jump (speed * tick-advance-amount) at every ticks tick, making
  ; tick delta the inverse of the speed of the fastest particles
  ; (1/max speed) assures that. Having each particles advance at most
  ; one patch-delta is necessary for it not to "jump over" a wall
  ; or another particles.
  ifelse any? particles with [ speed > 0 ] [
    set tick-advance-amount min list (1 / (ceiling max [speed] of particles )) max-tick-advance-amount
  ][
    set tick-advance-amount max-tick-advance-amount
  ]
end

to check-for-collision  ; particles procedure
  ; Here we impose a rule that collisions only take place when there
  ; are exactly two particles per patch.  We do this because when the
  ; student introduces new particles from the side, we want them to
  ; form a uniform wavefront.
  ;
  ; Why do we want a uniform wavefront?  Because it is actually more
  ; realistic.  (And also because the curriculum uses the uniform
  ; wavefront to help teach the relationship between particles collisions,
  ; wall hits, and pressure.)
  ;
  ; Why is it realistic to assume a uniform wavefront?  Because in reality,
  ; whether a collision takes place would depend on the actual headings
  ; of the particles, not merely on their proximity.  Since the particles
  ; in the wavefront have identical speeds and near-identical headings,
  ; in reality they would not collide.  So even though the two-particles
  ; rule is not itself realistic, it produces a realistic result.  Also,
  ; unless the number of particles is extremely large, it is very rare
  ; for three or  particles to land on the same patch (for example,
  ; with 400 particles it happens less than 1% of the time).  So imposing
  ; this additional rule should have only a negligible effect on the
  ; aggregate behavior of the system.
  ;
  ; Why does this rule produce a uniform wavefront?  The particles all
  ; start out on the same patch, which means that without the only-two
  ; rule, they would all start colliding with each other immediately,
  ; resulting in much random variation of speeds and headings.  With
  ; the only-two rule, they are prevented from colliding with each other
  ; until they have spread out a lot.  (And in fact, if you observe
  ; the wavefront closely, you will see that it is not completely smooth,
  ; because  collisions eventually do start occurring when it thins out while fanning.)

  if count other particles-here  in-radius 1 = 1 [
    ; the following conditions are imposed on collision candidates:
    ;   1. they must have a lower who number than my own, because collision
    ;      code is asymmetrical: it must always happen from the point of view
    ;      of just one particles.
    ;   2. they must not be the same particles that we last collided with on
    ;      this patch, so that we have a chance to leave the patch after we've
    ;      collided with someone.
    let candidate one-of other particles-here with [ who < [ who ] of myself and myself != last-collision ]
    ;; we also only collide if one of us has non-zero speed. It's useless
    ;; (and incorrect, actually) for two particles with zero speed to collide.
    if (candidate != nobody) and (speed > 0 or [ speed ] of candidate > 0) [
      collide-with candidate
      set last-collision candidate
      ask candidate [ set last-collision myself ]
    ]
  ]
end

; implements a collision with another particles.
;
; THIS IS THE HEART OF THE particles SIMULATION, AND YOU ARE STRONGLY ADVISED
; NOT TO CHANGE IT UNLESS YOU REALLY UNDERSTAND WHAT YOU'RE DOING!
;
; The two particles colliding are self and other-particles, and while the
; collision is performed from the point of view of self, both particles are
; modified to reflect its effects. This is somewhat complicated, so I'll
; give a general outline here:
;   1. Do initial setup, and determine the heading between particles centers
;      (call it theta).
;   2. Convert the representation of the velocity of each particles from
;      speed/heading to a theta-based vector whose first component is the
;      particle's speed along theta, and whose second component is the speed
;      perpendicular to theta.
;   3. Modify the velocity vectors to reflect the effects of the collision.
;      This involves:
;        a. computing the velocity of the center of mass of the whole system
;           along direction theta
;        b. updating the along-theta components of the two velocity vectors.
;   4. Convert from the theta-based vector representation of velocity back to
;      the usual speed/heading representation for each particles.
;   5. Perform final cleanup and update derived quantities.
to collide-with [ other-particles ] ;; particles procedure
  ; PHASE 1: initial setup

  ; for convenience, grab  quantities from other-particles
  let mass2 [ mass ] of other-particles
  let speed2 [ speed ] of other-particles
  let heading2 [ heading ] of other-particles

  ; since particles are modeled as zero-size points, theta isn't meaningfully
  ; defined. we can assign it randomly without affecting the model's outcome.
  let theta (random-float 360)

  ; PHASE 2: convert velocities to theta-based vector representation

  ; now convert my velocity from speed/heading representation to components
  ; along theta and perpendicular to theta
  let v1t (speed * cos (theta - heading))
  let v1l (speed * sin (theta - heading))

  ;; do the same for other-particles
  let v2t (speed2 * cos (theta - heading2))
  let v2l (speed2 * sin (theta - heading2))

  ; PHASE 3: manipulate vectors to implement collision

  ; compute the velocity of the system's center of mass along theta
  let vcm (((mass * v1t) + (mass2 * v2t)) / (mass + mass2) )

  ; now compute the new velocity for each particles along direction theta.
  ; velocity perpendicular to theta is unaffected by a collision along theta,
  ; so the next two lines actually implement the collision itself, in the
  ; sense that the effects of the collision are exactly the following changes
  ; in particles velocity.
  set v1t (2 * vcm - v1t)
  set v2t (2 * vcm - v2t)

  ; PHASE 4: convert back to normal speed/heading

  ; now convert my velocity vector into my new speed and heading
  set speed sqrt ((v1t ^ 2) + (v1l ^ 2))
  set energy (0.5 * mass * speed ^ 2)
  ; if the magnitude of the velocity vector is 0, atan is undefined. but
  ; speed will be 0, so heading is irrelevant anyway. therefore, in that
  ; case we'll just leave it unmodified.
  if v1l != 0 or v1t != 0 [ set heading (theta - (atan v1l v1t)) ]

  ;; and do the same for other-particle
  ask other-particles [
    set speed sqrt ((v2t ^ 2) + (v2l ^ 2))
    set energy (0.5 * mass * (speed ^ 2))
    if v2l != 0 or v2t != 0 [ set heading (theta - (atan v2l v2t)) ]
  ]
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  initialization procedures
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;



to initialize-this-wall
;  set valve-1? false
 ; set valve-2? false
;  set pressure? false
  ifelse random 2 = 0 [set shape "cell"][set shape "cell2"]
  set color [255 255 255 120]
  let turn random 4
  rt (turn * 90)
end

to make-small-particles
  create-particles initial-#-small-food-particles [
    setup-particles false
    set shape "small-molecule"
    random-position
  ]
end

to make-large-particles
  create-particles initial-#-large-food-particles [
    setup-particles true
    set shape "large-molecule"
    random-position
  ]
end


to setup-particles  [is-large]; particles procedure
 ; set shape "circle"
  set size particle-size
  set energy 150
  set color-type (violet - 3)
  set color color-type
  set mass (10)  ; atomic masses of oxygen atoms

  set speed speed-from-energy
  set last-collision nobody
  set absorbed? false
  set large? is-large
  set swept? false
end


; Place particles at random, but they must not be placed on top of wall atoms.
; This procedure takes into account the fact that wall molecules could have two possible arrangements,
; i.e. high-surface area ot low-surface area.
to random-position ;; particles procedure
  let open-patches nobody
  let open-patch nobody
  set open-patches patches with [not any? turtles-here and pxcor != max-pxcor and pxcor != min-pxcor and pycor != min-pycor and pycor != max-pycor]
  set open-patch one-of open-patches

  ; Reuven added the following "if" so that we can get through setup without a runtime error.
  if open-patch = nobody [
    user-message "No open patches found.  Exiting."
    stop
  ]

  setxy ([ pxcor ] of open-patch) ([ pycor ] of open-patch)
  set heading random 360
  fd random-float .4
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; wall penetration error handling procedure
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; if particles actually end up within the wall

to remove-from-walls
  let this-wall walls-here with [ not hidden? ]

  if count this-wall != 0 [
    let available-patches patches with [ not any? walls-here ]
    let closest-patch nobody
    if (any? available-patches) [
      set closest-patch min-one-of available-patches [ distance myself ]
      set heading towards closest-patch
      setxy ([ pxcor ] of closest-patch)  ([ pycor ] of closest-patch)
    ]
  ]
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;GRAPHS;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
to do-plotting


end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;REPORTERS;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to-report speed-from-energy
  report sqrt (2 * energy / mass)
end

to-report energy-from-speed
  report (mass * speed * speed / 2)
end

to-report limited-particle-energy
  let limited-energy energy
  if limited-energy > max-particle-energy [ set limited-energy max-particle-energy ]
  if limited-energy < min-particle-energy [ set limited-energy min-particle-energy ]
  report limited-energy
end


; Copyright 2006 Uri Wilensky.
; See Info tab for full copyright and license.
@#$#@#$#@
GRAPHICS-WINDOW
255
10
911
487
-1
-1
18.0
1
10
1
1
1
0
1
1
1
-15
20
-8
17
1
1
1
ticks
30.0

BUTTON
125
10
225
50
go/pause
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

BUTTON
20
10
122
50
setup/reset
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
15
130
250
163
initial-#-small-food-particles
initial-#-small-food-particles
0
200
100.0
1
1
NIL
HORIZONTAL

SLIDER
15
245
250
278
villi-height
villi-height
0
10
10.0
1
1
NIL
HORIZONTAL

SLIDER
15
205
250
238
end-simulation-run-at
end-simulation-run-at
0
20
20.0
1
1
hrs
HORIZONTAL

MONITOR
40
290
200
335
simulated # hrs.
floor ticks
17
1
11

MONITOR
15
395
235
440
particles absorbed into blood
count particles with [absorbed?]
17
1
11

MONITOR
15
345
235
390
particles still in small intestine
count particles with [not absorbed?]
17
1
11

MONITOR
15
445
235
490
% of particles absorbed into blood
100 * count particles with [absorbed?] / count particles
1
1
11

SLIDER
15
165
250
198
initial-#-large-food-particles
initial-#-large-food-particles
0
100
0.0
1
1
NIL
HORIZONTAL

BUTTON
60
90
200
123
NIL
watch-a-particle
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
60
55
200
88
NIL
reset-perspective\n
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

@#$#@#$#@
## WHAT IS IT?

This model supports a drawing style interface for "sketching" up representations of new systems to explore related to gas behavior and gas particles.  This model is part of the "Connected Chemistry" curriculum http://ccl.northwestern.edu/curriculum/ConnectedChemistry/ which explores the behavior of gases.

Most of the models in the Connected Chemistry curriculum use the same basic rules for simulating the behavior of gases.  Each model highlights different features of how gas behavior is related to gas particle behavior.

In all of the models, gas particles are assumed to move and to collide, both with each other and with objects such as walls.

In this model, particles can be added, color coded, and sped up or slowed down, by drawing with the mouse cursor in the WORLD & VIEW.  Also, additional types of removable and replaceable walls can be added to the WORLD.

This model enables students to draw a model of a real world system and then test that model.  A wide range of real world systems can be modeled with this simple interface (e.g. diffusion of perfume from an uncapped container, hot gas mixed with a cold gas, mixtures of gases).

## HOW IT WORKS

The particles are modeled as hard balls with no internal energy except that which is due to their motion.  Collisions between particles are elastic.  Collisions with the wall are not.

The exact way two particles collide is as follows:
1. A particle moves in a straight line without changing its speed, unless it collides with another particle or bounces off the wall.
2. Two particles "collide" if they find themselves on the same patch. In this model, two turtles are aimed so that they will collide at the origin.
3. An angle of collision for the particles is chosen, as if they were two solid balls that hit, and this angle describes the direction of the line connecting their centers.
4. The particles exchange momentum and energy only along this line, conforming to the conservation of momentum and energy for elastic collisions.
5. Each particle is assigned its new speed, heading and energy.

As the walls of the box are heated, the sides of the walls will change color from a deep red (cool) to a bright red, to pink to a pale pink white (hot).  The walls contain a constant heat value throughout the simulation.

The exact way particles gain energy from the walls of the box is as follows:
1. Particles check their state of energy (kinetic).
2. They hit or bounce off the wall.
3. They find wall energy and set their new energy to be the average of their old kinetic energy and the wall energy.
4. They change their speed and direction after the wall hit.

## HOW TO USE IT

Buttons:
SETUP - sets up the initial conditions set on the sliders.
GO/STOP - runs and stops the model.
MOUSE INTERACTION - when this is set to "none - let the particles interact" the particles will move and interact with each other and the surroundings.  When set to any other value you can then click in the WORLD & VIEW to paint, erase, color, or add various objects and properties.

Sliders:
INITIAL-#-PARTICLES - sets the number of gas particles in the box when the simulation starts.
INITIAL-GAS-TEMPERATURE sets the initial temperature of the gas.

Switches:
SHOW-WALL-HITS? turn visualization of when particles hits the walls (as flashes) on or off

Choosers:
VISUALIZE-SPEED? allows you to visualize particle speeds.  For example, selecting "arrows", creates a representation of each particle velocity using a scalar arrow.  Selecting "shades" creates representation of each particle speed using a brighter (faster) or darker (slower) shade of the particle's color.

MOUSE-INTERACTION  sets the type interaction the user can do with the mouse in the WORLD & VIEW.  Possible settings include:
"none - let the particles interact" - particles move about
"draw basic wall" - adds a gray wall under the mouse cursor
"draw red removable wall" - adds a red wall under the mouse cursor which can be alternatively removed and replaced (like a valve) using the REMOVE/REPLACE RED WALL.
"draw green removable wall" - adds a green wall under the mouse cursor which can be alternatively removed and replaced (like a valve) using the REMOVE/REPLACE GREEN WALL.
"big eraser" - erases all objects (except the yellow box boundary walls) under the mouse cursor.
"slow down particles" - increase the current speed of the particles by 10%.
"speed up particles" - reduces the current speed of the particles by 10%.
"paint particles green" - recolors the particles under the mouse cursor green (other settings include orange and purple)
"add green particles" - adds a couple of new particles under the mouse cursor (other settings include orange and purple)

Plots:
- 1: TEMPERATURE OF GASES VS. TIME: plots the temperature of the different gases in the model, as indicated by their color (orange particles, green particles, and purple particles)

## THINGS TO NOTICE

The mouse interaction can be used while the model is running as well as when it is stopped.

## THINGS TO TRY

Create a model of how odors move throughout a room.  Why do some people smell the odor before others?  Does the layout of furniture, large objects, and walls in the room effect the movement of the odor?  How about the temperature of the air in the room?

Create a model of diffusion of a perfume from a closed container.  How would you represent the different gases (the perfume and the surrounding air)?  What shape will the container be?  How will you model a removable cap or lid?

Create a model of room filled with cold air and a different room filled with warm air.  How will represent these different rooms of air?  What could you add to show what happens when they mix?

Create a model of heat transfer that shows what happens to the energy of one very fast moving gas particle when it hits a bunch of very slow moving gas particles.  What does this show happening to the energy of the initial gas particles?

## RELATED MODELS

See GasLab Models
See other Connected Chemistry models.

## CREDITS AND REFERENCES

This model is part of the Connected Chemistry curriculum.  See http://ccl.northwestern.edu/curriculum/chemistry/.

We would like to thank Sharona Levy and Michael Novak for their substantial contributions to this model.

## HOW TO CITE

If you mention this model or the NetLogo software in a publication, we ask that you include the citations below.

For the model itself:

* Wilensky, U. (2006).  NetLogo Connected Chemistry 8 Gas Particle Sandbox model.  http://ccl.northwestern.edu/netlogo/models/ConnectedChemistry8GasParticleSandbox.  Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

Please cite the NetLogo software as:

* Wilensky, U. (1999). NetLogo. http://ccl.northwestern.edu/netlogo/. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

To cite the Connected Chemistry curriculum as a whole, please use:

* Wilensky, U., Levy, S. T., & Novak, M. (2004). Connected Chemistry curriculum. http://ccl.northwestern.edu/curriculum/chemistry/. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

## COPYRIGHT AND LICENSE

Copyright 2006 Uri Wilensky.

![CC BY-NC-SA 3.0](http://ccl.northwestern.edu/images/creativecommons/byncsa.png)

This work is licensed under the Creative Commons Attribution-NonCommercial-ShareAlike 3.0 License.  To view a copy of this license, visit https://creativecommons.org/licenses/by-nc-sa/3.0/ or send a letter to Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.

Commercial licenses are also available. To inquire about commercial licenses, please contact Uri Wilensky at uri@northwestern.edu.

<!-- 2006 ConChem -->
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

carbon
true
0
Circle -1184463 true false 68 83 134

carbon-activated
true
0
Circle -1184463 true false 68 83 134
Line -2674135 false 135 90 135 210

carbon2
true
0
Circle -955883 true false 30 45 210

cell
true
0
Polygon -2064490 true false 45 15 30 30 15 60 15 150 15 225 30 270 60 285 240 285 270 270 285 240 285 75 270 30 240 15 75 15
Rectangle -16777216 true false 0 150 15 195
Rectangle -16777216 true false 0 60 15 105
Rectangle -16777216 true false 105 0 150 15
Polygon -16777216 true false 285 285 270 270 225 285 240 300
Rectangle -16777216 true false 195 0 240 15
Rectangle -16777216 true false 285 105 300 150
Rectangle -16777216 true false 285 195 300 240
Polygon -16777216 true false 15 285 30 270 15 225 0 240
Polygon -16777216 true false 285 15 270 30 285 75 300 60
Rectangle -16777216 true false 60 285 105 300
Rectangle -16777216 true false 150 285 195 300
Polygon -16777216 true false 15 15 30 30 75 15 60 0
Circle -8630108 true false 165 60 88

cell2
true
0
Polygon -2064490 true false 45 15 30 30 15 60 15 150 15 240 15 255 60 285 240 285 270 285 285 240 285 60 285 45 240 15 60 15
Rectangle -16777216 true false 0 180 15 225
Rectangle -16777216 true false 0 90 15 135
Rectangle -16777216 true false 75 0 120 15
Rectangle -16777216 true false 165 0 210 15
Rectangle -16777216 true false 285 75 300 120
Rectangle -16777216 true false 285 165 300 210
Polygon -16777216 true false 300 30 285 45 240 15 255 0
Rectangle -16777216 true false 90 285 135 300
Circle -8630108 true false 135 105 88
Rectangle -16777216 true false 180 285 225 300
Polygon -16777216 true false 0 270 15 255 60 285 45 300
Polygon -16777216 true false 30 0 45 15 15 60 0 45
Polygon -16777216 true false 270 300 255 285 285 240 300 255

circle
true
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 16 16 270
Circle -16777216 true false 46 46 210

clock
true
0
Circle -7500403 true true 30 30 240
Polygon -16777216 true false 150 31 128 75 143 75 143 150 158 150 158 75 173 75
Circle -16777216 true false 135 135 30

co2
true
0
Circle -13791810 true false 83 165 134
Circle -13791810 true false 83 0 134
Circle -1184463 true false 83 83 134

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

eraser
false
0
Rectangle -7500403 true true 0 0 300 300

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

heater-a
false
0
Rectangle -7500403 true true 0 0 300 300
Rectangle -16777216 true false 90 90 210 210

heater-b
false
0
Rectangle -7500403 true true 0 0 300 300
Rectangle -16777216 true false 30 30 135 135
Rectangle -16777216 true false 165 165 270 270

hex
false
0
Polygon -7500403 true true 0 150 75 30 225 30 300 150 225 270 75 270

hex-valve
false
0
Rectangle -7500403 false true 0 0 300 300
Polygon -7500403 false true 105 60 45 150 105 240 195 240 255 150 195 60

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

large-molecule
true
0
Polygon -7500403 true true 150 165 150 165 135 180 135 195 150 210 165 195 165 180
Polygon -7500403 true true 150 120 150 120 135 135 135 150 150 165 165 150 165 135
Polygon -7500403 true true 150 75 150 75 135 90 135 105 150 120 165 105 165 90

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

nitrogen
true
0
Circle -10899396 true false 83 135 134
Circle -10899396 true false 83 45 134

outline
true
0
Circle -7500403 false true 0 0 300

oxygen
true
0
Circle -13791810 true false 83 135 134
Circle -13791810 true false 83 45 134

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

small-molecule
true
0
Polygon -7500403 true true 150 135 150 135 135 150 135 165 150 180 165 165 165 150

spray paint
false
0
Rectangle -7500403 false true 0 0 300 300
Circle -7500403 false true 75 75 150

square
false
0
Rectangle -7500403 true true 0 0 300 300

square 2
false
3
Rectangle -6459832 true true 0 -15 300 300

square 3
false
3
Rectangle -6459832 true true 0 -15 300 300
Rectangle -7500403 true false 0 285 300 300

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

valve-1
false
0
Rectangle -7500403 false true 0 0 300 300
Rectangle -7500403 false true 120 120 180 180

valve-2
false
0
Rectangle -7500403 false true 0 0 300 300
Rectangle -7500403 false true 60 120 120 180
Rectangle -7500403 false true 165 120 225 180

valve-hex
false
0
Rectangle -7500403 false true 0 0 300 300
Polygon -7500403 false true 105 60 45 150 105 240 195 240 255 150 195 60

valve-triangle
false
0
Rectangle -7500403 true true 0 0 300 300
Polygon -16777216 true false 150 45 30 240 270 240

valves
false
0
Rectangle -7500403 false true 0 0 300 300

wall
false
0
Rectangle -7500403 true true 0 0 300 300

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.0.2
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
1
@#$#@#$#@
