;;;;;;CystiAgent Model;;;;;;;

;;;;PRIMING;;;;

extensions [gis]

globals [
  ;;; PARAMETER VARIABLES;;;

  ;;Village characteristics;;
  prop-pig-owners                                ;P0.1: Proportion of households that raise pigs
  prop-corrals                                   ;P0.2: Proportion of pig-owning households with corrals (or tie-up areas)
  prop-latrines                                  ;P0.3: Proportion of households with a latrine or bathroom
  humans-per-household                           ;P0.4: POISSON mean for number of humans per household
  pigs-per-household                             ;P0.5: EXPONENTIAL mean for number of pigs per household
  baseline-tn-prev                               ;P0.6: Baseline prevalence of taeniasis
  baseline-light-infection                       ;P0.7: Baseline prevalence of light cyst infection
  baseline-heavy-infection                       ;P0.8: Baseline prevalence of heavy cyst infection

  ;;Pig slaughter age;;
  slaughter-age-mean                             ;P1.1: Log-mean of pig age at slaughter (in months)
  slaughter-age-sd                               ;P1.1: Log-SD of pig age at slaughter (in months)

  ;;Human travel;;
  traveler-prop                                  ;P2.1: Proportion of households with a frequent traveler
  travel-freq                                    ;P2.2: Frequency of trips among traveler households (departure for a trip every X weeks)
  travel-duration                                ;P2.3: Mean (ENTER DISTRIBUTION) duration of trips
  travel-incidence                               ;P2.4: Incidence of tapeworm infection for traveling humans (to other endemic areas)

  ;;Tapeworm lifespan;;
  tn-incubation                                  ;P3.2: Tapeworm incubation time (weeks)
  tn-lifespan-mean                               ;P3.1: Tapeworm infection duration (mean)
  tn-lifespan-sd                                 ;P3.1: Tapeworm infection duration (sd)

  ;;Environmental contamination;;
  latrine-use                                    ;P4.1: Proportion of households with a latrine for which the latrine is in GOOD condition and ALWAYS used
  cont-radius-mean                               ;P4.2: Log-mean of the radius of contamination around tapeworm carrier house
  cont-radius-sd                                 ;P4.2: Log-SD of the radius of contamination around tapeworm carrier house

  ;;Environmental egg decay;;
  decay-mean                                     ;P5.1: Environmental egg decay rate parameter (probably of decay per week)

  ;;Pig roaming and corral use;;
  corral-always                                  ;P6.1: Corral-use (proportion ALWAYS) among pig-owners with corrals
  corral-sometimes                               ;P6.1: Corral-use (proportion SOMETIMES) among pig-owners with corrals
  prop-corral-sometimes                          ;P6.2: Proportion of pigs that are corraled among households household that SOMETIMES use corral
  home-range-mean                                ;P6.3: Home range radius (mean)
  home-range-sd                                  ;P6.3: Home range radius (sd)

  ;;Sale and distribution of pork;;
  pigs-sold                                      ;P7.1: Proportion of pigs that are sold prior to slaughter
  pigs-exported                                  ;P7.2: Proportion of sold pigs that are exported
  pig-import-rate                                ;P7.3: Rate of import of pigs from external communities
  import-prev                                    ;P7.4: Prevalence of infection among pigs imported from external communities
  light-to-heavy-prop                            ;P7.5: Proportion of infected imported pigs that are lightly infected (compared to heavy)
  hh-only-pork                                   ;P7.6-a: Proportion of pigs for which all meat consumed exclusively by pig-owner households
  sold-pork                                      ;P7.6-b: Proportion of pigs for which all meat was sold within the village (none consumed at home)
  shared-pork-dist                               ;P7.7: (NOT ACTIVE) Number of households a pig is divided between when slaughtered for sale
  shared-pork-hh                                 ;P7.8: Proportion of shared pork that is consumed by the pig-owner's household

  ;;;SPATIAL VARIABLES;;;
  household-dataset                              ;Shapefile specifying household coordinates in village
  data-location                                  ;File-path of shapefile for households coordinates
  roads-dataset                                  ;Shapefile of roads (if available)
  rivers-dataset                                 ;Shapefile of rivers (if available)
  scale                                          ;(Auto-generated variable) Transformation constant of meters to pixels in patch-environment
  hor                                            ;Width (in patches) of environment
  ver                                            ;Length (in patches) of environment

  ;;;MONITORS - ROUTINE;;;
  num-households                                 ;(Auto-generated variables) number of households in the environment
  total-pig-owners                               ;(Auto-generated variables) number of pig-owning households

  ;;;MONITORS - SEROINCIDENCE (PIGS);;;
  week-counter                                  ;(Auto-generated variable) counting weeks of exposure for 4-month incidence window in pigs
  seroconversions                               ;Counts the number of seronegative pigs that become seropositive during 4-month incidence window
  seroeligible                                  ;Counts the number of pigs that are seronegative at the beginning of the 4-month incidence window and pigs born into the population
  seroincidence                                 ;Calculation of seroconversions / seroeligible


 ;;;INTERVENTION SETTINGS;;;

  ;;General - human;;
  elisa-sens                                     ;Sensitivity of the ELISA copro-antigen assay for taeniasis screening
  treat1-eff                                     ;Efficacy of single-dose treatment for human taeniasis
  treat2-eff                                     ;Efficacy of two-dose treatment for human taeniasis
  screen-trt-eff                                 ;Efficacy of follow-to-cure treatment of human taeniasis after stool screening
  screen-trt-part                                ;Participation in treatment for human taeniasis AFTER a positive stool screening
  treat1-part                                    ;Proportion of humans that receive 1 dose of treatment for taeniasis
  treat2-part                                    ;Proportion of humans that receive 2 doses of treatment for taeniasis
  treat-mass-part                                ;Proportion of humans that participate in mass treatment
  screen-part                                    ;Proportion of humans that participate in stool screening for taeniasis

  ;;General - pigs;;
  age-eligible                                   ;Age at which pigs are eligible to participate in interventions
  treat-part-p                                   ;Proportion of pigs that receive treatment for porcine cysticercosis
  oxf-protection                                 ;Length of time infected pigs are protected from cyst infection after treatment with oxfendazole
  vacc-part                                      ;Proportion of pigs that participation in vaccination with TSOL18
  vacc-eff                                       ;Efficacy of TSOL18 vaccine for porcine cysticercosis (after 2 doses)
  saca-part                                      ;Proportion of pigs that are purchased and slaughtered in a community-wide necropsy ("saca")

  ;;Ring Strategy;;
  ring-size                                     ;Radius (in meters) of intervention rings after screening a tongue-positive pig (RING STRATEGY ONLY)
  tongue-part                                   ;Proportion of pigs screened with tongue inspection (RING STRATEGY ONLY)
  tongue-sens                                   ;Sensitivity (probability) of detecting a heavily infected pig with tongue inspection (RING STRATEGY ONLY)
  tongue-fp                                     ;False positive rate during tongue inspection (proportion on uninfected pigs with positive test) (RING STRATEGY ONLY) ;
  treat-final-part                              ;Proportion of humans that participate in a single round of mass treatment at the end of the study
]


;;;AGENTS;;;

breed [                                         ;INACTIVE: only for visualization of corrals
  corral2s corral2]

breed [
  humans human]

breed [
  pigs pig]

breed [                                         ;A slaughtered pig is divided into "carne" to be distributed within and between households
  carnes carne]

breed [                                         ;Tapeworm carriers contaminate the environment with "eggs" (not intended to represent a single biological egg, but the presence of eggs)
  eggs egg]

patches-own [
  ;;ENVIRONMENTAL VARIABLES;;
  household                                     ;FIXED, BINARY: Is the patch a household?
  pig-owner                                     ;FIXED, BINARY: Does the household raise pigs?
  corral                                        ;FIXED, BINARY: Does the pig-owning house have a pig corral?
  corraluse                                     ;FIXED, ORDINAL: What is the frequency of corral-use at that household (always, sometimes, never)?
  latrine                                       ;FIXED, BINARY: Does the household own a latrine?
  latrineusers                                  ;FIXED, BINARY: Do the household members always use the latrine?
  light-infected-pork-meal                      ;DYNAMIC: Did the household consume a light-infected pork meal this week?
  heavy-infected-pork-meal                      ;DYNAMIC: Did the household consume a light-infected pork meal this week?
  travelerhh                                    ;FIXED, BINARY: Does a member of the household regularly travel to other endemic areas?
  cont-radius-hh                                ;FIXED, CONTINUOUS: Distance from a household that environmental contamination is deposited (random angle)
]

humans-own [
  ;;HUMAN VARIABLES;;

  ;General;
  home-x                                        ;FIXED: X-coordinate of household location on grid
  home-y                                        ;FIXED: Y-coordinate of household location on grid
  latrine-user                                  ;FIXED, BINARY: Does the person belong to a latrine-using household?

  ;Infection status;
  tapeworm                                      ;DYNAMIC, BINARY: Is the person infected with a tapeworm?
  infection-duration                            ;DYNAMIC, DISCRETE: Pre-determined duration of tapeworm infection (new value randomly generated for every new infection)
  mature-tn                                     ;DYNAMIC: If the person is a tapeworm carrier, is the tapeworm mature (producing eggs/proglottids)?
  tn-released                                   ;DYNAMIC, BINARY: If the person has a mature tapeworm infection and does not use latrines, was the egg/proglottid released into the environment?
  time-since-infection                          ;DYNAMIC, DISCRETE: How many weeks since tapeworm infection occured (auto-generated counter)

  ;Travel-related;
  traveler                                      ;FIXED: Is the person a regular traveler to other endemic areas?
  travel-ticker                                 ;DYNAMIC: Automatic counter that determines time until the next trip to endemic areas (for travelers)
  traveling                                     ;DYNAMIC: Is the person travel outside of the village this week?
  travel-time                                   ;DYNAMIC: Amount of time this traveler will be out of the village on the current trip

  ;Interventions;
  screen-pos                                    ;INTERVENTIONS: Was this person screened positive for taeniasis?
  eligible                                      ;INTERVENTIONS: Is this person eligible for screening/treatment based on presence in a ring (RING STRATEGY ONLY)
]

pigs-own [
  ;;PIG VARIABLES;;

  ;Infection status;
  susceptible                                  ;DYNAMIC, BINARY: Is the pig susceptible to infection (not infected or resistant)?
  light-exp                                    ;DYNAMIC, BINARY: Was the pig exposed to eggs (light contamination) in the environment this week?
  heavy-exp                                    ;DYNAMIC, BINARY: Was the pig exposed to proglottids (heavy contamination) in the environment this week?
  heavy-infected                               ;DYNAMIC, BINARY: Is the pig heavily infected with cysts (>100 cysts)?
  light-infected                               ;DYNAMIC, BINARY: Is the pig lightly infected with cysts (<100 cysts)?
  seropositive                                 ;DYNAMIC, BINARY: Is the pig positive on a serological test (EITB)?

  ;General;
  home-x                                       ;FIXED: X-coordinate of the pig's household on grid
  home-y                                       ;FIXED: Y-coordinate of the pig's household on grid
  corraled                                     ;FIXED, ORDINAL: How frequently is the pig corraled (ALWAYS, SOMETIMES, NEVER)
  home-range                                   ;FIXED, CONTINUOUS: If the pig is free-roaming, what is the radius (in meters) of the pig's roaming area (home-range)?
  age                                          ;DYNAMIC, DISCRETE: Current age of the pig (in weeks)
  slaughter-age                                ;FIXED, DISCRETE: Pre-determined age (in weeks) at which pig will be slaughtered
  marked-for-slaughter                         ;DYNAMIC, BINARY: Is the pig due for slaughter this week (reached slaughter-age)?

  ;Interventions;
  eligible                                     ;INTERVENTIONS: Is the pig eligible for treatment based on precense in a ring (RING STRATEGY ONLY)
  protected                                    ;INTERVENTIONS: Is the pig protected from future infection because it received vaccination and/or OFZ treatment while infected
  protected-time                               ;INTERVENTIONS: Amount of time remaining that pig is protected from infection (automatic counter)
  vacc-dose                                    ;INTERVENTIONS: Number of vaccine doses received
  vaccinated                                   ;INTERVENTIONS: Did the pig receive the full vaccine dose and full protection?
]

carnes-own[
  light-infected                               ;BINARY: Is pork meat lightly infected?
  heavy-infected                               ;BINARY: Is pork meat heavily infected?
  hh                                           ;BINARY: Is pork meat consumed by the current household or sold/donated to a different household in the village?
]

eggs-own[
  id                                           ;DISCRETE: Auto-generated index of the human that released the egg
  proglottid                                   ;BINARY: Is this a proglottid (heavy contamination)?
  ova                                          ;BINARY: Is this eggs (light contamination)?
  cont-radius-egg                              ;DISCRETE: Distance radius from the household that the contamination may appear within
  time-since-infection                         ;DISCRETE: Number of weeks since the human tapeworm carrier was infected (automatic counter, mirrors human counter)
  infection-duration                           ;DISCRETE: Total number of weeks the human tapeworm infection will last (mirrors human variable)
  traveling-human                              ;BINARY: Is the human that produced the egg currently traveling (mirrors human variable)
]

;;;;WORLD SETUP;;;;

to setup
  clear-all
  setup-villages
  resize-world (- hor) hor (- ver) ver        ;Changes the dimensions of the world if specified for a particular village
  setup-patches
  ifelse Households = "Random"                               ;ADDED option to allow for model processing without access to SHAPEFILES
    [ask n-of 100 patches [set household true]
     set calibrated? false]
    [draw-gis-layers]
  setup-globals
  setup-pigs
  setup-humans
  reset-ticks
end

to setup-villages
    set data-location "CystiAgentTestData.shp"
    set hor 180
    set ver 200
end

to setup-patches
  ask patches [
    set household false
    set latrine false
    set latrineusers false
    set corral false
    set light-infected-pork-meal 0
    set heavy-infected-pork-meal 0
    set travelerhh false
    set cont-radius-hh false]
end

to draw-gis-layers
  set household-dataset gis:load-dataset data-location
  gis:set-transformation (gis:envelope-of household-dataset) (list (min-pxcor + 50) (max-pxcor - 50) (min-pycor + 50) (max-pycor - 50))
  foreach gis:feature-list-of household-dataset [ vector-feature ->
    let location gis:location-of (first(first(gis:vertex-lists-of vector-feature)))
    if not empty? location [
      ask patch item 0 location item 1 location [
        set household true]]]

;;If roads or rivers layers are available, they can be drawn in with this, but are not converted into patch variables

;set roads-dataset gis:load-dataset "C:/Users/pray/Documents/PhD Dissertation/NetLogoShapefiles/TTEMP_Roads_proj.shp"
;  foreach gis:feature-list-of roads-dataset [ vector-feature ->
;    gis:set-drawing-color gray
;    gis:draw vector-feature 2.0]
;
;  set rivers-dataset gis:load-dataset "C:/Users/pray/Documents/PhD Dissertation/NetLogoShapefiles/TTEMP_Rivers_proj.shp"
;  foreach gis:feature-list-of rivers-dataset [ vector-feature ->
;    gis:set-drawing-color 96
;    gis:draw vector-feature 2.0]
end

to setup-globals
;;CALIBRATED TUNING VARIABLES;;
if calibrated? = true [
  set light-inf 0.0163                                              ;Tuning paramter (via calibration): Probability of light pig infection upon egg consumption
  set heavy-inf 0.0046                                               ;Tuning paramter (via calibration): Probability of heavy pig infection upon proglottid consumption
  set pl2h 0.0074                                                    ;Tuning paramter (via calibration): Probability of human taeniasis upon lightly inf. pork consumption
  set ph2h 0.0059                                                    ;Tuning paramter (via calibration): Probability of human taeniasis upon heavily inf. pork consumption
  set light-all 0.0141                                               ;Tuning paramter (via calibration): Probability of light pig infection for all pigs
  set heavy-all 0.0182                                               ;Tuning paramter (via calibration): Probability of heavy pig infection for all pigs
  set light-sero 0.15                                                ;Tuning paramter (via calibration): Probability of seropositivity after light exposure
  set heavy-sero 0.1]                                                 ;Tuning paramter (via calibration): Probability of seropositivity after heavy exposure

;;MODEL PARAMETERS;;					
  set prop-pig-owners 0.39                                            ;(specified by village) P0.1: Proportion of households that raise pigs
  set prop-corrals 0.37                                               ;(specified by village) P0.2: Proportion of pig-owning households with corrals (or tie-up areas)
  set prop-latrines 0.78                                              ;(specified by village) P0.3: Proportion of households that have a latrine or bathroom
  set humans-per-household 3.86                                       ;(specified by village) P0.4: POISSON mean for number of humans per household
  set pigs-per-household 2.16                                         ;(specified by village) P0.5: EXPONENTIAL mean for number of pigs per household
  set baseline-tn-prev 0.0293                                         ;(specified by village) P0.6: Baseline prevalence of taeniasis
  set baseline-light-infection 0.1648                                 ;(specified by village) P0.7: Baseline prevalence of light cyst infection
  set baseline-heavy-infection 0.0674                                 ;(specified by village) P0.8: Baseline prevalence of heavy cyst infection

  set slaughter-age-mean 2.279                                        ;P1.1: Log-mean pig age at slaughter (in months, will be later converted to weeks)
  set slaughter-age-sd 0.515                                          ;P1.1: Log-SD of pig age at slaughter (in months)

  set traveler-prop 0.423                                             ;P2.1: Proportion of households with a frequent traveler
  set travel-freq 8                                                   ;P2.2: Frequency of trips among traveler households (departure for a trip every X weeks)
  set travel-duration 1.75                                            ;P2.3: Mean (ENTER DISTRIBUTION) duration of trips
  set travel-incidence 0.000231                                       ;P2.4: Incidence of taeniasis among humans traveling to other endemic areas

  set tn-lifespan-mean 104                                            ;P3.1: Tapeworm infection duration (mean)
  set tn-lifespan-sd 50                                               ;P3.1: Tapeworm infection duration (sd)
  set tn-incubation 8                                                 ;P3.2: Tapeworm incubation time

  set latrine-use 0.25                                                ;P4.1: Proportion of households with a latrine in GOOD condition that they ALWAYS use
  set cont-radius-mean 3.27                                           ;P4.2: Log-mean of the radius of contamination around tapeworm carrier house
  set cont-radius-sd 0.547                                            ;P4.2: Log-SD of the radius of contamination around tapeworm carrier house

  set decay-mean 0.125                                                ;P5.1: Environmental egg decay rate parameter (probably of decay per week)

  set corral-always 0.05                                              ;P6.1: Corral-use (proportion ALWAYS) among pig-owners with corrals
  set corral-sometimes 0.57                                           ;P6.1: Corral-use (proportion SOMETIMES) among pig-owners with corrals
  set prop-corral-sometimes 0.32                                      ;P6.2: Proportion of pigs that are corraled among households household that SOMETIMES use corral
  set home-range-mean 3.79                                            ;P6.3: Home range radius (mean)
  set home-range-sd 0.552                                             ;P6.3: Home range radius (sd)

  set pigs-sold 0.514                                                 ;P7.1: Proportion of pigs that are sold prior to slaughter
  set pigs-exported 0.731                                             ;P7.2: Proportion of sold pigs that are exported
  set pig-import-rate 0.00105                                         ;P7.3: Rate of import of pigs from external communities
  set import-prev 0.134                                               ;P7.4: Prevalence of infection among pigs imported from external communities
  set light-to-heavy-prop 0.755                                       ;P7.5: Proportion of infected imported pigs that are lightly infected (compared to heavy)
  set hh-only-pork 0.396                                              ;P7.6-a: Proportion of pigs for which all meat consumed exclusively by pig-owner households
  set sold-pork 0.115                                                 ;P7.6-b: Proportion of pigs for which all meat was sold within the village (none consumed at home)
  set shared-pork-dist 0                                              ;P7.7: Number of households a pig is divided between when slaughtered for sale (NOT ACTIVE)
  set shared-pork-hh 0.80                                             ;P7.8: Proportion of shared pork that is consumed by the pig-owner's household

  set scale (item 1 gis:world-envelope - item 0 gis:world-envelope) / (max-pxcor * 2)    ;Determines meters to cells conversion
  set num-households count patches with [household = true]                               ;Count number of households for future calculations
  set total-pig-owners round ((prop-pig-owners) * num-households)                        ;Calculate total pig-owning households

;;INTERVENTION SETTINGS;;

  ;General - human;
  set screen-part 0.75
  set elisa-sens 0.964
  set screen-trt-part 0.918
  set screen-trt-eff 0.933
  set treat1-part 0.15
  set treat2-part 0.70
  set treat1-eff 0.766
  set treat2-eff 0.866
  set treat-mass-part 0.78
  set treat-final-part 0.736

  ;General - pig;
  set age-eligible 10
  set treat-part-p 0.60
  set oxf-protection 17
  set vacc-part 0.7
  set vacc-eff 0.99
  set saca-part 0.482

  ;Ring Strategy;
  set ring-size 100
  set tongue-part 0.72
  set tongue-sens 0.909
  set tongue-fp 0.021
end

to setup-pigs
  ask n-of total-pig-owners patches with [household = true]                                                                       ;Set pig-owning households
    [set pig-owner true]
  ask n-of (round((prop-corrals) * count patches with [pig-owner = true])) patches with [pig-owner = true]                        ;Set pig-owning households with corrals
    [set corral true]
  ask patches with [pig-owner = true]                                                                                             ;Set household corral-use (always, sometimes, never) among pig-owners with corrals
    [set corraluse "never"]
  ask n-of (round((corral-always) * count patches with [corral = true])) patches with [corral = true]
    [set corraluse "always"]
  ask n-of (round((corral-sometimes) * count patches with [corral = true])) patches with [corral = true and corraluse = "never"]
    [set corraluse "sometimes"]
  ask patches [
     if pig-owner = true [
      sprout-pigs ceiling random-exponential pigs-per-household [                                             ;Populate pig-owning households with pigs
        set shape "pig"
        set color pink
        set size 10]]]
    ;if corral = true [                                                                                       ;Create object (turtle) to display corrals (INACTIVE)
     ;   sprout-corral2s 1 [
      ;  set shape "square 2"
       ; set size 20
        ;if corraluse = "always" [set color green]
        ;if corraluse = "sometimes" [set color yellow]
        ;if corraluse = "never" [set color red]]]]
  ask pigs [                                                                                                  ;Set initial variables for pigs
    set susceptible true
    set light-exp false
    set heavy-exp false
    set seropositive false
    set heavy-infected false
    set light-infected false
    set marked-for-slaughter false
    set eligible false
    set protected false
    set protected-time 0
    set vacc-dose 0
    set vaccinated false
    set corraled "never"
    set home-x xcor
    set home-y ycor
    if [corraluse] of patch-here = "always"                                                                   ;Assign corral usage for pigs based on household corral practice
      [set corraled "always"]
    if [corraluse] of patch-here = "sometimes"
       [set corraled "sometimes"]
    set slaughter-age round(4.35 * (exp(random-normal slaughter-age-mean slaughter-age-sd)))                  ;Set age at which each pig will be slaughteed (in weeks, 4.35 weeks per month)
    set home-range (exp(random-normal home-range-mean home-range-sd))                                         ;Set the radius of each pigs home-range areae
    set age (slaughter-age - random slaughter-age)                                                            ;Randomly assign starting age for first batch of pigs
    if random-float 1 < baseline-heavy-infection [heavy-infect]                                               ;Infect pigs at baseline (heavy)
    if (not heavy-infected) and (random-float 1 < baseline-light-infection) [light-infect]]                   ;Infect pigs at baseline (light)
  ;set total-pigs count pigs                                                                                  ;Count pigs at baseline (for INCIDENCE calculation - SENSITIVITY ANALYSIS)
end

to setup-humans
  ask patches [
    if household [
      let x random-poisson humans-per-household                                                               ;Assign humans to household (no zeros allowed)
      ifelse x > 0
        [sprout-humans x [
          set shape "house"
          set size 10
          set color yellow
          move-to patch-here]]
        [sprout-humans 1 [
          set shape "house"
          set size 10
          set color yellow
          move-to patch-here]]
      set cont-radius-hh (exp(random-normal cont-radius-mean cont-radius-sd))]]                              ;Set radius of environmental contamination for each household
  ask n-of (prop-latrines * num-households) patches with [household = true]                                  ;Assign latrines to households
    [set latrine true]
  ask n-of (latrine-use * count patches with [latrine = true]) patches with [latrine = true]                 ;Assign latrine use (BINARY) to households with latrines
    [set latrineusers true]
  ask n-of (traveler-prop * num-households) patches with [household = true]                                  ;Assign households as travelers (BINARY)
    [set travelerhh true]
  ask humans [                                                                                               ;Set initial variables for humans
    set latrine-user false
    set tapeworm false
    set traveling false
    set traveler false
    set time-since-infection 0
    set infection-duration 0
    set travel-time 0
    set mature-tn false
    set tn-released false
    set screen-pos false
    set eligible false
    set home-x xcor
    set home-y ycor
    if latrineusers = true [                                                                                ;Assign latrine use of household to individuals
      set latrine-user true]
    if random-float 1 < baseline-tn-prev [                                                                  ;Infect humans with taeniasis at baseline
      infect-humans-baseline]]
ask patches with [travelerhh = true][                                                                       ;Assign one person per traveling household to be a traveler
  ask one-of humans-here [
    set traveler true
    set travel-ticker random (travel-freq + 1)]]                                                           ;Begin timer to countdown the weeks until the next trip
;set num-humans count humans                                                                               ;Count number of human at baseline (for INCIDENCE calculation - SENSITIVITY ANALYSIS)
end

to infect-humans-baseline                                                                                 ;Infect humans at baseline
  set tapeworm true
  let x random-normal tn-lifespan-mean tn-lifespan-sd                                                     ;Set predetermined duration of tapeworm infection (0-truncated)
    ifelse x < 0
      [set infection-duration 0]
      [set infection-duration round x]
  let y random infection-duration                                                                         ;Randomly assign time-since-infection to a starting value
  set time-since-infection (infection-duration - y)
  if time-since-infection >= tn-incubation [                                                              ;Set tapeworm maturity
    set mature-tn true]
   ;set size 20 set color blue]                                                                           ;OPTIONAL: change color for tapeworm carriers
  if tapeworm = true and mature-tn = true and latrine-user = false [                                      ;Contamination only occurs in households with mature tapeworm carriers not using latrines
    set tn-released true
    hatch-eggs 1 [                                                                                        ;Set characteristics of environmental contamination
      set id [who] of myself                                                                              ;Adopt index of human that produced egg/proglottid
      set proglottid true                                                                                 ;Both egg and proglottid (meaning it can cause heavy or light infection)
      set ova true
      set traveling-human false                                                                           ;Human cannot be traveling yet (baseline infection)
      set cont-radius-egg cont-radius-hh
      move-to one-of patches in-radius (cont-radius-egg / scale)                                          ;Place egg/proglottid on any patch within the contamination radius
      set size 10
      set shape "circle"
      set color green
      ifelse x < 0                                                                                        ;Set same infection duration and time-since infection as specified for human (mirror)
        [set infection-duration 0]
        [set infection-duration round x]
      set time-since-infection (infection-duration - y)]]
 end


;;;;RUN;;;;

to go
  ask humans [advance-infection]
  ask eggs [eggs-advance]
  ask humans [return-home-humans]
  ask humans [travel]
  ask pigs [reset-exposure]
  ask pigs [mark-for-slaughter]
  ask pigs [birth-pigs]
  ask pigs [distribute-pigs]
  ask pigs [import-pigs]
  ask pigs [slaughter-pigs]
  ask carnes [distribute-pork]
  ask carnes [die]
  ask humans [eat-pork]
  ask humans [infect-travelers]
  ask pigs [slaughter]
  ask pigs [eat-eggs]
  ask pigs [seroconvert]
  ask pigs [infect-pigs]
  ask pigs [set age age + 1]
  ask humans [mature-infection]
  ask humans [tapeworm-die]
  ask eggs [environmental-decay]
  ask eggs [stop-contaminating]
  ask humans [clear-pork-meals]
  ask pigs [advance-protection]
  ;ask pigs [assign-color-pigs]                                                     ;OPTIONAL: if changing color for infection status
  ;set count-cyst count-heavy + count-light                                         ;Add up incident pig infection (for INCIDENCE calculation - SENSITIVITY ANALYSIS)
  ;recorder                                                                         ;Initiates script to record prevalence at given time-points (for CALIBRATION)
  track-incidence                                                                  ;Initiates script to record pig sero-incidence
  tick
end

to advance-infection                                                                                     ;;Advance taeniasis infection by 1 week;;
  if tapeworm = true [set time-since-infection time-since-infection + 1]
end

to eggs-advance                                                                                          ;;Advance age of environmental contamination by 1 week;;
  set time-since-infection time-since-infection + 1
end

to return-home-humans                                                                                    ;;Humans that are currently travel return home;;
  if traveling = true [
    set travel-time travel-time - 1
    if travel-time = 0 [
      set xcor home-x set ycor home-y
      if tapeworm = true and latrine-user = false and mature-tn = true and tn-released = false [         ;Humans that have taeniasis and return home begin contaminating the environment (only if tapeworm is mature and contamination not already released)
          set tn-released true
          hatch-eggs 1 [
            set id [who] of myself
            set proglottid true
            set ova true
            set traveling-human false
            set infection-duration infection-duration
            set time-since-infection tn-incubation
            set cont-radius-egg cont-radius-hh
            move-to one-of patches in-radius (cont-radius-egg / scale)
            set size 10
            set shape "circle"
            set color green]]
      if tapeworm = true and latrine-user = false and mature-tn = true and tn-released = true [         ;When humans return from traveling, if taeniasis is active and environmental contamination already begun, resume heavy contamination
        ask eggs with [id = [who] of myself] [
          set proglottid true
          set color green]]
      set traveling false
      ask eggs with [id = [who] of myself] [
        set traveling-human false]]]
end

to travel                                                                                               ;;Initiate travel of humans;;
  if traveler = true and travel-ticker = 0 [                                                            ;Begin travel when travel timer gets to 0
    set traveling true
    ask eggs with [id = [who] of myself] [                                                              ;When travel begins, associated environmental contamination turns to light (no active proglottids)
      set proglottid false
      set traveling-human true
      set color brown]
    set travel-time ceiling random-exponential travel-duration                                          ;Assign duration of travel based on exponential distribution
    set travel-ticker travel-freq + 1                                                                   ;Reset travel timer back to constant interval until next trip
    set xcor max-pxcor set ycor max-pycor]                                                              ;Human that is traveling is placed in the top right corner of the world
  if traveler = true [
    set travel-ticker travel-ticker - 1]
end

to reset-exposure                                                                                       ;Set all exposure to NULL to begin the week
  set light-exp false
  set heavy-exp false
end

to mark-for-slaughter                                                                                   ;If a pig reaches it pre-determined slaughter age, it is "marked" for slaughter
   if age = slaughter-age [set marked-for-slaughter true]
end

to birth-pigs                                                                                          ;For every pig that will be slaughtered, a new pig is born into the same household (to maintain constant population size)
  if marked-for-slaughter = true [
    hatch 1 [                                                                                          ;Set initial variables for new pigs (same as above for initial pig population)
        set shape "pig"
        set color pink
        set size 10
        set susceptible true
        set light-exp false
        set heavy-exp false
        set seropositive false
        set heavy-infected false
        set light-infected false
        set marked-for-slaughter false
        set eligible false
        set protected false
        set protected-time 0
        set vacc-dose 0
        set vaccinated false
        set corraled "never"
        set age 0
        set home-x xcor
        set home-y ycor
    if [corraluse] of patch-here = "always"
      [set corraled "always"]
    if [corraluse] of patch-here = "sometimes"
      [set corraled "sometimes"]
      set slaughter-age round(4.35 * (exp(random-normal slaughter-age-mean slaughter-age-sd)))
      set home-range (exp(random-normal home-range-mean home-range-sd))]
    ;set total-pigs total-pigs + 1                                                                  ;Add new pigs to population (for INCIDENCE calculation - SENSIVITY ANALYSIS)
    set seroeligible seroeligible + 1]                                                             ;Add new seronegative pigs to the population (to calculuate pig sero-incidence)
end

to distribute-pigs                                                                                  ;Sell and export pigs that will be slaughtered
  if marked-for-slaughter = true [
    let x random-float 1
    if x < pigs-sold [                                                                              ;Proportion of pigs that are sold prior to slaughter
      let y random-float 1
      ifelse y < pigs-exported                                                                      ;Proportion of sold pigs that are exported
         [die]                                                                                      ;If pigs are exported, they "die" and a not consumed
         [move-to one-of patches with [household = true]]]]                                         ;If pigs are sold, they go to a random household within the village
end

to import-pigs                                                                                     ;Import infected pigs
  let x random-float 1
  if x < (pig-import-rate * import-prev) [                                                         ;Probability of a pig being imported and that pig being infected, per pig in the population
    hatch-pigs 1 [
    set marked-for-slaughter true                                                                  ;Imported pigs are immediately marked for slaughter (do not contribute to population size or prevalence)
    let y random-float 1
    ifelse y < light-to-heavy-prop                                                                 ;Infected imported pigs are either heavily or lightly infected
      [set light-infected true]
      [set heavy-infected true]]]
end

to slaughter-pigs                                                                                 ;Pigs are slaughtered and eithe sold, shared, or consumed (process separate for heavy and light infected pigs, but not differentiated (yet) by parameters
  if light-infected = true and marked-for-slaughter = true [
    let x random-float 1
    hatch-carnes 10 [                                                                             ;When slaughtered, pig is divided into 10 pieces, but this is artibrary
      set light-infected true
      set hh false
      ifelse x < hh-only-pork                                                                     ;"x" is a random value generated for the whole pig to determine if the pig is consumed entirely by pig-owner households
         [set hh true]
        [ifelse x > (1 - sold-pork)                                                               ;"x" is a random value generated for the whole pig to determine if the pig is sold within the village (not consumed at home)
          [set hh false]
          [let y random-float 1
          if y < shared-pork-hh [                                                                 ;"y" is a random value generated by each "carne" among pigs that are "shared" to determine if each piece is eaten at home or sold
            set hh true]]]]]
  if heavy-infected = true and marked-for-slaughter = true [                                      ;SAME as above for light-infected
    let x random-float 1
    hatch-carnes 10 [
      set heavy-infected true
      set hh false
      ifelse x < hh-only-pork
        [ set hh true]
        [ifelse x > (1 - sold-pork)
          [set hh false]
          [let y random-float 1
           if y < shared-pork-hh [
             set hh true]]]]]
end

to distribute-pork                                                                               ;For pork that is not eaten at home, it is moved to a random household within the village
    if hh = false [
      move-to one-of patches with [household = true]]
  if light-infected = true [                                                                     ;Count the number of light-infected pork meals consumed by each household this week
    set light-infected-pork-meal (count carnes-here with [light-infected = true])]
  if heavy-infected = true [                                                                     ;Count the number of heavy-infected pork meals consumed by each household this week
    set heavy-infected-pork-meal (count carnes-here with [heavy-infected = true])]
end

to eat-pork                                                                                      ;Each household eats infected pork
  if [light-infected-pork-meal] of patch-here > 0 [                                              ;If light-infected pork is consumed, ALL members of the household are exposed to a probability of tapeworm infection from each infected pork meal
    repeat (light-infected-pork-meal)
      [let x random-float 1
        if x < pl2h [infect-humans]]]                                                            ;pl2h is the TUNING parameter determining probability of infection given exposure
  if [heavy-infected-pork-meal] of patch-here > 0 [                                              ;If heavy-infected pork is consumed, ALL members of the household are exposed to a probability of tapeworm infection from each infected pork meal
    repeat (heavy-infected-pork-meal)
      [let x random-float 1
          if x < ph2h [infect-humans]]]                                                          ;ph2h is the TUNING parameter determining probability of infection given exposure
end

to infect-humans                                                                                ;Human infection script
  ;if tapeworm = false [set count-tn count-tn + 1]                                              ;Counts incident taeniasis infections (for INCIDENCE calculation - SENSITIVITY ANALYSIS)
  set tapeworm true
  let x random-normal tn-lifespan-mean tn-lifespan-sd                                           ;Assign duration of taweworm infection (0-truncated)
    ifelse x < 0
      [set infection-duration 0]
      [set infection-duration round x]
  set time-since-infection 0
 end

to infect-travelers                                                                            ;Infect humans that are traveling
  if traveling = true [
    if random-float 1 < travel-incidence [                                                     ;Each traveler is exposued to possible infection based on endemic incidence rate
      ;if tapeworm = false [set count-tn count-tn + 1]                                         ;Counts incident taeniasis infection (for INCIDENCE calculation - SENSITIVITY ANALYSIS)
      set tapeworm true
      let x random-normal tn-lifespan-mean tn-lifespan-sd                                      ;Set lifespan of taweworm infection (0-truncated)
        ifelse x < 0
          [set infection-duration 0]
          [set infection-duration round x]
      set time-since-infection 0
      ]]
end

to slaughter                                                                                  ;Kill off pig that were slaughtered and already consumed
  if marked-for-slaughter = true [die]
end

to eat-eggs                                                                                                  ;Expose pigs to environmental contamination
  if (corraled = "never") or (corraled = "sometimes" and random-float 1 > prop-corral-sometimes) [           ;Pigs that are "never" corraled may be exposed, and pigs that are "sometimes" corraled have a given probability of being free-roaming each week
    let egg-exp (count eggs in-radius (home-range / scale))                                                  ;Count number of eggs/proglottids in each pigs home-range
    ;ask patches in-radius (home-range / scale) [set pcolor red]                                             ;OPTIONAL (slows computation): display home-range areas for each pig
    if egg-exp > 0 [                                                                                         ;If a pig is exposed to contamination...
      if (count eggs with [ova = true] in-radius (home-range / scale)) > 0 [                                 ;If contamination is light (ova), the pig is lightly exposed
        set light-exp true]
      if (count eggs with [proglottid = true] in-radius (home-range / scale)) > 0 [                          ;If contamination is heavy (proglottid), the pig is heavy exposed
        set heavy-exp true]]
    let factor (count humans with [tapeworm = true])                                                         ;Count the number of human tapeworm carriers in the village
    if random-float 1 < (1 - (1 - light-all) ^ factor) [                                                     ;Expose all free-roaming pigs to a probability of light exposure based on number of human carriers in village (irrespective of location)
      set light-exp true]
    if random-float 1 < (1 - (1 - heavy-all) ^ factor) [                                                     ;Expose all free-roaming pigs to a probability of heavy exposure based on number of human carriers in village (irrespective of location)
      set heavy-exp true]]
end

to seroconvert                                                                              ;Pigs that are exposed may develop serological responses (EITB)
  if light-exp = true and seropositive = false [                                            ;The probability of seroconversion after light exposure is based on the "light-sero" tuning parameter
    if random-float 1 < light-sero [
      set seropositive true
      set seroconversions seroconversions + 1]]                                            ;Count new seroconversions for calculation of sero-incidence
 if heavy-exp = true and seropositive = false [                                             ;The probability of seroconversion after heavy exposure is based on the "heavy-sero" tuning parameter
    if random-float 1 < heavy-sero [
      set seropositive true
      set seroconversions seroconversions + 1]]                                            ;Count new seroconversions for calculation of sero-incidence
end

to infect-pigs                                                                              ;If pigs are exposed, they may develop cyst infection
  if protected = false [                                                                    ;PROTECTION was added for INTERVENTIONS and may be caused by vaccination or treatment
    if heavy-exp = true [
      if susceptible = true [
        if random-float 1 < heavy-inf [                                                    ;Probability of heavy infection after heavy exposure is determined by the "heavy-inf" tuning parameter
          ;set count-heavy count-heavy + 1                                                 ;Count new heavy infection (for INCIDENCE calculuations - SENSITIVITY ANALYSIS)
          heavy-infect]]
      if light-infected = true [
        if random-float 1 < heavy-inf [                                                    ;Heavy infection can occur after heavy exposure if already lightly infected (here) or still susceptible (above)
          heavy-infect]]]
    if light-exp = true [
      if susceptible = true [                                                              ;Light infection can only occur if currently uninfected and lightly exposed
        if random-float 1 < light-inf [
          ;set count-light count-light + 1                                                 ;Count new light infections (for INCIDENCE calcuations - SENSITIVITY ANALYSIS)
           light-infect]]]]
end

to heavy-infect
  set heavy-infected true
  set light-infected false                                                                ;To prevent double-counting, pigs can only be heavy OR light infected
  set susceptible false
end

to light-infect
  set light-infected true
  set susceptible false
end

to mature-infection                                                                       ;Initiate environmental contamination on tapeworm infections that reach maturity
    if tapeworm = true [
      if time-since-infection = tn-incubation [                                           ;Initiate contamination when the infection duration reaches the incubation time
        set mature-tn true
        ;set color blue set size 20                                                       ;OPTIONAL: turn mature tapeworm infections blue
        if latrine-user = false and not traveling [                                       ;Only release environmental contamination into environment for non-latrine users and people NOT traveling
          set tn-released true
          hatch-eggs 1 [                                                                  ;Release 1 "egg" per tapeworm carrier (this "egg" represent environmental contamination generally)
            set id [who] of myself                                                        ;Tag egg with the index number of parent human (for mirroring)
            set proglottid true                                                           ;All eggs begins as both light (ova) and heavy (proglottid) contamination, until tapeworm dies and only light contamination remains
            set ova true
            set traveling-human false
            set infection-duration infection-duration                                     ;Assign same duration for egg as parent tapeworm (mirror)
            set time-since-infection tn-incubation                                        ;Assign time-since infection as the current time (incubation window has passed)
            set cont-radius-egg cont-radius-hh
            move-to one-of patches in-radius (cont-radius-egg / scale)                    ;Place egg in a random location within the pre-determined contamination radius
            set size 10
            set shape "circle"
            set color green]]]]
end

to tapeworm-die                                                                          ;Tapeworm infection reaches its predetermined time of death
  if tapeworm = true [
    if time-since-infection = infection-duration [
      set tapeworm false
      set time-since-infection 0
      set infection-duration 0
      set mature-tn false
      set tn-released false
      ;set color yellow set size 10
]]
end

to environmental-decay                                                                  ;For environmental contamination after tapeworm death, there is a constant probability of decay each week
  if traveling-human = false [
    if ova = true and proglottid = false and (random-float 1 < decay-mean) [
      die]]
end

to stop-contaminating                                                                   ;When tapeworm infections clear, the environmental contamination goes from heavy (proglottid) to light (ova) and is sucsceptible to probablistic decay
  if time-since-infection = infection-duration [
    set proglottid false
    set color brown]
end

to clear-pork-meals                                                                     ;At the end of each week, all pork meals are cleared and the cycle begins again
    set light-infected-pork-meal 0
    set heavy-infected-pork-meal 0
end

to assign-color-pigs                                                                    ;OPTIONAL - change color of infected pigs
  if heavy-infected = true [set color red]
  if light-infected = true [set color red]
  if not heavy-infected and not light-infected [set color pink]
end

to advance-protection                                                                  ;for INTERVENTIONS - for protection due to treatment with OFZ, protection lasts a set amount of time (protected-time) and then fades (for vaccination, protection is life-long)
  if protected = true and vaccinated = false [
      set protected-time protected-time - 1
      if protected-time = 0 [
        set protected false]]
end

;to recorder                                                                            ;for CALIBRATION - records the prevalence of TN, light and heavy pig infections, seroprevalence, seroincidence at set time-points
;  if ticks = 500 [
;    set tn500 (count humans with [tapeworm = true] / count humans)
;    set light500 (count pigs with [light-infected = true] / count pigs)
;    set heavy500 (count pigs with [heavy-infected = true] / count pigs)
;    set seroprev500 (count pigs with [seropositive = true] / count pigs)
;    set sero500 seroincidence]
;  if ticks = 600 [
;    set tn600 (count humans with [tapeworm = true] / count humans)
;    set light600 (count pigs with [light-infected = true] / count pigs)
;    set seroprev600 (count pigs with [seropositive = true] / count pigs)
;    set heavy600 (count pigs with [heavy-infected = true] / count pigs)
;    set sero600 seroincidence]
; if ticks = 700 [
;    set tn700 (count humans with [tapeworm = true] / count humans)
;    set light700 (count pigs with [light-infected = true] / count pigs)
;    set heavy700 (count pigs with [heavy-infected = true] / count pigs)
;    set seroprev700 (count pigs with [seropositive = true] / count pigs)
;    set sero700 seroincidence]
; if ticks = 800 [
;    set tn800 (count humans with [tapeworm = true] / count humans)
;    set light800 (count pigs with [light-infected = true] / count pigs)
;    set heavy800 (count pigs with [heavy-infected = true] / count pigs)
;    set seroprev800 (count pigs with [seropositive = true] / count pigs)
;    set sero800 seroincidence]
; if ticks = 900 [
;    set tn900 (count humans with [tapeworm = true] / count humans)
;    set light900 (count pigs with [light-infected = true] / count pigs)
;    set heavy900 (count pigs with [heavy-infected = true] / count pigs)
;    set seroprev900 (count pigs with [seropositive = true] / count pigs)
;    set sero900 seroincidence]
;  if ticks = 1000 [
;    set tn1000 (count humans with [tapeworm = true] / count humans)
;    set light1000 (count pigs with [light-infected = true] / count pigs)
;    set heavy1000 (count pigs with [heavy-infected = true] / count pigs)
;    set seroprev1000 (count pigs with [seropositive = true] / count pigs)
;    set sero1000 seroincidence]
;end

to track-incidence                                                                    ;for calcuation of pig sero-incidence (need to activate other sero-incidence code chunks above)
  if week-counter = 0 [                                                               ;At beginning of interval, calculate base population and set seroconversions to 0
    set seroeligible count pigs with [seropositive = false]
    set seroconversions 0]
  set week-counter week-counter + 1                                                   ;Count up to a pre-determined interval of weeks (here we used a 4-month incidence window (17 weeks)(
  if week-counter = 17 [                                                              ;At the end of the window, calculate the incidence by dividing total seroconversions by total eligible population
    set seroincidence (seroconversions / seroeligible)
    set week-counter 0]
end



;;;;;;;;INTERVENTIONS;;;;;;;;;

;;;Combined Functions;;;

;;RING STRATEGY;;

to human-ring-screening                            ;RING SCREENING (HUMAN ONLY): Sequence of human ring screening (tongue-inspection, human screening, human treatment)
  reset-variables
  tongue-inspect-h
  repeat 1 [go]
  ring-screen-h
  repeat 4 [go]
  ring-screen-treat-h
  repeat ((Interval * 5) - 5) [go]
end

to combined-ring-screening                         ;RING SCREENING (HUMAN AND PIG): Sequence of combined ring screening (tongue-inspection, pig treatment, human screening, human treatment)
  reset-variables
  tongue-inspect-p
  repeat 1 [go]
  ring-screen-h
  ring-treat-p
  repeat 4 [go]
  ring-screen-treat-h
  repeat ((Interval * 5) - 5) [go]
end

to human-ring-treatment                           ;RING TREATMENT (HUMAN ONLY): Sequence of human ring treatment (tongue-inspection, human treatment)
  reset-variables
  tongue-inspect-h
  repeat 1 [go]
  ring-treat-h
  repeat ((Interval * 5) - 1) [go]
end

to combined-ring-treatment                        ;RING TREATMENT (HUMAN AND PIG): Sequence of combined ring treatment (tongue-inspection, pig treatment, human treatment)
  reset-variables
  tongue-inspect-h
  repeat 1 [go]
  ring-treat-h
  ring-treat-p
  repeat ((Interval * 5) - 1) [go]
end


;;MASS APPLIED INTERVENTIONS;;

to combined-mass-treatment                        ;MASS TREATMENT (HUMAN AND PIG)
  reset-variables
  mass-treat-h
  mass-treat-p
  repeat (Interval * 5) [go]
end

to human-mass-treatment                          ;MASS TREATMENT (HUMAN ONLY)
  reset-variables
  mass-treat-h
 repeat (Interval * 5) [go]
end

to combined-mass-screening                       ;MASS TREATMENT (HUMAN AND PIG)
  reset-variables
  mass-screen-h
  mass-treat-p
  repeat 4 [go]
  mass-screen-treat-h
  repeat (Interval * 5) [go]
end

to human-mass-screening                         ;MASS TREATMENT (HUMAN ONLY)
  reset-variables
  mass-screen-h
  repeat 4 [go]
  mass-screen-treat-h
  repeat (Interval * 5) [go]
end

to pig-vaccinate                                ;MASS PIG VACCINATION: Applied two rounds of pig vaccine two weeks apart
  reset-variables
  vaccinate
  repeat 2 [go]
  vaccinate
  repeat ((Interval * 5) - 2) [go]
end

to pig-vaccinate-treat
  reset-variables
  vaccinate-treat
  repeat 2 [go]
  vaccinate
  repeat ((Interval * 5) - 2) [go]
end


to pig-replacement                              ;MASS PIG REPLACEMENT
  reset-variables
  saca
  repeat (Interval * 5) [go]
end

;;;Generic Functions;;;

;;;RING STRATEGY;;;

to reset-variables                                                                 ;Reset all visualizations
  ask humans [
    set eligible false
    set screen-pos false
    set color yellow
    set size 10]
  ask pigs [
    set eligible false
    set size 10
    set color pink]
  ask patches [set pcolor black]
end

to tongue-inspect-h                                                                 ;HUMAN-ONLY RING-STRATEGY: Screening pigs for heavy cyst infection using tongue inspection, and mark HUMANS ONLY as eligible for intervention if within ring
   ask pigs [
     if age >= age-eligible [                                                       ;Only tongue-inspect age-eligible pigs
       if random-float 1 < tongue-part [                                            ;Tongue inspection applied to a proportion of age-eligible pigs that participate ("tongue-part")
         if not heavy-infected and random-float 1 < tongue-fp [                     ;Uninfected pigs may be detected as tongue-positive based on a probability of false positivity ("tongue-fp")
           ask patches in-radius (ring-size / scale) [                              ;OPTIONAL - visualize the intervention rings with BLUE
             set pcolor blue]
           ask humans in-radius (ring-size / scale) [                               ;Humans currently present inside a ring are marked as eligible for intervention
             set eligible true]
           set marked-for-slaughter true]                                           ;Pigs that are tongue-positive are marked for slaughter to be replaced
       if heavy-infected and random-float 1 < tongue-sens [                         ;Heavily infected pigs are detected as tongue-positive based on the sensitivity of tongue-inspection for detecting heavy cyst infection ("tongue-sens")
         ask patches in-radius (ring-size / scale) [                                ;OPTIONAL - visualize intervention rings with BLUE
           set pcolor blue]
         ask humans in-radius (ring-size / scale) [                                 ;Humans currently present inside a ring are marked as eligible for intervention
           set eligible true]
         set marked-for-slaughter true]]                                            ;Pigs that are tongue-positive are marked for slaughter to be replaced
       birth-pigs                                                                   ;Replace tongue-positive pigs with new uninfected pigs
       slaughter]]                                                                  ;Slaughter (kill) tongue-positive pigs without consuming
    ;assign-color-pigs]                                                             ;OPTIONAL - refresh color for pigs
  ;ask humans with [eligible = true][set size 20]                                   ;OPTIONAL - mark humans that are eligible for interventions
end

to ring-screen-h                                                                   ;HUMAN RING SCREENING: Humans that are marked as eligible based on rings are screened for taeniasis
  ask humans [
    ;set size 10
    if eligible = true [
      if random-float 1 < screen-part [                                            ;Apply screening to eligible humans that participate in stool screening ("screen-part")
        if tapeworm = true [
         if random-float 1 < elisa-sens [                                          ;A proportion of screened humans that have taeniasis will actually test positive based on sensitivity of the ELISA assay ("elisa-sens")
            set screen-pos true                                                    ;Mark positively screened as "screen-pos" and eligible for treatment (false positives are not treated because it would be inconsequential)
          ;set color red set size 20                                               ;OPTIONAL - mark positive humans
   ]]]]
  set eligible false]                                                              ;Reset eligibility to NULL
end

to ring-screen-treat-h                                                             ;HUMAN RING SCREENING: Humans that screen positive for taeniasis are treated
  ask humans [
    if screen-pos = true and random-float 1 < screen-trt-part [                    ;A proportion of positively screened individuals will agree to participate in treatment based on ("screen-trt-part")
        if random-float 1 < screen-trt-eff [                                       ;A proportion of treated individuals will be cured (post-screening treatment assumed a specific efficacy for follow-to-cure therapy, "screen-trt-eff")
          set tapeworm false
          set time-since-infection 0
          set infection-duration 0
          set mature-tn false
          set tn-released false
          ;set color orange
          ask eggs with [id = [who] of myself][                                    ;When treatment is successful, heavy environmental contamination ceases, leaving only light contamination (ova) in the environment
            set proglottid false
            set color brown]]]
  set screen-pos false]                                                            ;Reset screening status to NULL
end

to ring-treat-h                                                                     ;HUMAN RING TREATMENT: Humans that are marked as eligible based on rings are treated presumptively for taeniasis
  ask humans [
    ;set size 10
    if eligible = true [
      let x random-float 1
        if x < treat1-part [                                                        ;A proportion of eligible humans receive only a SINGLE DOSE of treatment ("treat1-part")
          if tapeworm = true [
            if random-float 1 < treat1-eff [                                        ;For people with taeniasis that are treated with one dose, SINGLE-DOSE treatment efficacy is defined by "treat1-eff"
            ;set size 20 set color red                                              ;OPTIONAL - change color of cured humans
            set tapeworm false
            set time-since-infection 0
            set infection-duration 0
            set mature-tn false
            set tn-released false
            ask eggs with [id = [who] of myself][                                   ;When treatment is successful, heavy environmental contamination ceases, leaving only light contamination (ova) in the environment
              set proglottid false
              set color brown]]]]
      if x > (1 - treat2-part) [                                                    ;A proportion of eligible humans receive TWO DOSES of treatment ("treat2-part")
        if tapeworm = true [
          if random-float 1 < treat2-eff [                                          ;For people with taeniasis that are treated with two doses, TWO DOSE treatment efficacy is defined by "treat2-eff"
            ;set size 20 set color orange                                           ;OPTIONAL - change color of cured humans
            set tapeworm false
            set time-since-infection 0
            set infection-duration 0
            set mature-tn false
            set tn-released false
            ask eggs with [id = [who] of myself][                                   ;When treatment is successful, heavy environmental contamination ceases, leaving only light contamination (ova) in the environment
              set proglottid false
              set color brown]]]]]
  set eligible false]                                                               ;Reset eligible status to NULL
end

to tongue-inspect-p                                                                 ;COMBINED HUMAN AND PIG RING STRATEGY: Screening pigs for heavy cyst infection using tongue inspection, and mark HUMANS AND PIGS as eligible for intervention if within ring
   ask pigs [
     if age >= age-eligible [                                                       ;Only tongue-inspect age-eligible pigs
       if random-float 1 < tongue-part [                                            ;Tongue inspection applied to a proportion of age-eligible pigs that participate ("tongue-part")
         if not heavy-infected and random-float 1 < tongue-fp [                     ;Uninfected pigs may be detected as tongue-positive based on a probability of false positivity ("tongue-fp")
           ask patches in-radius (ring-size / scale) [                              ;OPTIONAL - visualize the intervention rings with BLUE
             set pcolor blue]
           ask humans in-radius (ring-size / scale) [                               ;Humans currently present inside a ring are marked as eligible for intervention
             set eligible true]
           ask pigs in-radius (ring-size / scale) [                                 ;Pigs currently present inside a ring are marked as eligible for intervention
             set eligible true]
           set marked-for-slaughter true]                                           ;Pigs that are tongue-positive are marked for slaughter to be replaced
       if heavy-infected and random-float 1 < tongue-sens [                         ;Heavily infected pigs are detected as tongue-positive based on the sensitivity of tongue-inspection for detecting heavy cyst infection ("tongue-sens")
         ask patches in-radius (ring-size / scale) [                                ;OPTIONAL - visualize intervention rings with BLUE
           set pcolor blue]
         ask humans in-radius (ring-size / scale) [                                 ;Humans currently present inside a ring are marked as eligible for intervention
           set eligible true]
         ask pigs in-radius (ring-size / scale) [                                   ;Pigs currently present inside a ring are marked as eligible for intervention
           set eligible true]
         set marked-for-slaughter true]]                                            ;Pigs that are tongue-positive are marked for slaughter to be replaced
       birth-pigs                                                                   ;Replace tongue-positive pigs with new uninfected pigs
       slaughter]]                                                                  ;Slaughter (kill) tongue-positive pigs without consuming
    ;assign-color-pigs]                                                             ;OPTIONAL - refresh color for pigs
  ;ask humans with [eligible = true][set size 20]                                   ;OPTIONAL - mark humans that are eligible for interventions
  ;ask pigs with [eligible = true][set size 40]                                     ;OPTIONAL - mark pigs that are eligible for interventions
end

to ring-treat-p                                                                     ;TREATMENT OF PIGS FOR RING STRATEGY: Pigs marked as eligible for ring treatment based on presence within ring are treated
  ask pigs [
    if age >= age-eligible [                                                        ;Only age-eligible pigs are treated
      if eligible = true and random-float 1 < treat-part-p [                        ;A proportion of pigs that are age-eligible participate in the treatment ("treat-part-p")
        if heavy-infected = true or light-infected = true [                         ;OFZ is expected to be ~100% efficacious against cyst infection, thus no efficacy variables is applied, all treated pigs assumed to be cured
          ;set color brown set size 40                                              ;OPTIONAL - change color of treated pigs
          set heavy-infected false
          set light-infected false
          set susceptible true
          set protected true                                                        ;Pigs that were infected and treated with OFZ confer protection against future infections that persists a set amount of time ("oxf-protection")
          set protected-time oxf-protection]]]
  set eligible false]                                                               ;Reset eligible status to NULL
end

to mass-treat-final                                                                 ;FINAL-ROUND MASS TREATMENT OF HUMANS: At conclusion of trial, all humans are screened with follow-up treatment offered to all
  ask humans [
    if not traveling [                                                              ;Only individuals NOT traveling are eligible to participate
      if random-float 1 < treat-final-part [                                        ;Among eligible individuals, a proportion will participate based on "treat-final-part"
        if tapeworm = true [
          if random-float 1 < screen-trt-eff [                                      ;A proportion of treated individuals will be cured (post-screening treatment assumed a specific efficacy for follow-to-cure therapy, "screen-trt-eff")
            ;set size 20 set color red                                              ;OPTIONAL - mark humans that are cured
            set tapeworm false
            set time-since-infection 0
            set infection-duration 0
            set mature-tn false
            set tn-released false
            ask eggs with [id = [who] of myself][                                  ;When treatment is successful, heavy environmental contamination ceases, leaving only light contamination (ova) in the environment
              set proglottid false
              set color brown]]]]]]
end


;;MASS APPLIED INTERVENTIONS;;

to mass-treat-h                                                                    ;MASS TREATMENT for humans to treat taeniasis
  ask humans [
    if not traveling [                                                             ;Only humans not currently traveling are eligible to participate
      if random-float 1 < treat-mass-part [                                        ;Among eligible individuals, a proportion will participate based on "treat-mass-part"
        if tapeworm = true [
          if random-float 1 < treat1-eff [                                         ;All participating individuals recieve a SINGLE-DOSE of treatment with an efficacy of "treat1-eff"
            set tapeworm false
            set time-since-infection 0
            set infection-duration 0
            set mature-tn false
            set tn-released false
            ask eggs with [id = [who] of myself][                                 ;When treatment is successful, heavy environmental contamination ceases, leaving only light contamination (ova) in the environment
              set proglottid false
              set color brown]]]]]]
end

to mass-treat-p                                                                   ;MASS TREATMENT for pigs to treat cyst infection
  ask pigs [
    if age >= age-eligible [                                                      ;Only age-eligible pigs may receive treatment
      if random-float 1 < treat-part-p [                                          ;Among eligible pigs, a proportion will participate based on "treat-part-p"
        if heavy-infected = true or light-infected = true [
          set heavy-infected false
          set light-infected false
          set susceptible true
          set protected true                                                      ;Pigs that were infected and treated with OFZ confer protection against future infections that persists a set amount of time ("oxf-protection")
          set protected-time oxf-protection]]]]
end

to mass-screen-h                                                                  ;MASS SCREENING of humans for taeniasis
  ask humans [
    if not traveling [                                                            ;Only humans not currently traveling are eligible to participate
      if random-float 1 < screen-part [                                           ;Among eligible individuals, a proportion will participate based on "screen-part"
        if tapeworm = true [
          if random-float 1 < treat1-eff [                                        ;Despite being a screening program, all eligible individuals received a SINGLE dose of treatment with an efficacy of "treat1-eff"
            set tapeworm false
            set time-since-infection 0
            set infection-duration 0
            set mature-tn false
            set tn-released false
            ask eggs with [id = [who] of myself][                                 ;When treatment is successful, heavy environmental contamination ceases, leaving only light contamination (ova) in the environment
              set proglottid false
              set color brown]]
         if random-float 1 < elisa-sens [                                         ;A proportion of screened humans that have taeniasis will actually test positive based on sensitivity of the ELISA assay ("elisa-sens")
            set screen-pos true]]]]]
end

to mass-screen-treat-h                                                            ;MASS SCREENING: Humans that are screened positively for taeniasis receive follow-up treatment
  ask humans [
    if screen-pos = true and random-float 1 < screen-trt-part [                   ;A proportion of positively screened individuals will agree to participate in treatment based on ("screen-trt-part")
      if random-float 1 < screen-trt-eff [                                        ;A proportion of treated individuals will be cured (post-screening treatment assumed a specific efficacy for follow-to-cure therapy, "screen-trt-eff")
        set tapeworm false
        set time-since-infection 0
        set infection-duration 0
        set mature-tn false
        set tn-released false
        ask eggs with [id = [who] of myself][                                     ;When treatment is successful, heavy environmental contamination ceases, leaving only light contamination (ova) in the environment
          set proglottid false
          set color brown]]]
    set screen-pos false]
end

to vaccinate                                                                      ;PIG VACCINATION: Pigs receive two independent doses of vaccine that may confer protection against future infection
  ask pigs [
    if age >= age-eligible [                                                      ;Only age-eligible pigs receive vaccine
      if random-float 1 < vacc-part [                                             ;A proportion of eligible pigs participate in vaccination
        set vacc-dose vacc-dose + 1]]                                             ;Counts the number of doses each pig has received
    if vacc-dose = 2 [                                                            ;If a pig receives a second dose...
        ;set vacc-dose 0
        if random-float 1 < vacc-eff [                                            ;Protection/immunity is acheived based on the vaccine efficacy ("vacc-eff")
          set vaccinated true                                                     ;Pig is marked as vaccinated, and its protection will not wane over time
          set protected-time 0
          set protected true]]]                                                   ;Pig is protected from future infections
end

to vaccinate-treat                                                                ;PIG VACCINATION AND TREATMENT: Pigs receive two independent doses of vaccine that may confer protection against future infection
  ask pigs [
    if age >= age-eligible [                                                      ;Only age-eligible pigs receive vaccine
      if random-float 1 < vacc-part [                                             ;A proportion of eligible pigs participate in vaccination
        set vacc-dose vacc-dose + 1                                             ;Counts the number of doses each pig has received
          if heavy-infected = true or light-infected = true [
            set heavy-infected false
            set light-infected false
            set susceptible true
            set protected true                                                    ;Pigs that were infected and treated with OFZ confer protection against future infections that persists a set amount of time ("oxf-protection")
            set protected-time oxf-protection]
    if vacc-dose = 2 [                                                            ;If a pig receives a second dose...
        ;set vacc-dose 0
        if random-float 1 < vacc-eff [                                            ;Protection/immunity is acheived based on the vaccine efficacy ("vacc-eff")
          set vaccinated true                                                     ;Pig is marked as vaccinated, and its protection will not wane over time
          set protected-time 0
            set protected true]]]]]                                               ;Pig is protected from future infections
end
to saca                                                                           ;MASS SALE AND PIG NECROSPY: A proportion of pigs are purchased and slaughtered
  ask pigs [
    if age >= age-eligible [                                                      ;Only age-eligible pigs are included
      if seropositive = true [                                                    ;In this SACA, only seropositive pigs were eligible to be purchased
        if random-float 1 < saca-part [                                           ;A proportion of eligible seropositive pigs participated in the saca
          set marked-for-slaughter true]]]                                        ;Participating pigs are marked for slaughter, replaced with new pigs, and then killed
      birth-pigs
      slaughter]
end
@#$#@#$#@
GRAPHICS-WINDOW
271
181
688
644
-1
-1
1.133
1
10
1
1
1
0
0
0
1
-180
180
-200
200
0
0
1
ticks
30.0

BUTTON
271
91
391
144
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
503
10
616
43
pl2h
pl2h
0
0.01
0.0074
0.0001
1
NIL
HORIZONTAL

SLIDER
502
45
614
78
ph2h
ph2h
0
.01
0.0059
0.0001
1
NIL
HORIZONTAL

BUTTON
272
146
433
179
NIL
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
434
146
615
179
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
393
45
501
78
heavy-inf
heavy-inf
0
0.01
0.0046
0.0001
1
NIL
HORIZONTAL

PLOT
3
10
268
172
Humans
Weeks
Prevalence
0.0
10.0
0.0
0.1
true
false
"" ""
PENS
"Taeniasis" 1.0 0 -13345367 true "" "plotxy ticks (count humans with [tapeworm = true and not traveling]) / (count humans)"

PLOT
3
175
270
352
Pigs
Weeks
Prevalence
0.0
10.0
0.0
0.5
true
false
"" ""
PENS
"default" 1.0 0 -2674135 true "" "plotxy ticks (count pigs with [heavy-infected = true] / count pigs)"
"pen-1" 1.0 0 -817084 true "" "plotxy ticks (count pigs with [light-infected = true and not heavy-infected] / count pigs)"

SLIDER
393
10
501
43
light-inf
light-inf
0
0.1
0.0163
0.001
1
NIL
HORIZONTAL

SLIDER
393
79
502
112
light-all
light-all
0
0.1
0.0141
0.001
1
NIL
HORIZONTAL

SLIDER
392
112
502
145
heavy-all
heavy-all
0
0.1
0.0182
0.001
1
NIL
HORIZONTAL

SLIDER
503
79
615
112
light-sero
light-sero
0
1
0.15
0.01
1
NIL
HORIZONTAL

SLIDER
503
112
615
145
heavy-sero
heavy-sero
0
1
0.1
0.01
1
NIL
HORIZONTAL

MONITOR
110
705
180
750
NIL
count pigs
17
1
11

MONITOR
6
706
109
751
NIL
num-households
17
1
11

MONITOR
185
705
254
750
NIL
count humans
17
1
11

PLOT
5
354
267
535
Serology
Weeks
Seroprevalence
0.0
10.0
0.0
1.0
true
false
"" ""
PENS
"default" 1.0 0 -13840069 true "" "plotxy ticks (count pigs with [seropositive = true] / count pigs)"

PLOT
5
536
267
704
Seroincidence (4-month)
Weeks
Seroincidence
0.0
10.0
0.0
1.0
true
false
"" ""
PENS
"default" 1.0 0 -10899396 true "" "plotxy ticks seroincidence"

SLIDER
636
10
766
43
Interval
Interval
2
24
12.0
2
1
Months
HORIZONTAL

BUTTON
636
44
766
77
NIL
human-ring-screening
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
768
45
903
78
NIL
combined-ring-screening
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
636
78
766
111
NIL
human-ring-treatment
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
768
79
904
112
NIL
combined-ring-treatment
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
635
111
769
144
NIL
human-mass-treatment
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
769
112
904
145
NIL
combined-mass-treatment
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
905
46
1037
79
NIL
human-mass-screening
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
905
80
1037
113
NIL
combined-mass-screening
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
768
10
904
43
NIL
pig-vaccinate
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
907
10
1037
43
NIL
pig-replacement
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

CHOOSER
271
10
392
55
Households
Households
"XY-coordinates" "Random"
0

PLOT
779
149
1033
332
Vaccination
Weeks
Vaccinated
0.0
1.0
0.0
1.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plotxy ticks (count pigs with [vaccinated = true] / count pigs)"

BUTTON
906
113
1035
146
NIL
pig-vaccinate-treat
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SWITCH
269
56
392
89
Calibrated?
Calibrated?
0
1
-1000

@#$#@#$#@
## WHAT IS IT?

(a general understanding of what the model is trying to show or explain)

## HOW IT WORKS

(what rules the agents use to create the overall behavior of the model)

## HOW TO USE IT

(how to use the model, including a description of each of the items in the Interface tab)

## THINGS TO NOTICE

(suggested things for the user to notice while running the model)

## THINGS TO TRY

(suggested things for the user to try to do (move sliders, switches, etc.) with the model)

## EXTENDING THE MODEL

(suggested things to add or change in the Code tab to make the model more complicated, detailed, accurate, etc.)

## NETLOGO FEATURES

(interesting or unusual features of NetLogo that the model uses, particularly in the Code tab; or where workarounds were needed for missing features)

## RELATED MODELS

(models in the NetLogo Models Library and elsewhere which are of related interest)

## CREDITS AND REFERENCES

(a reference to the model's URL on the web if it has one, as well as any other necessary credits, citations, and links)
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

pig
false
0
Polygon -7500403 true true 45 150 60 135 75 120 90 120 120 120 150 120 165 120 195 135 210 120 210 105 225 105 225 120 225 120 240 120 240 105 255 105 255 120 255 135 270 150 270 165 255 180 240 180 210 180 195 180 165 195 165 210 165 225 150 225 150 195 60 195 60 210 60 225 45 225 45 195 45 165 30 180 45 150

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.1.1
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
