;; ASE6102 System of Systems
;; Swarm Robotics - Demonstration Model
;; Based on system described in Jamshidi, Ch 14, 1st ed.
;; Team6


;;-----------------------------------
;; declarations
;;-----------------------------------

__includes ;; Moved to end of code since includes do not work with commons
[ ;;"messaging.nls"     ; handles agent messaging
  ;;"sxl-utils.nls"     ; small set of NetLogo utilities
]

globals
[ mine-active-color
  mine-deactive-color
  target_X                       ; xcoord of target mine
  target_Y                       ; ycoord of target mine
  swarmTeamSize                  ; number of robots in a swarm team
  disarm_Count                   ; count for number of disarming swarmRobots
  target_mine                    ; name of mine to help with
  leader                         ; lead swarmRobot (he who(m?) found the mine)
  closestSwarmMembers
]

frags-own [
  dim             ; used for fading particles
]

;directed-link-breed [red-links red-link]

;;-----------------------------------
;; breed data
;;-----------------------------------
breed [sensors sensor]
sensors-own
[ status          ;; off/monitoring/detecting
  name
]

breed [swarmRobots swarmRobot]
swarmRobots-own
[ status          ;; foraging/following_scent/waiting/homing/disarming
  target          ;; target mine
  leadSwarmBot    ;; lead swarmRobot
  start-time
  time-taken
  name
]

breed [baseRobots baseRobot]
baseRobots-own
[ status          ;; on/tx
  name
]

breed [mines mine]
mines-own
[ status          ;; hidden/found/disarmed[die]
  start-time
  time-taken
  name
]

breed [dynamicTargets dyanmicTarget]
dynamicTargets-own
[status
 start-time
 time-taken
 name
]

breed [obstacles obstacle]
obstacles-own
[ ob-dia
  ob-color
]

breed [ frags frag ]

turtles-own [
  col             ; sets color of an explosion particle
  x-vel           ; x-velocity
  y-vel           ; y-velocity
  #msgQ          ;; the message queue
  #msg-id        ;; the agent name/id for messaging
]

;;-----------------------------------
;; general
;;-----------------------------------

to setup-turtles
  ask turtles[
    set col 0
    set color 2
  ]
end


to setup
  clear-all
  setup-globals
  setup-patches
  setup-obsticals
  setup-mines
  setup-swarmRobots
  setup-baseRobots
  setup-sensors
  setup-dynamicTargets
  reset-ticks
end

to setup-globals
  ;; set globals
  set mine-active-color red
  set mine-deactive-color green
  set swarmTeamSize swarm-teamSize - 1
end

to setup-patches
  ask patches [set pcolor gray]
end

to setup-mines
  create-mines no-of-mines
  ask mines
  [ set name (word "mine_" who)
    setxy random-xcor random-ycor
    ;setxy 0 10
    set color black
    set shape "ufo top"
    set status "hidden"
    ask mines
    [
    ifelse (any? obstacles-on patch-here)
      [
        wiggle
      ][
        fd 2
      ]
    ]
  ]
end

to setup-swarmRobots
  create-swarmRobots no-of-swarmRobots
  ask swarmRobots
  [ set name (word "swarmrobot " who)
    ; make-msgQ (word "swarm_" who)
    setxy 0 0
    set color blue
    set status "foraging"
  ]
end

to setup-baseRobots
  create-baseRobots no-of-baseRobots
  ask baseRobots
  [ set name (word "base_" who)
    make-msgQ (word "base_" who)
    setxy 0 0
    set color green
    set size 1.5
    ;;set shape "square"
    set status "ON"
  ]
end

to setup-sensors
  create-sensors 10
  [
    set shape "star"
    set size .5
    set color yellow
    let $n 0  ;; actual number of turtles on this patch
    let $s 0  ;; current number of turtles on a side of square pattern - 1
    set heading 0
    ask patch-here []
    repeat no-of-sensors   ;; number of needed turtles
    [ hatch 1 jump Grid   ;; make one turtle and move
      set $n $n + 1   ;; increment count of curent side
      ask patch-here []
      if $n > $s  ;; if side finished...
      [
        right 90 set $n 0   ;; turn and reset count
        ask patch-here []
        ;; if headed up or down, increment side count
        if heading mod 180 = 0 [ set $s $s + 1
          ]
      ]
    ]
   die
  ]
end

to setup-dynamicTargets
  create-dynamicTargets no-of-dynamicTargets
  ask dynamicTargets
  [
    setxy random-xcor 20
    set shape "person"
    set size 1
    set color black
  ]
end

to setup-obsticals
  create-obstacles no-of-obstacles
  ask obstacles
  [
    setxy random-xcor random-ycor
    set shape "square"
    set size random 5
    set col one-of base-colors
  ]
end

to go
  if ticks > Game_Time_Out [ setup go ]   ;; stop when ticks run high
  if not any? mines [ setup go ]        ;; stop when no mines
  if count swarmRobots < swarm-teamSize [ setup go ] ;; cant disarm mines anymore
  forage-SwarmRobot
  homming
  comms-baseRobots
  following-scent
  ;obs-avoidance
  disarming
  timeOut
  blowup
  ask frags
  [
        fade
  ]
  every reset_Time
  [
    setup
    go
  ]
  tick
end

to moveForage
  ask swarmRobots
  [
    if status = "foraging"
    [
      wiggle
      forward .1
      ; obs-avoidance
    ]
  ]
end

to forage-SwarmRobot
  ask swarmRobots
  [
    if status = "foraging"
    [
      ;; look for any mines using the robots sensor
      ifelse any? mines in-cone sensor-range sensor-FoV  ;; [in-cone range fov_angle]
      [
        ;; check to see of the mine is already found and being disarmed
        ask nearest-of mines
        [
          ifelse status != "found"
          [
            set status "found"
            set start-time ticks
            set color red
            ;; update the status of the robot that found the mine
            ask nearest-of swarmRobots
            [
              ;; update status of swarmRobot that has found a mine
              set status "waiting"
              set color yellow
              set target nearest-of mines
              ;;fd 1 ; move forward one... this should be calculated
              face nearest-of mines

              ;; send message to baseRobot with x,y coordinates
              broadcast baseRobots (list ([xcor] of nearest-of mines) ([ycor] of nearest-of mines) nearest-of swarmRobots who)
              ;;ask nearest-of swarmRobots [
                ;create-red-link-to nearest-of baseRobots
                ;show link-with nearest-of baseRobots
              ;]
            ]
          ]
          [
            ;; tell any other robots to move along if they are not waiting
            ;; if they see the mine
            ask swarmRobots
            [
              if status != "waiting" [moveForage]
            ]

          ]
        ]
      ]
      ;; else keep foraging for other mines
      [
        moveForage
      ]
    ]
  ]
end

to comms-baseRobots
  ;; base robots get told about mines to disarm by swarmRobot and sends helper swarmRobots
  ask baseRobots
    [ if msg-waiting? ;; check for a message [Message = swarmRobot#(mineX, mineY)
      [
        ;; added a queue hold for next bot if two mines are detected at the same time with
        ;; less than the required bots foraging to send to disarm
        ifelse count swarmRobots with [status = "foraging"] < swarm-teamSize - 1
        [
          set color yellow
          if swarmRobots with [status = "scent-following"] != nobody
          [
          set closestSwarmMembers up-to-n-of (swarm-teamSize - 1) swarmRobots with [status = "following_scent"]
          ;show closestSwarmMembers
          ask closestSwarmMembers ;up-to-n-of (swarm-teamSize - 1) swarmRobots with [status = "foraging"]
            [
              right 180
              fd 1
              set status "foraging"
              set color blue
            ]
          ]

        ;; do nothing set
        ]
        [
            set color black
            set status "tx"
            let m get-msg ;; fetch message - clears queue
            let from item 0 m ;; mineX
            let msg  item 1 m ;; mineY
            set target_X   item 0 msg ;; move value to var
            set target_Y   item 1 msg ;; move value to var
            set leader     item 2 msg ;; lead swarm robot name

            ;; set the target mine variable to the mine name
            set target_mine one-of mines with
                  [(xcor = [target_X] of myself) and (ycor = [target_Y] of myself)]

            ;; now ask some others of the swarm to go help with the disarming of the mine
            ;; right now this is random but a good upgrade would be to grab the closest avalible
          set closestSwarmMembers up-to-n-of (swarm-teamSize - 1) swarmRobots with [status = "foraging"]
          ;show closestSwarmMembers
          ask closestSwarmMembers ;up-to-n-of (swarm-teamSize - 1) swarmRobots with [status = "foraging"]
            [
              set status "following_scent"
              set start-time ticks
              set color orange
              set target target_mine
              set leadSwarmBot leader
              facexy target_X target_Y
            ]
          ]
        ]
      ]

end

to following-scent
  ask swarmRobots with [status = "following_scent"]
  [
    ifelse target != nobody
    [
      ;; either move to target or start disarming the mine
      ifelse distance target > 1
      [
        ifelse any? mines in-cone sensor-range sensor-FoV ;; [in-cone range fov_angle]
        [
          if mines != target
          [
            ifelse obs-detection = true
            [
              obs-avoidance
            ]
            [ ;; Simple obsticle avoidance
              right random 60
              left random 60
              fd .1
            ]

          ]
        ]
        [
          face target ;; important prevents zombie swarmbots
          fd .1
        ]
      ]
      [
        set status "disarming"
        set start-time ticks
        set color red
      ]
    ]
    [
      set status "foraging"
      set color blue
    ]
  ]

end

to homming
  ;; when a bot finds a mine its at the distance near its sensor range
  ;; need to walk the bot up to the mine and then set bot status to disarming
  ;; need timeout
  ask swarmRobots with [status = "waiting"]
  [
    if target != nobody
    [
      ;; either move to target or start disarming the mine
      ifelse distance target > 1
      [
        face target ;; important prevents zombie swarmbots
        fd .1
        ask baseRobots
        [
          set color green
          set status "ON"
        ]
      ]
      [
        ; start disarming the mine
        set status "disarming"
        set start-time ticks
        set color red
      ]
    ]

  ]
end

to disarming
  ;; check around the mine and make sure that there are the number of
  ;; required bots then set mine status to disarm and reset bots
  ;; and set bot status to foraging
  ask mines with [status = "found"]
  [
    ;; if the number of bots around the mine is corrent then return the bots to
    ;; foraging and destry the mine
    if (count swarmRobots in-radius 1) >= swarm-teamSize [
      ask swarmRobots in-radius 1 ;; check for bots in a radius around the mine
      [set status "foraging"
       set color blue
      ]
      set status "disarmed"
      ask mines with [status = "disarmed"] [die]
    ]
  ]
end

to timeOut ;; this is a super important part of the system of systems... no timeout = zombies
  ;; resets a swarmBot that does not get help to a foraging bot
  ;; and idetntifies the mine
  ;; ***need to send this data to the baseRobot as a list of mines
  if mines != nobody[

    ask swarmRobots with [status = "disarming"]
    [
      if target != nobody
      [
        if (ticks - start-time) > Disarm_TimeOut
        [
          right 180
          fd random 10
          set status "foraging"
          set start-time ticks
          set color blue
          ask target [
            set status "hidden"
            set start-time ticks
            set color yellow
          ]

        ]
      ]
    ]

    ask mines with [status = "found"]
    [
      if mines != nobody
      [
        if (ticks - start-time) > Found_TimeOut [
          set status "hidden"
          set start-time ticks
          set size 1
          set color white
        ]
      ]
    ]

    ask swarmRobots with [status = "following_scent"]
    [
      if target != nobody
      [
        if (ticks - start-time) > Follow_Scent_TimeOut [
          set status "foraging"
          set start-time ticks
          set color blue
        ]

      ]
    ]

    ask swarmRobots with [status = "waiting"]
    [
      if target != nobody
      [
        if (ticks - start-time) > Follow_Scent_TimeOut [
          set status "foraging"
          set start-time ticks
          set color blue
        ]

      ]
    ]

  ]


end

to blowup
  ;; this is fun - if my obsticle avoidance algo does not work well then the system will blow up.
  ask swarmRobots [
    if min-one-of other mines in-radius trigger-radius [distance myself]!= nobody
     [
      if min-one-of mines in-radius trigger-radius [distance myself] != target
        [
          if mines != nobody or swarmRobots != nobody [
            explode
            ask mines in-radius blast-radius [die]
            ask swarmRobots in-radius blast-radius [die]
          ]
        ]
     ]
  ]

  ask baseRobots [
    if min-one-of other mines in-radius trigger-radius [distance myself] != nobody
    [
      if mines != nobody or swarmRobots != nobody [
        explode
        ask baseRobots in-radius blast-radius [die]
        ask mines in-radius blast-radius [die]
        ask swarmRobots in-radius blast-radius [die]
        ]
    ]
  ]

end


; This is where the explosion is created.
; EXPLODE calls hatch a number of times indicated by the slider FRAGMENTS.
to explode ; turtle procedure
  hatch-frags 10 [
    set dim 0
    rt random 360
    set size blast-radius
    set x-vel (x-vel * .5 + dx + (random-float 2.0) - 1)
    set y-vel (y-vel * .3 + dy + (random-float 2.0) - 1)
    pen-down
  ]
end

; This function changes the color of a frag.
; Each frag fades its color by an amount proportional to FADE-AMOUNT.
to fade ; frag procedure
  ;set col one-of base-colors
  ;set color (col + 2)
  set dim dim - (10 / 10)
  set color scale-color color dim 5 .5
  ;show col
  if (color < (col - 3.0)) [ die ]
end

to obs-avoidance
  ;; more complex obstical avoidance but I need to look at this
  ifelse min-one-of mines in-radius 1 [distance myself]!= nobody or min-one-of obstacles in-radius 1 [distance myself]!= nobody
   [face patch-right-and-ahead 90 1 fd 1 repeat 2 [face patch-left-and-ahead 90 1   fd 1] face patch-right-and-ahead 90 1 fd 1]
   [pd fd 1]
end

to move_up
  ask baseRobots [set heading 0 fd 1]
end

to move_rt
  ask baseRobots [set heading 90 fd 1]
end

to move_lt
  ask baseRobots [set heading 270 fd 1]
end

to move_dn
  ask baseRobots [set heading 180 fd 1]
end


;; Includes from above
to make-msgQ [#id]
  set #msg-id #id
  set #msgQ []
end


to-report msg-waiting?
  report (not empty? #msgQ)
end


to-report get-msg
  let #tmp (first #msgQ)
  set #msgQ (but-first #msgQ)
  report #tmp
end


to send-msg [#who-to #msg]
  let #who-from self
  ask turtles with [#msg-id = #who-to]
  [ set #msgQ (lput (list #who-from #msg) #msgQ)
  ]
end


to broadcast [#breed-to #msg]
  let #who-from self
  set #msg (list #who-from #msg)
  ask turtles with [breed = #breed-to]
  [ set #msgQ (lput #msg #msgQ)
  ]
end


to flush-msgQ [#id]
  set #msgQ []
end


to flush-all-msgQs
  ask turtles
  [ set #msgQ []
  ]
end

;; sxl-utils
to wiggle
  right 45 - (random 90)
end


to-report nearest-of [#breed]
  report min-one-of #breed [distance myself]
end

to-report nearest-of-xy [#breed #corx #cory]
  report min-one-of #breed [distancexy #corx #cory]
end

to-report trigger [#prob]
  report (random 1000) < (#prob * 10)
end



to-report mutate [#val #delta]
  let #mutator (random (#delta * 2 + 1)) - #delta
  report (#val + #mutator)
end



to-report smooth-value [#last #inc #alpha]
  ;
  ; used in plotting to smooth graph data
  ; #alpha is the smoothing value 0-1
  ; larger #alphas are smoother, 0: no smoothing, 1: no recent value
  ;
  set #alpha (bounds 0 #alpha 1)
  let #new (#alpha * #last + (1 - #alpha) * #inc)
  report #new
end



to-report bounds [#lo #x #hi]
  ;
  ; return #x bounded by #lo & #hi
  ;
  report (min (list #hi (max (list #lo #x))))
end
@#$#@#$#@
GRAPHICS-WINDOW
187
17
728
559
-1
-1
13.0
1
10
1
1
1
0
1
1
1
-20
20
-20
20
1
1
1
time
30.0

BUTTON
17
10
80
43
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
96
52
159
85
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
18
51
79
84
go once
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
15
96
158
129
no-of-mines
no-of-mines
1
50
30.0
1
1
NIL
HORIZONTAL

SLIDER
15
134
160
167
no-of-swarmRobots
no-of-swarmRobots
2
32
4.0
1
1
NIL
HORIZONTAL

SLIDER
16
172
159
205
no-of-baseRobots
no-of-baseRobots
1
1
1.0
1
1
NIL
HORIZONTAL

SLIDER
17
209
159
242
swarm-teamSize
swarm-teamSize
1
8
4.0
1
1
NIL
HORIZONTAL

SLIDER
16
248
162
281
sensor-range
sensor-range
1
7
2.0
1
1
NIL
HORIZONTAL

SLIDER
16
286
161
319
sensor-FoV
sensor-FoV
30
180
105.0
15
1
Degrees
HORIZONTAL

SLIDER
8
413
179
446
Game_Time_Out
Game_Time_Out
1000
10000
10000.0
200
1
ticks
HORIZONTAL

SLIDER
8
453
180
486
Disarm_TimeOut
Disarm_TimeOut
100
1000
300.0
100
1
NIL
HORIZONTAL

SLIDER
7
493
179
526
Found_TimeOut
Found_TimeOut
100
1000
300.0
200
1
NIL
HORIZONTAL

SLIDER
7
535
179
568
Follow_Scent_TimeOut
Follow_Scent_TimeOut
100
1000
300.0
100
1
NIL
HORIZONTAL

SLIDER
15
324
161
357
no-of-sensors
no-of-sensors
0
50
0.0
1
1
NIL
HORIZONTAL

SLIDER
14
364
160
397
no-of-dynamicTargets
no-of-dynamicTargets
0
10
0.0
1
1
NIL
HORIZONTAL

SLIDER
90
11
182
44
Grid
Grid
1
10
1.0
1
1
NIL
HORIZONTAL

SLIDER
8
574
180
607
blast-radius
blast-radius
0
10
3.0
1
1
NIL
HORIZONTAL

SLIDER
8
612
180
645
trigger-radius
trigger-radius
.1
1
0.5
.1
1
NIL
HORIZONTAL

MONITOR
232
564
330
609
Bots Destroyed
no-of-swarmRobots - count swarmRobots
0
1
11

MONITOR
232
614
334
659
Mines Disarmed 
no-of-mines - count mines
17
1
11

SLIDER
605
573
728
606
no-of-obstacles
no-of-obstacles
0
20
0.0
1
1
NIL
HORIZONTAL

SWITCH
605
609
729
642
obs-detection
obs-detection
1
1
-1000

BUTTON
792
57
855
90
UP
move_up
NIL
1
T
OBSERVER
NIL
W
NIL
NIL
1

BUTTON
794
124
863
157
DOWN
move_dn
NIL
1
T
OBSERVER
NIL
X
NIL
NIL
1

BUTTON
730
91
793
124
LEFT
move_lt
NIL
1
T
OBSERVER
NIL
A
NIL
NIL
1

BUTTON
858
89
925
122
RIGHT
move_rt
NIL
1
T
OBSERVER
NIL
D
NIL
NIL
1

SLIDER
391
589
563
622
reset_Time
reset_Time
60
300
180.0
10
1
NIL
HORIZONTAL

@#$#@#$#@
### V3x - Added msg queue hold (line 301) <br> V4x - Improved random swarm calls to closest members of breed

## WHAT IS IT - Systems of System Demo Model - ASE6102 Team6

<b>System of Systems model for ASE6102.</b> Based on the System of Systems described in Jamshidi, 1st ed, Ch 14, System of autonomous rovers and their applications. The enviornemtn contains multiple mines, a single base robot (although multiple may be added) and multiple swarm robots. Mines can be disarmed by a team of swarm robots. When all the mines are disarrmed the model stops running.

## HOW IT WORKS

<b>There are three types of agents: mines, base robots, and swarm robots.</b>

<b>Swarm robots</b> have sensors which can detect a mine mine within their range and field of view (FoV) but in order to disarm a mine a group of swarm robots of a predetermined size must physically contact the mine.

<b>Base Robots</b> can only recieve notification from any swarm robot that a mine has been detected and the mines X,Y location. The base robot will then randomly select the required number of swarm robots to disarm the mine from the curently foraging robots, assign the a leader bot (the bot that found the mine), and direct them to the detected mines location for disarming.

<b>Mines</b> are static objects that are targets for disamring for the swarm robots. The mines have a XY location, a trigger radius, and a blast radius.  


<b>Base Robots and Swarm Robots work together:</b> Swarm robots find mines then send messages to the base robot describing where the mines are located. Swarm robots perforrm a random walk during foragin and periodically get messages from base robots containing information about the location of mines that have been found by other swarm robots. If the swarm robot is not currently detecting within its sensor or is in the process of disarming a mine the bot travels to target mines identified by the base robot in support of the bot teams disarming proceedure.

Note: this is roughly similar to an example in Ch14 with the exception of identifying dynamic theats and utilizing sensors both of which are deleted (dont think that I didnt want to add them :). 

The idea here is to extract the overall behavoirs and performance attributes in support of development of our teams artifacts for the presentation. Without this little model a lot of necessary information would have been overlooked during the block diagram building.

## SYSTEM STATES

### BASE ROBOT: In this model the base robot is simply a communcations tool utilized by swarm robots in support of mine disarming. 

The base robot has two basic states:

ON (green) - The base robot is monitoring message queue.
TX (blink-black) - The base robot has recieved a message, selected the supporting swarm robots, and is tramsitting data to the support robots.
WAITING - Msg in queue waiting on enough foraging swarm bots to deliver

Data Recieved: detecing_robot_Id, target_X, target_Y
Data Transmitted: leadSwarmBot_Id, target_Id, target_Vector(X,Y) 

### SWARM ROBOTS: The swarm robot has several states (and state colors) described below:

Foraging (blue) - In this state the swarm robot performs a random walk about the model looking for mines. (Although there should be...) No communcation to the outside world is happening when in the foraging state prior to finding a mine. During this state the swarm robot is using its sensor to look for mines where the sensor has a range and a field of view attribute. When a mine is found the swarm robot broadcasts the location to the base robots.

HOMING/WAITING (yellow) - When a swarm robot happens upon a mine that is idetified within its field of view the robot enters the homing/waiting state. During this state the robot is identified as a leader for the other swarm robots need for disarming and starts a homing procedure to slowly move the robot direclty next the edge of the mine to be disarmed. A timeout is set to ensure that if the disarming support robots do not arrive the system can reorder a new set of robots needed to disarm the mine.                                  

SCENT-FOLLOWING (orange): Once a robot has found a mine and has been desginated a leader the base robot assignes a number of current foraging swarm robots to the disarming team and sets their state scent-following (as in ant swarming). The lead robot, mine X cooridiante, mine Y corrodinate, and the XY vector to the mine is sent to each of the selected swarm robots. During the robots travel to the identified mines location the robots sensor continues to look for mines (if we roll over a mine we blow up). If a mine is detected simple evacive measures are taken to no travel over the mine while continueing to track the vector to the mine to be disamred. 

Data Recieved: leader_id, target_id, target_x, target_y, target_vector(x,y)
Data Transmitted: detecing_robot_id, target_X, target_Y

### MINES: The mines have several states noted below:

HIDDEN (black) - Each mine is intiaized in the hidden state.
FOUND (red) - When a swarm robot finds a mine the mines state is changed to Found.
DISARMED (deleted from world) - When the swarm is able to disarm the mine the mines state is Disabled and the mine is asked to die removing it from the game.

### TIMEOUTS: During the simualtion there are conditions that occure resulting in swarm robots that may become stuck or zombies. In order to ensure that each game finished the following timeout states are provided:

DISARMING: If a swarm robot is in the diarming state and the supporting robots are not able to reach the mines location within a resonable time the robot will perform a manuver to safely move away from the mine and set its status to Foraging to resume looking for mines. The mine status will be set to hidden but now the mine is colored white to idicate that it has been found one but was not disarmed.

FOUND: If a mine is found but not disarmed within the timeout limit its status will be changed back to hidden and its color will change to wite to idicate that it has been found one but was not disarmed.

FOLLOWING_SCENT: If a robot has been following scent too long (Zombie Robot) a timeout will occure and its status will be set back to foraging.

WAITING: If a swarm robot that has found a mine has been in the Homing/Waiting state too long a timeout will occure and its state will be set back to foraging.

### OBSTACLE AVOIDANCE (currently only a sub-state of 'Following-Scent')
During the running of the models it became apparent particularly when the swarm robots are in the 'follow-scent' state that some sort of obsticle avoidance and decision making is necessary. 

For instance when in the 'follow-scent' state the helper swarmBot travels a path straight to the mine where the leader is waiting. If another mine is within the path of the straight line between the helper bot and the leader what to do??? 

In order to accomidate this scenario the helper swarmBot's sensor remains on and when a mine (other than its target mine) is detected an avoidance manuver is executed with the bot being reoriented toward its target mine after the manuver. 

For more fun a trigger radius is provided to assess the capability of the obstacle avoidance algorithm. If a swarmBot is located within the trigger radius of any mine the mine will detonate and kill any swarmBot within its blast radius. Any mines within the blast radius are also killed but the do not detonate. 

Currently there are two fairly simple avoidance algorithms with the second being slight more complex and which can be turned on by setting 'obs-detection' to 'ON'. This second algorithm will show the path history of each bot for review. 

### Base Robots Movments: 

The user can move the base robot with the buttons in the upper right side of the screen. 
Motion is limited to a single base robot. This fuction is intended to be a part of a future behavior related to limited communcaitons range assesments. 

## HOW TO USE IT

The model allows users to specify the number of mines, number of swarm robots, and the number of swarm robots necessary to disarm a mine, a swarmBots sensor range, and sensor field of view angle. The base robot is in the middle of the environment (xcor=0, ycor=0) and all of the swarm robot agents start there. Mines are randomly scattered around the environment.

Mines are circular black & white blobs, the base robot a green square shape, swarm robots are blue (standard turtle shape) when foraging, orange when following scent, yellow when homing/waiting on support robots, and red when they are disarming.

When the model starts running, detectors (blue) leave the base robot area, each randomly searching for a mine. When a swarm robot sensor detects a mine it send the base robot the X,Y corrdinates and its name. The base robot selects support robots (at random) and sends them a message about the mine's location and the robot that found the mine. The support robots (orange) then travel to the mine to be disarmed while continuing to look for mines so that can avoid the trigger-range of other mines (boom!!!).

When the required number of support robots reach the mine to be disarmed the robots turn red and the mine symbol disappears from the visual environment, the leader and the support robots then return to foraging (turning blue) for other hidden mines.

<u>Players can vary paramters to determine how quickly a number of swarm robots for a given disarm group size can clean the mine field.</u>

## THINGS TO NOTICE
<b>Object Detection and Obsticle Avoidance:</b> 
When working with the simulation model it becomes appartent that object detection and the associated algorithms for addressing these objects are very imoportant and become the most complex part of the SoS. A good strategy would be to utilize the base robot as the processing power for any complex object detection/avoidance algos as the swarmBots are to remain simple in nature due to the quantities. Well beyond the scope of this class a full study of the detection and avoidance algos would be super fun particuarly when using this super easy to code tool (netlogo). There is a 'R' plug-in for NetLogo which would allow easier introduction of maching learning tools... keep this in mine for future efforts.

## FURTHER INFORMATION
jdehart@gatech.edu
john@int64.tech

## CREDITS AND REFERENCES
1- The origination of this model and the agent messaging script (messaging.nls) was developed as part of the tutorial for NL Boris - see www.agent-domain.org for additional tutorials and updates.
2 - Based on the model described in Chapter 14 of Jamshidi - System of autonomous rovers and their applications.
3 - The supporint script sxl-utils.nls was devloped by S.C.Lynch (http://s573859921.websitehome.co.uk/pub/netlogo/cogs_nl_tools.htm#utils)
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

garbage can
false
0
Polygon -16777216 false false 60 240 66 257 90 285 134 299 164 299 209 284 234 259 240 240
Rectangle -7500403 true true 60 75 240 240
Polygon -7500403 true true 60 238 66 256 90 283 135 298 165 298 210 283 235 256 240 238
Polygon -7500403 true true 60 75 66 57 90 30 135 15 165 15 210 30 235 57 240 75
Polygon -7500403 true true 60 75 66 93 90 120 135 135 165 135 210 120 235 93 240 75
Polygon -16777216 false false 59 75 66 57 89 30 134 15 164 15 209 30 234 56 239 75 235 91 209 120 164 135 134 135 89 120 64 90
Line -16777216 false 210 120 210 285
Line -16777216 false 90 120 90 285
Line -16777216 false 125 131 125 296
Line -16777216 false 65 93 65 258
Line -16777216 false 175 131 175 296
Line -16777216 false 235 93 235 258
Polygon -16777216 false false 112 52 112 66 127 51 162 64 170 87 185 85 192 71 180 54 155 39 127 36

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

tree pine
false
0
Rectangle -6459832 true false 120 225 180 300
Polygon -7500403 true true 150 240 240 270 150 135 60 270
Polygon -7500403 true true 150 75 75 210 150 195 225 210
Polygon -7500403 true true 150 7 90 157 150 142 210 157 150 7

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

ufo top
false
0
Circle -1 true false 15 15 270
Circle -16777216 false false 15 15 270
Circle -7500403 true true 75 75 150
Circle -16777216 false false 75 75 150
Circle -7500403 true true 60 60 30
Circle -7500403 true true 135 30 30
Circle -7500403 true true 210 60 30
Circle -7500403 true true 240 135 30
Circle -7500403 true true 210 210 30
Circle -7500403 true true 135 240 30
Circle -7500403 true true 60 210 30
Circle -7500403 true true 30 135 30
Circle -16777216 false false 30 135 30
Circle -16777216 false false 60 210 30
Circle -16777216 false false 135 240 30
Circle -16777216 false false 210 210 30
Circle -16777216 false false 240 135 30
Circle -16777216 false false 210 60 30
Circle -16777216 false false 135 30 30
Circle -16777216 false false 60 60 30

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.2.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="experiment" repetitions="20" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <enumeratedValueSet variable="smart-foxes-on">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="no-of-hounds">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="no-of-foxes">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="smart-hounds-on">
      <value value="true"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="hound-memory" repetitions="1" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>count turtles</metric>
    <enumeratedValueSet variable="smart-foxes-on">
      <value value="true"/>
    </enumeratedValueSet>
    <steppedValueSet variable="global-memory-duration" first="15" step="1" last="25"/>
    <enumeratedValueSet variable="no-of-hounds">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="no-of-foxes">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="smart-hounds-on">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="total-ticks">
      <value value="0"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="experiment" repetitions="10" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <enumeratedValueSet variable="smart-foxes-on">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="no-of-hounds">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="no-of-foxes">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="smart-hounds-on">
      <value value="true"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Vary mines, bots, range, fov" repetitions="1" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="10000"/>
    <metric>count mines</metric>
    <enumeratedValueSet variable="Game_Time_Out">
      <value value="10000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Follow_Scent_TimeOut">
      <value value="500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="no-of-baseRobots">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Disarm_TimeOut">
      <value value="500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="no-of-sensors">
      <value value="0"/>
    </enumeratedValueSet>
    <steppedValueSet variable="no-of-swarmRobots" first="4" step="8" last="28"/>
    <enumeratedValueSet variable="Grid">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Found_TimeOut">
      <value value="500"/>
    </enumeratedValueSet>
    <steppedValueSet variable="sensor-FoV" first="60" step="60" last="180"/>
    <steppedValueSet variable="sensor-range" first="1" step="2" last="7"/>
    <steppedValueSet variable="no-of-mines" first="2" step="10" last="52"/>
    <enumeratedValueSet variable="swarm-teamSize">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="no-of-dynamicTargets">
      <value value="0"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
