patches-own [
height
]

to setup
  clear-all
  set seed-a random-float 100000
  set seed-x random-float 1000
  set seed-y random-float 1000
  generate
  reset-ticks
end


to go
  generate
  tick
end

;; Generate several layers of noise and superimpose them.
to generate
  ask patches [
    set height 0
    let amplitude 1
    let frequency 1 / (max-pxcor - min-pxcor)
    repeat iterations [
     set height height + amplitude *  (noise (pxcor * zoom * frequency) (pycor * zoom * frequency))
     set frequency frequency * frequency-scaling ;; Each layer of noise is a higher frequency than the previous, creating increasingly fine details
     set amplitude amplitude / amplitude-scaling ;; Each layer of noise is a lower amplitude than the previous so that small terrain details don't cover up larger ones
    ]
  ]

  rescale-height
  if ridgify [
    ask patches [ set height -1 * abs (height * -1) ]
    rescale-height
  ]
  if squarify [
    ask patches [ set height height / sqrt (abs height) ]
    rescale-height
  ]
  ask patches [set pcolor scale-color white height -1 1]

end

;; Rescale all cells to be between -1 and 1
to rescale-height
  let height-max [height] of max-one-of patches [height]
  let height-min [height] of min-one-of patches [height]
  let delta height-max - height-min
  let m 2 / delta
  let b (-1 - m * height-min)
  ask patches
  [
    set height (m * height + b + offset)
  ]
end

;; Fractal Noise generator
;; We imagine a square with 4 corner points, with our point of interest somewhere inside
;; We assign each point a random value, and consider the dot product between our point of interest and each corner point
;; Then we interpolate between the 4 nearest points to get smooth transitions
to-report noise [x y]
  ;; The "floor" part of our vector tells us which square inside the noise cloud we are located within
  ;; The "fractional" part of our vector tells us where we are within that square
  let floor-x floor x
  let frac-x x - floor-x
  let floor-y floor y
  let frac-y y - floor-y

  ;; This is a special cubic interpolation function, which will gives us smoother results than linear interpolation
  let u-x (frac-x * frac-x * (3 - 2 * frac-x))
  let u-y (frac-y * frac-y * (3 - 2 * frac-y))

  ;; These scalar values represent the value at each corner of the grid containing our point of interest.
  let a rand floor-x floor-y
  let b rand (floor-x + 1) floor-y
  let c rand floor-x (floor-y + 1)
  let d rand (floor-x + 1) (floor-y + 1)

  ;; We interpolate between our 4 points to get the value at our point of interest
  let int_1 (b - a) * u-x + a
  let int_2 (c - a) * u-y * (1 - u-x)
  let int_3 (d - b) * u-x * u-y
  report int_1 + int_2 + int_3
end

;; Generate a random scalar vector
;; We use this instead of random-float because it allows our noise cloud to remain the same from patch to patch - otherwise, each patch would
;; base its value on a completely differently seeded noise cloud, and the result would be random static!
to-report rand [x y]
  let dot seed-a * sin(x * seed-x + y * seed-y)
  report dot - floor dot
end
@#$#@#$#@
GRAPHICS-WINDOW
484
19
1262
798
-1
-1
5.97
1
10
1
1
1
0
1
1
1
-64
64
-64
64
0
0
1
ticks
30.0

BUTTON
12
66
76
99
Setup
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
27
122
199
155
Iterations
Iterations
1
30
10.0
1
1
NIL
HORIZONTAL

SLIDER
27
275
199
308
Zoom
Zoom
.1
50
4.0
.1
1
NIL
HORIZONTAL

SLIDER
1263
56
1435
89
seed-x
seed-x
1
100
300.25197351819565
1
1
NIL
HORIZONTAL

SLIDER
1263
96
1435
129
seed-y
seed-y
1
100
998.103342336033
1
1
NIL
HORIZONTAL

BUTTON
156
66
303
99
Update Continuously
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
27
177
199
210
Amplitude-Scaling
Amplitude-Scaling
1
10
2.0
.1
1
NIL
HORIZONTAL

SLIDER
27
228
199
261
Frequency-Scaling
Frequency-Scaling
1
10
2.0
.1
1
NIL
HORIZONTAL

SWITCH
45
472
148
505
Ridgify
Ridgify
1
1
-1000

SLIDER
27
326
199
359
Offset
Offset
-1
1
0.0
.01
1
NIL
HORIZONTAL

BUTTON
80
66
152
99
Update
go\n
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SWITCH
282
474
385
507
Squarify
Squarify
1
1
-1000

PLOT
33
564
295
766
Histogram
Height
Patches
0.0
10.0
0.0
10.0
true
false
"set-plot-x-range -1 1\nset-plot-y-range 0 count patches / 5\nset-histogram-num-bars 10" ""
PENS
"default" 0.1 1 -16777216 true "" "histogram [height] of patches"

TEXTBOX
15
10
252
54
Fractal Noise Generator
18
0.0
1

SLIDER
1263
133
1435
166
seed-a
seed-a
1
100
18763.690471645965
1
1
NIL
HORIZONTAL

TEXTBOX
208
124
429
345
Higher iterations results in more detailed noise\n\nHigher amplitude scaling makes smoother noise\n\nHigher frequency affects the size of each successive layer of noise\n\nZoom scales the map\n\n\nOffset adds a flat value to all cells
14
0.0
1

TEXTBOX
274
407
424
463
Squarify takes the square root of all cells and then rescales, creating more dramatic terrain with fewer intermediate values
11
0.0
1

TEXTBOX
37
405
187
461
Ridgify takes the absolute value of all cells and then rescales, creating sharp ridges rather than smooth transitions
11
0.0
1

TEXTBOX
16
38
166
56
Model by Ethan Cuka
11
0.0
1

@#$#@#$#@
## WHAT IS IT?

Fractal Noise is a method for generating a field of semi-random values, where each region blends smoothly into its neighbors, but also has details of its own. By adding more layers of noise, representing smaller regions, it is possible to create an arbitrary amount of increasingly fine details.

This method is commonly used in computer graphics for procedurally generated textures and terrain: by beginning with a cloud of fractal noise and adding a variety of effects and filters, artists and programmers can create very rich patterns that appear convincing and naturalistic.

This model allows users to explore the different parameters of noise generation and how they can be used to create different textures.

## HOW IT WORKS

The algorithm considers a square grid of points. At each point on the grid, it assigns a value between 1 and -1. At points not directly on the grid, it interpolates between its 4 nearest neighbors to smoothly transition between grid points.

The process is repeated, with a new grid of twice the frequency (smaller grid spacing) and half the amplitude (-0.5 to +0.5) This new grid is superimposed on top to add a new layer of smaller details.

By adding higher frequencies of noise on top of each other, the algorithm generates a cloud-like pattern, with both large variations in texture over broad regions, and small variations over narrow regions.

Finally, the resulting values are linearly scaled to be between -1 and 1.

## HOW TO USE IT


* **Iterations** controls how many layers of noise to generate. Fewer iterations will result in less detailed noise. There are diminishing returns for higher iterations based on the amplitude scaling.

* **Amplitude Scaling** controls how strong each successive layer will be in the final product. At high amplitude scaling, only the first few iterations of noise, which represent low-frequency components, will be noticable. This results in smoother terrain. At low amplitude scaling, higher frequencies will be more visible. This results in more chaotic noise.

* **Frequency Scaling** controls the dimensions of each successive layer. At the default (2), each layer of noise is twice as tightly-spaced as the one above it.

* **Zoom** simply scales the noise cloud to examine wider or narrower details.

* **Offset** adds a flat amount to each point, changing the threshold/"brightness" of the overall pattern.

* **Ridgify** takes the absolute value of each point in the cloud and then rescales the cloud. At points where the noise passes through zero, this creates sharp lines.

* **Squarify** finds each point's square root while preserving sign. This makes high and low regions more likely, while avoiding middle regions. (Observe how the histogram changes when Squarify is applied)

* The **seed"** sliders affect the way underlying random values are generated.

## THINGS TO NOTICE

* As you vary each setting, consider what sort of textures an artist might be able to create with the resulting pattern.

* Turn the number of iterations down to 1 to see the lowest level of noise generated. Why does it have a "blocky" texture? Why does this blockiness diminish when more iterations are added?

* How does varying the amplitude scaling affect the "graininess" of the noise? Why?

* Perfect fractal noise would look approximately the same at all zoom levels. How does the texture of this noise change at different zoom levels? Why? (Hint: the fundamental frequency of the generated noise is based off of the World size)

### Seeding and Random Scalar Generation

For these questions, examine the rand function in the code, which is used to generate a random scalar between -1 and 1.

* Why does the center of the noise cloud always look the same, no matter what seed parameters are used?

* What happens when the X and Y seed are the same, or integer multiples of each other? Why does this happen?

* What happens when seed-a is very low? Why does this happen?

## THINGS TO TRY

* The Squarify and Ridgify switches apply some new effects to the noise to create new patterns. What other effects can you think of to apply to the noise cloud?

* Each iteration of noise uses the same seed to generate itself. As a result, each layer is just a scaled down version of the layer above. This produces an "alias"-like stripe that pass through the center of the noise. How would you prevent this? (There are multiple ways to do so!)

## EXTENDING THE MODEL

This model uses Value Noise, where each point on the grid is a scalar value between -1 and 1. A common alternative is Perlin Noise, or Gradient noise, where each grid point is a vector with a random direction, and points that do not sit on the grid have a value based on the dot product of their offset vector and the gradient vector.

The result is often noise with more convincing variations between regions. How would you alter the noise function in this program to generate Perlin noise, rather than Value noise?

## CREDITS AND REFERENCES

This Netlogo model by Ethan Cuka, 2023

These two resources were used for help in generating convincing random values, and for interpolating between values.

https://thebookofshaders.com/13/
https://www.shadertoy.com/view/4dS3Wd
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.3.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
