;;
;; This model is developed by Dr. Lin Xiang at the University of Kentucky. Contact: lin.xiang@uky.edu
;;
;; If you see this page rather than a "download" button when downloading the model, click the "download" icon
;; in your browser to download this model file.
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


turtles-own[day infected susceptible hospitalized symptom]
Globals [RM LM num-beds x y max-daily-cases]
breed [persons person]
breed [doctors doctor]

to-report hospital   ;label the divider patches
  report patches with [ pycor < max-pycor and pycor > 16 and pxcor > min-pxcor and pxcor < max-pxcor]
end

to-report susceptible-people     ;call all suscepitble people
  report turtles with [infected = false and susceptible = true and symptom = false and hospitalized = false]
end

to-report carriers   ;call all people who are infected but have not shown symptoms
  report turtles with [infected = true and susceptible = false and symptom = false and hospitalized = false]
end

to-report infected-with-symptom   ;call all people who are infected and have shown symptoms
  report turtles with [infected = true and susceptible = false and symptom = true]
end

to-report patients   ;call all people who are infected and have shown symptoms and have been hospitalized
  report turtles with [infected = true and susceptible = false and symptom = true and hospitalized = true]
end

to-report current-patients
  report count patients
end

to-report recovered-people   ;call all people who have recovered
  report turtles with [infected = false and susceptible = false and symptom = false]
end

to-report available-beds ; availble inpatient beds
  report num-beds - count persons with [pcolor = 9.9]
end


;;;;;;;;;;;;;;;;;;;;main setup procedures;;;;;;;;;;;;;;

to setup
  clear-all
  setup-patches
  setup-turtles
  reset-ticks
end



to setup-patches

  ask patches with [ abs pxcor = max-pxcor ] [set pcolor 116]   ;setup boundary
  ask patches with [ abs pycor = max-pycor ] [set pcolor 116]   ;setup boundary
  ask patches with [ pycor = 16 ] [set pcolor 116]  ;setup boundary
  ask hospital  [set pcolor 9.9]    ;set hospital area
end


to setup-turtles

  create-persons Population-size
  [set color 68
    set size 1.75
    set shape "person-1"
    set day 0
    set infected false
    set susceptible true
    set hospitalized false
    set symptom false
    set-position
  ]

 set num-beds Population-size * 0.05
  set x -21
  set y 18

   create-doctors 1
  [ set color 37
    set size 3
    set shape "person doctor"
    setxy 22.5 26.5
  ]

end


to set-position  ;set people position
  setxy random-xcor random-ycor
  if pcolor != 0 [set-position]
end

to set-beds   ;set hispital bed positions
   setxy x y
   if any? other turtles-here [
    (ifelse
      x < 21 [set x x + 3]
      x = 21 [set x -21 set y y + 3])
      set-beds]

   set x -21
   set y 18
end

;;;;;;;;;;;;;;;;;;;;main go procedures;;;;;;;;;;;;;;
to go

  if ticks >= Days [stop]

  move
  transmission
  sickness
  isolation
  ifelse Hospitalization? [hospitalization-recovery-death][recovery-death]
  find-max-daily-cases
  tick

end

to move
  ask persons with [hospitalized = false and shape = "person-1"]    ;Keep patients and isolated people from moving
  [ifelse [pcolor] of patch-ahead 1 = 0 [forward Mobility rt random 360] [set heading heading - random 180]]

end

to add-a-carrier
  create-persons 1
  [set size 1.75
    set shape "person-1"
    set color 68
    set infected true
    set susceptible false
    set symptom false
    set hospitalized false
    set-position]
end

to transmission       ;set up transmission
  ask persons with [infected = true and shape = "person-1"]  ;shape of "person" excludes the isolated carriers
  [let healthy-person one-of other turtles in-radius 1.5 with [susceptible = true]
    if healthy-person != nobody
    [ask healthy-person [
      if random 100 < 90  ;transimission rate is 90%
      [set infected true set susceptible false set symptom false]]]  ;susceptible people first become carriers.
  ]
end

to sickness     ;process of developing symptoms
  ask persons with [infected = true]
  [
    ifelse Show-carriers? = true    ;allow switch to show and hide carriers
    [ifelse symptom = false [set color gray][set color orange]]
    [ifelse symptom = false [set color 68][set color orange]]

    set day day + 1  ;accumulate days
    if random 100 < (10 * day - 30) [set symptom true set color orange]   ; The infected people show symptoms after 3 days and the chance increase 10% every day after.

  ]
end

to watch-a-sick-person ; can be either a carrier or a sick person
  watch one-of persons with [infected = true]
end


to hospitalization-recovery-death
  ask persons with [infected = true and symptom = true]  ; call sick people
  [if available-beds > 0 [
    set shape "patient"
    set hospitalized true
    set-beds]

  if day >= 15
    [ifelse hospitalized = true
      [ifelse random 100 < 20 * 0.5
          [set LM LM + 1 die  ]
          [set infected false set hospitalized false set susceptible false set symptom false set color blue set shape "person-1" set-position]]
      [ifelse random 100 < 20
          [set LM LM + 1 die  ]
          [set infected false set hospitalized false set susceptible false set symptom false set color blue set shape "person-1" set-position]]
      ]
  ]
end

to recovery-death
   ask persons with [infected = true and symptom = true]
     [if day >= 15
      [ifelse random 100 < 20
          [set LM LM + 1 die  ]
          [set infected false set hospitalized false set susceptible false set symptom false set color blue set shape "person-1" set-position]]
      ]
end


to isolation
  ask persons with [infected = true and symptom = true and hospitalized = false]
      [if random 100 < %-isolated [set shape "house"]]
  ask carriers
      [if random 100 < %-quarantined [set shape "house"]]
end

to find-max-daily-cases
  if count turtles with [color = orange ] > max-daily-cases        ;Count the infectious. If it is greater than the current record of max daily cases
  [set max-daily-cases count turtles with [color = orange ]]       ;update the max daily case
end
@#$#@#$#@
GRAPHICS-WINDOW
17
87
433
568
-1
-1
8.0
1
10
1
1
1
0
1
1
1
-25
25
-29
29
0
0
1
ticks
30.0

BUTTON
575
89
669
157
Set up/Reset
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
759
88
861
121
Run/Pause
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

SLIDER
17
10
169
43
Population-size
Population-size
0
800
800.0
50
1
NIL
HORIZONTAL

BUTTON
443
88
570
121
+1 infected case
add-a-carrier
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

MONITOR
443
438
574
479
Susceptible
count susceptible-people
17
1
10

BUTTON
759
124
861
157
Run one day
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

SLIDER
17
48
168
81
Mobility
Mobility
0
1
1.0
0.1
1
NIL
HORIZONTAL

PLOT
443
166
862
432
Infection in Populations over Time
Days
People
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"Carriers" 1.0 0 -5987164 true "" "plot count carriers"
"Sick" 1.0 0 -955883 true "" "plot count infected-with-symptom"
"Inpatient Capacity" 1.0 0 -13791810 true "" "plot num-beds"

MONITOR
443
482
590
523
Sick (symptomatic)
count infected-with-symptom
17
1
10

MONITOR
594
483
738
524
Recovered/Immune
count recovered-people
17
1
10

MONITOR
594
527
658
568
Days
ticks
17
1
10

BUTTON
443
125
570
158
Watch 1 infected case
ifelse count turtles with [infected = true] >= 1\n[watch-a-sick-person]\n[user-message \"There are no infected people.\"]
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

INPUTBOX
676
88
753
157
Days
180.0
1
0
Number

MONITOR
742
438
863
479
Total Deaths
LM
17
1
10

TEXTBOX
18
574
187
592
|Green: Susceptible\n
10
54.0
1

TEXTBOX
281
572
455
590
|Orange: Sick (symptomatic)
10
25.0
1

TEXTBOX
18
588
168
606
|Blue: Recovered/Immune
10
105.0
1

TEXTBOX
115
573
282
591
|Gray: Carriers (pre-symptomatic)
10
3.0
1

TEXTBOX
355
103
402
129
Hospital
10
15.0
1

MONITOR
742
483
863
524
Total survival
Count persons
17
1
10

SWITCH
444
10
583
43
Hospitalization?
Hospitalization?
0
1
-1000

MONITOR
443
527
590
568
Available Inpatient Beds
available-beds
17
1
10

SWITCH
445
49
584
82
Show-carriers?
Show-carriers?
1
1
-1000

MONITOR
576
438
740
479
Carriers (pre-symptomatic)
count carriers
17
1
10

SLIDER
591
10
720
43
%-isolated
%-isolated
0
100
0.0
1
1
NIL
HORIZONTAL

TEXTBOX
676
64
813
89
Change time to run a longer or shorter time period
9
125.0
1

SLIDER
726
10
860
43
%-quarantined
%-quarantined
0
100
0.0
1
1
NIL
HORIZONTAL

TEXTBOX
193
10
426
80
--------------------------------\n  Transmission rate = 90% \n  Mortality (not hospitalized) = 20%\n  Mortality (hospitalized) = 10%\n--------------------------------\n
11
0.0
1

MONITOR
662
527
798
568
Maximum Daily Cases
max-daily-cases
17
1
10

@#$#@#$#@
## WHAT IS IT?

This model simulates the impacts of hospital inpatient capacity, isolation, and quarantine on the infectious disease outbreak dynamics, such as the COVID-19 pandemic. The infected cases include both carriers, the infected people who have not shown symptoms, and sick, the infected people who have shown symptoms.

 
## HOW IT WORKS

The model starts with a human population in which all individuals are healthy but susceptible (green persons) to the incoming pathogen. Once a carrier (also green but can be shown as gray by the show-carriers? switch) appears in the population, it will pass the disease to one of the susceptible people nearby (within a radius of 1.5) at the transmission rate of 90%. The infected people are able to transmit the disease for 14 days if they are not hospitalized or isolated. The infected people do not show symptoms in the first three days but are still contagious. After three days, infected people are more likely to show symptoms as time passes. Once they show symptoms, they are sick, indicated by an orange color. Depending on the settings, the sick can be hospitalized, isolated, or left by themselves. By the 15th day of being infected, the sick people either die (disappear from the model) at the mortality rate of 20% or recover and become immunized (blue color). If sick people are hospitalized, their mortality rate is lowered to 10%.

To simplify the model, this model uses a fixed transmission rate and mortality rate, 90% and 20%, respectively.

Buttons, Sliders, and Switches:

* The **population size** slider is self-explanatory. So are the buttons of **Set up/Reset**, **Run/Pause**, and **Run one day**. 

* The **mobility** slider controls how far an individual can move each time. Smaller mobility means people come into contact with fewer other people. 

* The **hospitalization?** switch can enable and disable hospitalization. The hospital inpatient capacity is set as 5% of the population size. This is actually higher than the  USA average hospital inpatient capacity, which is about 0.3%. 

* The **show-carriers?** switch may make the carriers visible or invisible. 

* The **%-isolated** slider set how many percent of sick people who are not hospitalized are isolated. Sick people stop infecting susceptible people once isolated. 

* The **%-quarantined** slider set how many percent of carriers are quarantined. Carriers stop infecting susceptible people once quarantined. 

* The **+1 infected case** button adds **a carrier** into the population.

* The **Watch 1 infected case** button allows you to focus on a single infected person, either a carrier or a sick person.  


## HOW TO USE IT

1. First, choose the factors, such as population size,  mobility level, hospitalization on or off, etc.

2. Click on Set up/Reset, then Run/Pause. The model is initially set to stop on the 180th day. Change the number in "Days" if you want to run the model for a longer or shorter time period.

3. Observe the infection changes in the population in the plot and monitors.

4. Use Run one day to run the model in a controlled way and collecting day-by-day data.


## THINGS TO TRY

There are so many things you can try in this model. Here are only very a few quick ideas to explore:

1. How many people die when hospitalization is available vs. unavailable?
2. How does hospitalization affect the number of deaths in the outbreak?
3. How does hospitalization affect the duration of the outbreak?
4. How does the isolation rate affect the number of deaths in the outbreak?
5. How does the isolation rate affect the duration of the outbreak?
6. How does the quarantine rate affect the number of deaths in the outbreak?
7. How does the quarantine rate affect the duration of the outbreak?
8. How do infections change in the population when isolation, quarantine, or low mobility is practiced 2 days, 3 days, or a week after introducing the carrier?
9. How do infections change in the population if we only isolate the sick people but do nothing to the carriers?
10. Can we both lower the number of deaths and shorten the duration of the outbreak? How?

## RELATED MODELS

Find more epidemic models at [3dsciencemodeling.com/COVID-19](https://sites.google.com/view/3d-science-abm/covid-19)

## CREDITS AND REFERENCES

Dr. Lin Xiang ([lin.xiang@uky.edu] (mailto:lin.xiang@uky.edu)) created this module at the University of Kentucky in 2020. If you mention this model in a publication, we ask that you include the citations below.

Xiang, L. (2020). Infectious Disease Outbreak-HealthCare, Isolation and Quarantine. Department of STEM Education, University of Kentucky, Lexington, KY.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

bed
false
0
Rectangle -7500403 false true 45 15 255 285

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -1 true false 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 0 120 150 15 300 120
Line -16777216 false 0 120 300 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

patient
false
0
Rectangle -7500403 false true 0 45 300 255
Circle -7500403 true true 15 120 60
Polygon -7500403 true true 90 195 180 180 285 210 300 195 300 180 210 150 300 120 300 105 270 90 180 120 90 105
Rectangle -7500403 true true 75 135 94 165
Polygon -7500403 true true 105 105 150 60 180 75 105 135
Polygon -7500403 true true 90 180 150 240 180 225 105 165

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 120 15 60
Polygon -7500403 true true 105 90 120 180 90 285 105 300 120 300 150 225 180 300 195 300 210 285 180 180 195 90
Rectangle -7500403 true true 135 75 165 94
Polygon -7500403 true true 195 105 240 150 225 180 165 105
Polygon -7500403 true true 120 90 60 150 75 180 135 105

person doctor
false
0
Circle -16777216 true false 181 1 30
Polygon -16777216 true false 107 92 62 197 92 212 116 158 122 197 92 272 212 272 182 197 188 157 212 212 242 197 197 92 167 92 152 152 137 92
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Polygon -13345367 true false 135 90 150 105 135 135 150 150 165 135 150 105 165 90
Polygon -7500403 true true 105 90 60 195 90 210 135 105
Polygon -7500403 true true 195 90 240 195 210 210 165 105
Circle -7500403 true true 110 5 80
Rectangle -7500403 true true 127 79 172 94
Polygon -1 true false 105 90 60 195 90 210 114 156 120 195 90 270 210 270 180 195 186 155 210 210 240 195 195 90 165 90 150 150 135 90
Line -16777216 false 150 148 150 270
Line -16777216 false 196 90 151 149
Line -16777216 false 104 90 149 149
Circle -1 true false 180 0 30
Line -16777216 false 180 15 120 15
Line -16777216 false 150 195 165 195
Line -16777216 false 150 240 165 240
Line -16777216 false 150 150 165 150

person-1
false
0
Circle -7500403 true true 120 0 60
Polygon -7500403 true true 150 75 135 75 120 75 75 165 90 180 120 120 120 180 90 270 120 300 150 195 180 300 210 270 180 180 180 120 210 180 225 165 180 75

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.2.2
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
